function outputArray = vitter673(varargin)
%VITTER673 Encode/decode array/file by dynamic Huffman code.
%
% Syntax
%
% outputArray = VITTER673(Name, Value)
%
% Name-Value Pair Arguments
%
% Specify comma-separated pairs of Name, Value arguments. Name is the
% argument name and Value is the corresponding value. Name must appear
% inside single quotes (' '). You can specify up to five name and value
% pair arguments. Mandatory arguments are 'mode' and 'input'.
% Any order of the Name-Value pairs is allowed.
%
% - mode            - Mode of coding (mandatory)
% 'encode' | 'decode'
%
% - input           - Input array or name of input file (mandatory)
%
% - inputKind       - Kind of input (optional)
% 'array' (default) | 'file'
%
% - outputFilename  - Name of output file (optional)
%
% - auxiliaryOutput - Auxiliary output (optional)
% 'on' (default) | 'off'
%
% Output Arguments
%
% - outputArray     - Output array
%
% Description
%
% VITTER673(Name, Value) encodes/decodes array/file using up to five Name,
% Value pair arguments.
%
% VITTER673 works in 2 modes:
% - Encoding mode - computes dynamic Huffman code either of an array of
% characters or integers or of a file and returns an array of '1' and '0',
% - Decoding mode - decodes dynamic Huffman code either of an array of
% the symbols '1' and '0' or of a file and returns a decoded array.
%
% For input array of characters VITTER673 returns an array of characters,
% for input array of integers it returns array of integers.
% Input integer array should contain only integers in range 0...255.
% Input file is read as sequence of bytes in range 0..255 and then
% processed as an array of integers.
%
% Optionally the returned array can be saved in file.
% In encoding mode an output array of integers is saved as a sequence of
% bytes with the values 0 and 1 (binary file), an output array of
% characters is saved as sequence of characters '0' and '1' (text file).
% 
% Examples
%
% Encode the character array 'abacabdabaceabacabdfg' into array
% huffmanCode with dynamic Huffman code (character array of '1' and '0').
%
% inputArray = 'abacabdabaceabacabdfg'
% huffmanCode = VITTER673('mode', 'encode', 'input', inputArray)
% or
% huffmanCode = VITTER673('mode', 'encode', 'input', inputArray, 'inputKind', 'array')
%
% The same without auxiliary output:
%
% huffmanCode = VITTER673('mode', 'encode', 'input', inputArray, 'auxiliaryOutput', 'off')
%
% The same with output to file outputEncoded.txt:
%
% huffmanCode = VITTER673('mode', 'encode', 'input', inputArray, 'outputFilename', 'outputEncoded.txt')
%
% Encode the integer array inputArray into array huffmanCode with dynamic
% Huffman code (integer array of '1' and '0').
% In this example the input array contains the ASCII decimal values of
% characters from previous example.
%
% inputArray = int16([97,98,97,99,97,98,100,97,98,97,99,101,97,98,97,99,97,98,100,102,103])
% huffmanCode = VITTER673('mode', 'encode', 'input', inputArray)
%
% Encode the file input.txt into the output array huffmanCode with dynamic
% Huffman code (output array is an array of integers '1' and '0').
%
% huffmanCode = VITTER673('mode', 'encode', 'input', 'input.txt', 'inputKind', 'file')
%
% Decode the array huffmanCode with dynamic Huffman code (array of integers
% or characters '1' and '0') into array sourceMessage.
%
% sourceMessage = VITTER673('mode', 'decode', 'input', huffmanCode)
%
% More About
%
% VITTER673 is a MATLAB version of the Pascal program, which implements
% the dynamic Huffman coding algorithm (algorithm Lambda). Pascal program
% is presented in the paper "Algorithm 673: Dynamic Huffman Coding" by
% J.S. Vitter, ACM Transactions on Mathematical Software, Vol. 15, No. 2,
% June 1989, Pages 158-167.
%
% The original well-known Huffman algorithm needs two passes for
% data compression. The dynamic Huffman algorithm constructs dynamic
% Huffman codes in one pass that provides strong advantage and allows an
% application of the algorithm for online compression and data transmission.
%
% The main difference of the MATLAB version from the Pascal version is an
% output of the resulting Huffman codeword for each input symbol and the
% resulting Huffman tree for the whole input. It is implemented using the
% functions analyzeTree() and getHuffmanCode().
%
% There are few other differences between the MATLAB and Pascal versions:
% 1. In the MATLAB version, the alphabet size is exactly 256 symbols.
% 2. In the MATLAB version, the input and output can be either arrays or
% files. Therefore, the original functions receive() and readString() are
% removed, and the functions bitReceive(), encodeAndTransmit(),
% getOptions(), putName() are modified.
% 3. The 'end-of-line' character in the input array or file is processed as
% any other character.
% 4. Original Pascal version handles character values 1 through 256.
% The MATLAB version handles the range of single byte values 0 through 255.
%
% Information about MATLAB version, revision and date
%
% MATLAB version: R2013b
% Revision:       4.1
% Date:           2015/03/01
%
% The current implementation is not intended to be used for large-scale
% problem instances.
%
% Authors
%
% A. Novoselsky, The Weizmann Institute of Science
% E. Kagan, Ariel University and the Weizmann Institute of Science

% - - - - - - - - - - - - - - - - - - - - - - - -
% constants

ROLE_ENCODE     = 'encode';
ROLE_DECODE     = 'decode';

INPUT_INTEGER   = 'integer';
INPUT_CHARACTER = 'character';

INPUTKIND_ARRAY = 'array';
INPUTKIND_FILE  = 'file';

n = int32(256);         % alphabet size

% - - - - - - - - - - - - - - - - - - - - - - - -
% global variables

pointerInputForDecode = int32(1);

outputArray = [];

stack     = zeros(1, n, 'int32');
alpha     = zeros(1, n, 'int32');
rep       = zeros(1, n, 'int32');

M = int32(0);           % number of zero-weight letters in the alphabet
E = int32(0);
R = int32(0);
availBlock = int32(0);
Z = int32(0);

parent    = zeros(1, 2*n, 'int32');
rtChild   = zeros(1, 2*n, 'int32');
parity    = zeros(1, 2*n, 'int32');
block     = zeros(1, 2*n, 'int32');
prevBlock = zeros(1, 2*n, 'int32');
nextBlock = zeros(1, 2*n, 'int32');
first     = zeros(1, 2*n, 'int32');
last      = zeros(1, 2*n, 'int32');
weight    = zeros(1, 2*n, 'int32');

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function returnValue = findChild (j, parity) % line 67 in 673.pas

        delta = 2 * (first(block(j)) - j) + 1 - int32(parity);

        right = rtChild(block(j));
        gap = right - last(block(right));

        if delta <= gap
            returnValue = right - delta;
        else
            delta = delta - gap - 1;

            right = first(prevBlock(block(right)));
            gap = right - last(block(right));

            if delta <= gap
                returnValue = right - delta;
            else
                returnValue = first(prevBlock(block(right))) - delta + gap + 1;
            end

        end

    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function returnValue = bitReceive() % line 97 in 673.pas

        returnValue = int32(inputArray(pointerInputForDecode) - 48);    % convert char to integer 

        pointerInputForDecode = pointerInputForDecode + 1;
    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function encodeAndTransmit(k, inputByteNumber)       % line 107 in 673.pas

        q = rep(k);
        i = int32(0);

        if q <= M           % Encode letter of zero weight
            q = q - 1;

            if q < 2 * R
                t = E + 1;
            else
                q = q - R;
                t = E;
            end

            for ii = 1 : t
                i = i + 1;
                stack(i) = mod(q, 2);
                q = idivide(q, 2);
            end

            q = M;
        end

        if M == n
            root = n;
        else
            root = Z;
        end

        while q ~= root     % Traverse up the tree
            i = i + 1;
            stack(i) = mod( first(block(q)) - q + parity(block(q)), 2);

            qTemp = first(block(q)) - q + 1 - parity(block(q));
            qTemp = bitshift(qTemp, -1);
            q = parent(block(q)) - qTemp;
        end

        % Following the indicated paper by J.S. Vitter (ACM Transactions on 
        % Mathematical Software, Vol. 15, No. 2, June 1989, Pages 158-167):
        % "this version of the algorithm outputs each bit in a simple-minded way,
        % which in most implentations uses one byte to store each bit.  This can
        % be easily fixed for any given implementation.  We have it as is so that
        % the implementor can more easily interpret the codes by hand in case there
        % are any problems during implementation.  Later, when everything is verified,
        % a more efficient output routine that packs the bits together
        % can be used instead".

        % convert array of single digit integers to string
        letterCode = char(stack(i : -1 : 1) + 48);

        outputArray{inputByteNumber} = letterCode;

        if auxiliaryOutput
            maxLineLength = 256;

            outputArrayToPrint = strjoin(outputArray(1:inputByteNumber), '');

            % convert range 1..256 to 0..255 by subtraction of 1

            if length(outputArrayToPrint) > maxLineLength
                fprintf('Letter: %s, Code: %10s\n', ...
                    char(k - 1), letterCode);
            else
                fprintf('Letter: %s, Code: %10s, Output: %s\n', ...
                    char(k - 1), letterCode, outputArrayToPrint);
            end
        end

    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function returnValue = receiveAndDecode()         % line 148 in 673.pas

        % Set |q| to the root node
        if M == n
            q = n;
        else
            q = Z;
        end

        %  Traverse down the tree
        while q > n
            q = findChild(q, bitReceive());
        end

        if q == M % Decode 0-node
            q = 0;

            for j = 1 : E
                q = 2 * q + bitReceive();
            end

            if q < R
                q = 2 * q + bitReceive();
            else
                q = q + R;
            end

            q = q + 1;
        end;

        returnValue = alpha(q);

    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function interchangeLeaves(e1, e2)  % line 164 in 673.pas

        rep(alpha(e1)) = e2;
        rep(alpha(e2)) = e1;

        temp = alpha(e1);
        alpha(e1) = alpha(e2);
        alpha(e2) = temp;

    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function update(k)                  % line 171 in 673.pas

        q           = int32(0);
        leafToIncrement = int32(0);
        bq          = int32(0);
        b           = int32(0);
        oldParent   = int32(0);
        oldParity   = int32(0);
        nbq         = int32(0);
        par         = int32(0);
        bpar        = int32(0);

        slide       = false;

        % - - - - - - - - - - - - - - - - - - - - - - - -
        % nested function, in function update()
        function findNode()             % line 176 in 673.pas

            q = rep(k);
            leafToIncrement = 0;

            if q <= M           % A zero weight becomes positive
                interchangeLeaves(q, M);

                if R == 0
                    R = idivide(M, 2);

                    if R > 0
                        E = E - 1;
                    end
                end

                M = M - 1;
                R = R - 1;
                q = M + 1;
                bq = block(q);

                if M > 0
                    % New 0-node is node |M|; old 0-node is node |M + 1|;
                    % new parent of nodes |M| and |M + 1| is node |M + n|
                    block(M) = bq;
                    last(bq) = M;
                    oldParent = parent(bq);

                    parent(bq) = M + n;
                    parity(bq) = 1;

                    % Create new internal block of zero weight for node |M + n|
                    b = availBlock;
                    availBlock = nextBlock(availBlock);

                    prevBlock(b) = bq;
                    nextBlock(b) = nextBlock(bq);

                    prevBlock(nextBlock(bq)) = b;
                    nextBlock(bq) = b;

                    parent(b) = oldParent;
                    parity(b) = 0;
                    rtChild(b) = q;

                    block(M + n) = b;
                    weight(b) = 0;

                    first(b) = M + n;
                    last(b) = M + n;

                    leafToIncrement = q;
                    q = M + n;
                end
            else                % Interchange |q| with the first node in |q|'s block
                interchangeLeaves(q, first(block(q)));
                q = first(block(q));

                if (q == M + 1) && (M > 0)
                    leafToIncrement = q;
                    q = parent(block(q));
                end
            end

        end     % function findNode()

        % - - - - - - - - - - - - - - - - - - - - - - - -
        % nested function, in function update()
        function slideAndIncrement()    % line 209 in 673.pas

            % |q| is currently the first node in its block

            bq = block(q);
            nbq = nextBlock(bq);

            par = parent(bq);
            oldParent = par;
            oldParity = parity(bq);

            if ((q <= n) && (first(nbq) > n) && (weight(nbq) == weight(bq))) || ((q > n) && (first(nbq) <= n) && (weight(nbq) == weight(bq) + 1))
                % Slide |q| over the next block
                slide = true;

                oldParent = parent(nbq);
                oldParity = parity(nbq);

                % Adjust child pointers for next-higher level in tree
                if par > 0
                    bpar = block(par);

                    if rtChild(bpar) == q
                        rtChild(bpar) = last(nbq);
                    else
                        if rtChild(bpar) == first(nbq)
                            rtChild(bpar) = q;
                        else
                            rtChild(bpar) = rtChild(bpar) + 1;
                        end
                    end

                    if par ~= Z
                        if block(par + 1) ~= bpar
                            if rtChild(block(par + 1)) == first(nbq)
                                rtChild(block(par + 1)) = q;
                            else
                                if block(rtChild(block(par + 1))) == nbq
                                    rtChild(block(par + 1)) = rtChild(block(par + 1)) + 1;
                                end
                            end
                        end
                    end

                end % par > 0

                % Adjust parent pointers for block |nbq|

                parent(nbq) = parent(nbq) -1 +parity(nbq);
                parity(nbq) = 1 - parity(nbq);

                nbq = nextBlock(nbq);
            else
                slide = false;
            end

            if (((q <= n) && (first(nbq) <= n)) || ((q > n) && (first(nbq) > n))) && (weight(nbq) == weight(bq) + 1)
                % Merge |q| into the block of weight one higher

                block(q) = nbq;
                last(nbq) = q;

                if last(bq) == q
                    % |q|'s old block disappears
                    nextBlock(prevBlock(bq)) = nextBlock(bq);
                    prevBlock(nextBlock(bq)) = prevBlock(bq);
                    nextBlock(bq) = availBlock;
                    availBlock = bq;
                else
                    if q > n
                        rtChild(bq) = findChild(q - 1, 1);
                    end

                    if parity(bq) == 0
                        parent(bq) = parent(bq) - 1;
                    end

                    parity(bq) = 1 - parity(bq);
                    first(bq) = q - 1;
                end
            else
                if last(bq) == q
                    if slide
                        % |q|'s block is slid forward in the block list
                        prevBlock(nextBlock(bq)) = prevBlock(bq);
                        nextBlock(prevBlock(bq)) = nextBlock(bq);

                        prevBlock(bq) = prevBlock(nbq);
                        nextBlock(bq) = nbq;

                        prevBlock(nbq) = bq;
                        nextBlock(prevBlock(bq)) = bq;

                        parent(bq) = oldParent;
                        parity(bq) = oldParity;
                    end

                    weight(bq) = weight(bq) + 1;
                else
                    % A new block is created for |q|
                    b = availBlock;
                    availBlock = nextBlock(availBlock);

                    block(q) = b;
                    first(b) = q;
                    last(b)  = q;

                    if q > n
                        rtChild(b) = rtChild(bq);
                        rtChild(bq) = findChild(q - 1, 1);

                        if rtChild(b) == q - 1
                            parent(bq) = q;
                        else
                            if parity(bq) == 0
                                parent(bq) = parent(bq) - 1;
                            end
                        end
                    else
                        if parity(bq) == 0
                            parent(bq) = parent(bq) - 1;
                        end
                    end

                    first(bq) = q - 1;
                    parity(bq) = 1 - parity(bq);

                    % Insert |q|'s block in its proper place in the block list
                    prevBlock(b) = prevBlock(nbq);
                    nextBlock(b) = nbq;

                    prevBlock(nbq) = b;
                    nextBlock(prevBlock(b)) = b;

                    weight(b) = weight(bq) + 1;

                    parent(b) = oldParent;
                    parity(b) = oldParity;
                end
            end

            % Move |q| one level higher in the tree
            if q <= n
                q = oldParent;
            else
                q = par;
            end

        end     % function slideAndIncrement()


        % function update(), begin, line 286 in 673.pas

        % Set |q| to the node whose weight should increase
        findNode();

        while q > 0
            % At this point, |q| is the first node in its block.  Increment
            % |q|'s weight by 1 and slide if necessary to maintain invariant (*)
            slideAndIncrement();
        end

        % Finish up some special cases involving the 0-node
        if leafToIncrement ~= 0
            q = leafToIncrement;
            slideAndIncrement();
        end

    end     % function update()

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function initialize()               % line 301 in 673.pas

        M = int32(0);
        E = int32(0);
        R = int32(-1);
        Z = 2 * n - 1;

        for i = 1 : n
            M = M + 1;
            R = R + 1;

            if 2 * R == M
                E = E + 1;
                R = int32(0);
            end
        end

        alpha(1 : n) = 1 : n;
        rep(1 : n) = 1 : n;

        % Initialize node |n| as the 0-node

        block(n)     = 1;
        prevBlock(1) = 1;
        nextBlock(1) = 1;
        weight(1)    = 0;

        first(1)  = n;
        last(1)   = n;
        parity(1) = 0;
        parent(1) = 0;

        % Initialize available block list

        availBlock = int32(2);

        i = availBlock : (Z - 1);
        nextBlock(i) = i + 1;

        nextBlock(Z) = 0;

    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function getOptions()               % line 329 in 673.pas

        if ischar(inputArray)
            if strcmp(theRole, ROLE_DECODE) && any( regexp(inputArray, '[^01]') )
                error('Wrong input argument: Array should contain only ''0'' and ''1'' ');
            end

            inputType = INPUT_CHARACTER;
        else
            if ~isinteger(inputArray)
                error('Wrong input argument: Array should contain only integers');
            end

            if strcmp(theRole, ROLE_ENCODE) && ...
                    ( any(inputArray < 0) || any(inputArray > 255) )
                error('Wrong input argument: Array should contain only integers in range 0...255');
            end

            if strcmp(theRole, ROLE_DECODE)
                if any(inputArray < 0) || any(inputArray > 1)
                    error('Wrong input argument: Array should contain only ''0'' and ''1'' ');
                end
                
                % convert array of single digit integers to string
                inputArray = char(inputArray + 48);
            end

            inputType = INPUT_INTEGER;
        end

        fprintf('Type of input:\t%s array\n', inputType);

    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function putName(val, inputByteNumber)  % line 352 in 673.pas

        % convert range 1..256 to 0..255 by subtraction of 1
        val = val - 1;

        outputArray{inputByteNumber} = val;

    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function runIt()                    % line 359 in 673.pas

        inputSize = length(inputArray);

        fprintf('Input size:\t\t%d bytes\n', inputSize);

        outputArray = cell(1, inputSize);

        if strcmp(theRole, ROLE_ENCODE)

            for i = 1 : inputSize
                % convert char to integer
                c = int32( inputArray(i) );

                % convert range 0..255 to 1..256 by addition of 1
                c = c + 1;

                encodeAndTransmit(c, i);
                update(c);
            end

        else
            for i = 1 : inputSize
                if pointerInputForDecode <= inputSize
                    c = receiveAndDecode();
                    putName(c, i);
                    update(c);
                end
            end
        end

    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function analyzeTree()

        fprintf('\nAlphabet size:\t\t%3d\n', n);
        fprintf('0-weight letters:\t%3d\n', M);
        fprintf('Distinct letters:\t%3d\n', n-M);

        if M > 0
            fprintf('  Letter NYT     is represented by: ');
            getHuffmanCode(M);
            fprintf('     Node: %3d, Block: %3d, Node weight: %d\n', M, block(M), weight(block(M)));
        end

        for i = M+1 : n
            % convert range 1..256 to 0..255 by subtraction of 1
            fprintf('  Letter %3d (%s) is represented by: ', alpha(i) - 1, char(alpha(i) - 1));

            getHuffmanCode(i);
            fprintf('     Node: %3d, Block: %3d, Node weight: %d\n', i, block(i), weight(block(i)));

            if first(block(i)) == i
                fprintf('     Node is leader: yes, its parent: %d',  parent(block(i)));
                if parity(block(i)) == 0
                     fprintf(', Left child\n');
                else
                     fprintf('\n');
                end
            else
                fprintf('     Node is leader:  no\n');
            end
        end

        blockFirst = n + max(1, M);
        blockLast  = 2*n - 1;

        internalNodes = blockLast - blockFirst +1;
        fprintf('\nInternal nodes:\t\t%3d\n', internalNodes);

        for i = blockFirst : blockLast
            fprintf('     Node: %3d, Block: %3d, Node weight: %d\n', i, block(i), weight(block(i)));

            if first(block(i)) == i
                fprintf('     Node is leader: yes, its parent: %d',  parent(block(i)));
                fprintf(', Its r/child: %d\n', rtChild(block(i)));
            else
                fprintf('     Node is leader:  no\n');
            end
        end

    end

    % - - - - - - - - - - - - - - - - - - - - - - - -
    % nested function
    function getHuffmanCode(q)

        i = int32(0);

        while q ~= Z     % Traverse up the tree
            i = i + 1;
            stack(i) = mod( first(block(q)) - q + parity(block(q)), 2);
            q = parent(block(q)) - idivide( first(block(q)) - q + 1 - parity(block(q)), 2);
        end

        % convert array of single digit integers to string
        code = char(stack(i : -1 : 1) + 48);

        fprintf('%s\n', code);

    end

% - - - - - - - - - - - - - - - - - - - - - - - -
% begin of main function, line 381 in 673.pas

if nargin == 0
    error('Input arguments are NOT passed to function')
end

% number of input arguments must be 4-10
narginchk(4, 10)

% must have name/value pairs
if mod(length(varargin), 2) ~= 0
    error('Number of input arguments should be even (arguments should be name/value pairs)')
end

inputKind = INPUTKIND_ARRAY;
outputFilename = '';
auxiliaryOutput = true;

okParName = {'mode', 'input', 'inputKind', ...
    'outputFilename', 'auxiliaryOutput'};

for curArg = 1 : 2 : nargin
    parName  = varargin{curArg};
    parValue = varargin{curArg + 1};

    k = find(strcmpi(parName, okParName));

    if isempty(k)
        error( ['Wrong input argument: Unknown parameter ''', parName, ''''] )
    else
        switch(k)
            case 1 % mode
                theRole = parValue;

                okRole = {ROLE_ENCODE, ROLE_DECODE};

                checkResult = strcmpi(theRole, okRole); 

                if sum(checkResult) == 0
                        error( ['Wrong input argument: Mode should be ''', ...
                            ROLE_ENCODE, ''' or ''', ROLE_DECODE, ''''] );
                else
                    fprintf('Mode of coding:\t%s\n', theRole);
                end

            case 2 % input

                inputArray = parValue;

            case 3 % inputKind

                inputKind = parValue;

                okInputKind = {INPUTKIND_ARRAY, INPUTKIND_FILE};

                checkResult = strcmpi(inputKind, okInputKind);

                if sum(checkResult) == 0
                    error( ['Wrong input argument: Input kind should be ''', ...
                        INPUTKIND_ARRAY, ''' or ''', INPUTKIND_FILE, ''''] );
                else
                    fprintf('Input kind:\t\t%s\n', inputKind);
                end

            case 4 % outputFilename

                outputFilename = parValue;

            case 5 % auxiliaryOutput

                switch parValue
                    case 'on'
                        auxiliaryOutput = true;
                    case 'off'
                        auxiliaryOutput = false;
                    otherwise
                        error('Wrong input argument: auxiliary output should be ''on'' or ''off'' ');
                end
        end
    end
end

if strcmp(inputKind, INPUTKIND_FILE)
    % read input file

    status = exist(inputArray, 'file');

    if status == 2
        fprintf('Input file:\t\t%s\n', inputArray);
    else
        error( ['File ''', inputArray, ''' does not exist'] )
    end
    
    fileID = fopen(inputArray);

    inputArray = int16( fread(fileID) );

    if strcmp(theRole, ROLE_DECODE)
        if all(inputArray >= 0) && all(inputArray <= 1)
            fprintf('Type of file:\tbinary file of ''1'' and ''0''\n');
        elseif all(inputArray >= 48) && all(inputArray <=49)
            fprintf('Type of file:\ttext file of ''1'' and ''0''\n');
            inputArray = char(inputArray)';
        end
    end

    fprintf('Size:\t\t\t%d bytes\n', length(inputArray));

    fclose(fileID);
end

getOptions();
initialize();
runIt();

if strcmp(theRole, ROLE_ENCODE)
    outputArray = strjoin(outputArray, '');
else
    outputArray = [outputArray{:}];
end

% if input is integer array, convert output to integer array
if strcmp(inputType, INPUT_INTEGER)
    if strcmp(theRole, ROLE_ENCODE)
        outputArray = int8(outputArray) - 48;
    elseif strcmp(theRole, ROLE_DECODE)
        outputArray = int16(outputArray);
    end
else
    outputArray = char(outputArray);
end

if auxiliaryOutput
    analyzeTree();
end

if ~isempty(outputFilename)
    % write output file

    fprintf('Output file:\t%s\n', outputFilename);

    fileID = fopen(outputFilename, 'w');

    writtenBytes = fwrite(fileID, outputArray);
    fprintf('Size:\t\t\t%d bytes\n', writtenBytes);

    fclose(fileID);
end

end % end of main function