/*******************************************************************************

   This module stores all the precomputed coordinates of the lattices which
   comprise the lattice augmentation sequence generated from the seed lattice
   (1,230, 311)/823 after the Sidi6 periodizing transformation has been
   applied. (See Li & Robinson, LNCS 2007, for details of the sequence and its
   goodness in terms of index of merit and trignometric degree of precision.)
   
   Each of the points in the (transformed) seed lattice and 2's-copy lattices
   can be constructed from different combinations of the precomputed
   coordinates. 
  
   Steps
    1. Generate the array C0: C0[i] = i/823, i = 0,...,822.
    2. Shift the elements of C0 by 0.5 modulo 1 to get C1.
    3. Shift C0+C1 by 0.25 modulo 1 to get C2.
    4. Shift C0+C1+C2 by 0.125 modulo 1 to get C3.
    5. Then C = C0+C1+C2+C3.
    6. Apply the Sidi6 periodizing transformation to C to get 'coords'.
    7. Apply the first derivative transformation to C to get 'weights'.
  
   All computations have been carried out in high precision to ensure the 
   values are correct to 16 significant figures.
  
   Also included is an 'offsets' array that is needed to retrieve, from the
   precomputed values, the coordinates and weight for an individual cubature
   point.

 ********************************************************************************/

#include <precomputedCoords.h>

// Precomputed coordinates
static const double coords[] = {
    0,
    1.7184915697668805e-18,
    2.1995944221355406e-16,
    3.758000323533539e-15,
    2.8150979819971717e-14,
    1.342206409832805e-13,
    4.8087687058864426e-13,
    1.4144820149690352e-12,
    3.6013657341435971e-12,
    8.2120429153767818e-12,
    1.716564492597901e-11,
    3.3443025269046024e-11,
    6.147694454506526e-11,
    1.0762767655128267e-10,
    1.8075230740804855e-10,
    2.9287592282988868e-10,
    4.5997279510581026e-10,
    7.0286559106955164e-10,
    1.0482505253814559e-09,
    1.5298562798688343e-09,
    2.1897443995445983e-09,
    3.0797587593121402e-09,
    4.2631315723394728e-09,
    5.8162532817230087e-09,
    7.8306135414405344e-09,
    1.0414920350797134e-08,
    1.3697404258684287e-08,
    1.7828314400091903e-08,
    2.2982612967530332e-08,
    2.9362874554432803e-08,
    3.7202396636319974e-08,
    4.6768527278623221e-08,
    5.8366215977690367e-08,
    7.234179335374996e-08,
    8.9086985221602713e-08,
    1.0904316636666191e-07,
    1.3270585915079888e-07,
    1.6062948186439291e-07,
    1.9343235152816357e-07,
    2.3180194563091084e-07,
    2.7650042706733857e-07,
    3.2837043631382683e-07,
    3.8834115464948946e-07,
    4.5743464199524904e-07,
    5.3677245270512444e-07,
    6.2758253240160901e-07,
    7.3120639870106736e-07,
    8.4910660842565395e-07,
    9.8287451364550149e-07,
    1.1342383086390181e-06,
    1.3050713696002067e-06,
    1.4974008886601619e-06,
    1.7134168035254531e-06,
    1.9554810237691475e-06,
    2.2261369545409268e-06,
    2.5281193181912748e-06,
    2.8643642740312143e-06,
    3.2380198361737804e-06,
    3.6524565891264071e-06,
    4.1112787005249692e-06,
    4.6183352301204273e-06,
    5.1777317338481326e-06,
    5.7938421615279906e-06,
    6.4713210464610397e-06,
    7.2151159849047889e-06,
    8.0304804031260061e-06,
    8.9229866094457849e-06,
    9.8985391284077996e-06,
    1.0963388313916827e-05,
    1.2124144237911211e-05,
    1.3387790850849869e-05,
    1.4761700410012208e-05,
    1.6253648171327858e-05,
    1.7871827340172671e-05,
    1.9624864276288385e-05,
    2.152183394770543e-05,
    2.3572275628272302e-05,
    2.5786208833120484e-05,
    2.8174149486121366e-05,
    3.0747126313121396e-05,
    3.3516697454473619e-05,
    3.6494967290118209e-05,
    3.969460347020171e-05,
    4.3128854143964797e-05,
    4.6811565379371097e-05,
    5.0757198765696063e-05,
    5.4980849191044135e-05,
    5.9498262786515644e-05,
    6.4325855028501538e-05,
    6.9480728990344698e-05,
    7.4980693734371181e-05,
    8.0844282835063327e-05,
    8.7090773023920301e-05,
    9.3740202946328776e-05,
    0.00010081339202054902,
    0.00010833195938870885,
    0.00011631834294948968,
    0.00012479581846198666,
    0.00013378851871002662,
    0.00014332145271603612,
    0.00015342052499336484,
    0.00016411255482578899,
    0.00017542529556274374,
    0.00018738745391866553,
    0.00020002870926466096,
    0.00021337973290056308,
    0.00022747220729528428,
    0.00024233884528323242,
    0.0002580134092044175,
    0.0002745307299757471,
    0.00029192672608088327,
    0.00031023842246591734,
    0.00032950396932800846,
    0.00034976266078402937,
    0.00037105495340616494,
    0.00039342248461132334,
    0.00041690809089113599,
    0.00044155582586924945,
    0.00046741097817254673,
    0.00049452008910287506,
    0.00052293097009580838,
    0.00055269271995292727,
    0.00058385574183406487,
    0.00061647175999593896,
    0.00065059383626357172,
    0.00068627638622088425,
    0.00072357519510685251,
    0.00076254743340361272,
    0.00080325167210291797,
    0.00084574789763736764,
    0.00089009752646285981,
    0.00093636341927875381,
    0.00098460989487227373,
    0.0010349027435737389,
    0.001087309240309266,
    0.0011418981572376579,
    0.0011987397759582736,
    0.0012579058992767567,
    0.0013194698625155939,
    0.0013835065443565809,
    0.0014500923772023796,
    0.001519305357044467,
    0.0015912250528249123,
    0.00166593261527954,
    0.0017435107852501891,
    0.0018240439014539284,
    0.0019076179076972398,
    0.0019943203595233542,
    0.0020842404302811,
    0.0021774689166037969,
    0.002274098243286931,
    0.0023742224675535337,
    0.0024779372826963944,
    0.0025853400210864548,
    0.0026965296565369477,
    0.0028116068060130678,
    0.0029306737306772053,
    0.0030538343362600069,
    0.0031811941727477788,
    0.0033128604333770083,
    0.0034489419529270312,
    0.0035895492053021551,
    0.0037347943003948158,
    0.003884790980221623,
    0.0040396546143244561,
    0.004199502194429044,
    0.0043644523283537882,
    0.0045346252331618661,
    0.0047101427275499973,
    0.0048911282234675481,
    0.0050777067169599874,
    0.0052700047782310291,
    0.0054681505409181408,
    0.0056722736905764235,
    0.0058825054523662336,
    0.0060989785779402414,
    0.0063218273315260014,
    0.0065511874752004423,
    0.0067871962533530804,
    0.007029992376335086,
    0.0072797160032917439,
    0.0075365087241761964,
    0.0078005135409427457,
    0.0080718748479183828,
    0.0083507384113515673,
    0.0086372513481377203,
    0.0089315621037212144,
    0.0092338204291741114,
    0.0095441773574522423,
    0.0098627851788296427,
    0.010189797415512757,
    0.010525368795436224,
    0.010869655225242462,
    0.011222813762447678,
    0.011585002586797314,
    0.011956380970814407,
    0.012337109249544653,
    0.012727348789502478,
    0.013127261956822765,
    0.013537012084623287,
    0.013956763439583374,
    0.014386681187744671,
    0.014826931359540311,
    0.015277680814059197,
    0.015739097202552534,
    0.016211348931190129,
    0.016694605123074361,
    0.017189035579520211,
    0.017694810740610065,
    0.018212101645032383,
    0.018741079889213863,
    0.019281917585754947,
    0.019834787321179055,
    0.020399862113006213,
    0.020977315366162239,
    0.021567320828734905,
    0.022170052547089016,
    0.022785684820352606,
    0.023414392154286902,
    0.024056349214553019,
    0.024711730779388784,
    0.025380711691709381,
    0.026063466810645874,
    0.026760170962536117,
    0.027470998891382725,
    0.028196125208793301,
    0.028935724343418335,
    0.029689970489902587,
    0.030459037557366023,
    0.031243099117430818,
    0.032042328351811061,
    0.032856897999482305,
    0.033686980303448293,
    0.034532746957122461,
    0.035394369050342218,
    0.036272017015034179,
    0.037165860570548874,
    0.038076068668683634,
    0.039002809438412715,
    0.039946250130343919,
    0.040906557060921202,
    0.041883895556393054,
    0.042878429896566539,
    0.043890323258367288,
    0.044919737659225807,
    0.045966833900310651,
    0.047031771509629387,
    0.048114708685018173,
    0.049215802237041285,
    0.05033520753182176,
    0.051473078433824787,
    0.052629567248615373,
    0.053804824665612129,
    0.05499899970085903,
    0.056212239639837298,
    0.057444689980339352,
    0.058696494375427261,
    0.059967794576497986,
    0.061258730376477707,
    0.06256943955316796,
    0.063900057812765892,
    0.065250718733581528,
    0.066621553709974302,
    0.068012691896532099,
    0.069424260152514863,
    0.070856382986586169,
    0.072309182501854849,
    0.073782778341249911,
    0.07527728763325113,
    0.076792824937998119,
    0.078329502193800457,
    0.079887428664071503,
    0.081466710884708415,
    0.083067452611940831,
    0.084689754770670506,
    0.08633371540332431,
    0.0879994296192428,
    0.0896869895446262,
    0.091396484273060088,
    0.093127999816642285,
    0.094881619057732888,
    0.096657421701348847,
    0.098455484228224352,
    0.10027587984855856,
    0.10211867845647124,
    0.10398394658518761,
    0.10587174736297247,
    0.10778214046983459,
    0.10971518209502111,
    0.11167092489532214,
    0.11364941795420515,
    0.11565070674179867,
    0.11767483307574457,
    0.11972183508293766,
    0.12179174716217143,
    0.12388459994770834,
    0.12600042027379246,
    0.1281392311401226,
    0.13030105167830294,
    0.13248589711928888,
    0.13469377876184413,
    0.13692470394202633,
    0.1391786760037165,
    0.14145569427020885,
    0.14375575401687565,
    0.14607884644492267,
    0.14842495865624938,
    0.15079407362942873,
    0.15318617019681974,
    0.15560122302282667,
    0.15803920258331808,
    0.16050007514621759,
    0.16298380275327937,
    0.1654903432030598,
    0.16801965003509653,
    0.17057167251530586,
    0.17314635562260913,
    0.17574364003679788,
    0.17836346212764725,
    0.18100575394528698,
    0.18367044321183831,
    0.18635745331432507,
    0.18906670329886646,
    0.19179810786615892,
    0.19455157736825354,
    0.1973270178066352,
    0.20012433083160924,
    0.20294341374300082,
    0.20578415949217149,
    0.20864645668535742,
    0.21153018958833258,
    0.21443523813240054,
    0.21736147792171681,
    0.22030878024194464,
    0.2232770120702452,
    0.22626603608660362,
    0.22927571068649139,
    0.23230588999486482,
    0.23535642388149966,
    0.23842715797766018,
    0.24151793369410185,
    0.24462858824040487,
    0.24775895464563644,
    0.25090886178033855,
    0.25407813437983728,
    0.2572665930688699,
    0.2604740543875248,
    0.26370033081848909,
    0.26694523081559796,
    0.27020855883367978,
    0.27349011535968981,
    0.27678969694512556,
    0.2801070962397158,
    0.28344210202637449,
    0.28679449925741168,
    0.29016406909199083,
    0.29355058893482383,
    0.29695383247609231,
    0.30037356973258533,
    0.3038095670900412,
    0.30726158734668202,
    0.3107293897579288,
    0.31421273008228356,
    0.31771136062836614,
    0.32122503030309085,
    0.32475348466096982,
    0.32829646595452711,
    0.33185371318580958,
    0.33542496215897816,
    0.33900994553396391,
    0.34260839288117162,
    0.3462200307372153,
    0.34984458266166668,
    0.35348176929480052,
    0.3571313084163168,
    0.36079291500502297,
    0.36446630129945617,
    0.36815117685942605,
    0.37184724862845892,
    0.37555422099712299,
    0.37927179586721405,
    0.38299967271678048,
    0.38673754866596705,
    0.39048511854365542,
    0.39424207495487984,
    0.39800810834899553,
    0.40178290708857717,
    0.40556615751902536,
    0.40935754403885727,
    0.41315674917065809,
    0.41696345363267001,
    0.42077733641099485,
    0.42459807483238526,
    0.42842534463760151,
    0.43225882005530769,
    0.43609817387648347,
    0.43994307752932604,
    0.44379320115461657,
    0.4476482136815263,
    0.4515077829038357,
    0.45537157555654206,
    0.459239257392828,
    0.46311049326136611,
    0.46698494718393246,
    0.47086228243330291,
    0.47474216161140559,
    0.47862424672770293,
    0.48250819927777638,
    0.4863936803220868,
    0.49028035056488423,
    0.49416787043323906,
    0.49805590015616868,
    0.50194409984383126,
    0.50583212956676094,
    0.50971964943511583,
    0.51360631967791326,
    0.51749180072222367,
    0.52137575327229702,
    0.52525783838859441,
    0.52913771756669714,
    0.53301505281606754,
    0.53688950673863389,
    0.540760742607172,
    0.54462842444345794,
    0.54849221709616425,
    0.5523517863184737,
    0.55620679884538338,
    0.56005692247067396,
    0.56390182612351658,
    0.56774117994469231,
    0.57157465536239849,
    0.57540192516761468,
    0.5792226635890052,
    0.58303654636732993,
    0.58684325082934197,
    0.59064245596114273,
    0.59443384248097464,
    0.59821709291142289,
    0.60199189165100453,
    0.60575792504512016,
    0.60951488145634458,
    0.61326245133403301,
    0.61700032728321952,
    0.62072820413278595,
    0.62444577900287701,
    0.62815275137154114,
    0.631848823140574,
    0.63553369870054377,
    0.63920708499497703,
    0.6428686915836832,
    0.64651823070519954,
    0.65015541733833326,
    0.6537799692627847,
    0.65739160711882838,
    0.66099005446603609,
    0.66457503784102179,
    0.66814628681419042,
    0.67170353404547289,
    0.67524651533903013,
    0.67877496969690909,
    0.68228863937163386,
    0.68578726991771644,
    0.68927061024207126,
    0.69273841265331793,
    0.69619043290995886,
    0.69962643026741467,
    0.70304616752390769,
    0.70644941106517622,
    0.70983593090800912,
    0.71320550074258837,
    0.71655789797362546,
    0.71989290376028425,
    0.72321030305487444,
    0.72650988464031019,
    0.72979144116632022,
    0.73305476918440204,
    0.73629966918151091,
    0.7395259456124752,
    0.7427334069311301,
    0.74592186562016272,
    0.74909113821966145,
    0.75224104535436354,
    0.7553714117595951,
    0.75848206630589809,
    0.76157284202233977,
    0.76464357611850031,
    0.76769411000513521,
    0.77072428931350856,
    0.77373396391339633,
    0.7767229879297548,
    0.77969121975805533,
    0.78263852207828322,
    0.78556476186759949,
    0.78846981041166742,
    0.79135354331464258,
    0.79421584050782845,
    0.79705658625699916,
    0.7998756691683907,
    0.8026729821933648,
    0.80544842263174643,
    0.80820189213384108,
    0.81093329670113357,
    0.81364254668567493,
    0.81632955678816166,
    0.81899424605471305,
    0.8216365378723528,
    0.82425635996320212,
    0.82685364437739084,
    0.82942832748469408,
    0.83198034996490344,
    0.83450965679694022,
    0.83701619724672061,
    0.83949992485378244,
    0.84196079741668195,
    0.84439877697717336,
    0.84681382980318032,
    0.84920592637057124,
    0.85157504134375062,
    0.85392115355507736,
    0.85624424598312432,
    0.85854430572979112,
    0.86082132399628353,
    0.86307529605797373,
    0.86530622123815581,
    0.86751410288071107,
    0.86969894832169703,
    0.8718607688598774,
    0.87399957972620756,
    0.87611540005229172,
    0.87820825283782855,
    0.88027816491706234,
    0.88232516692425544,
    0.88434929325820133,
    0.88635058204579487,
    0.88832907510467785,
    0.89028481790497893,
    0.89221785953016541,
    0.89412825263702755,
    0.89601605341481239,
    0.89788132154352873,
    0.89972412015144143,
    0.90154451577177563,
    0.90334257829865117,
    0.90511838094226715,
    0.90687200018335767,
    0.90860351572693987,
    0.91031301045537383,
    0.91200057038075721,
    0.91366628459667565,
    0.91531024522932947,
    0.91693254738805918,
    0.91853328911529153,
    0.9201125713359285,
    0.92167049780619958,
    0.92320717506200189,
    0.92472271236674886,
    0.92621722165875009,
    0.92769081749814519,
    0.92914361701341386,
    0.93057573984748509,
    0.93198730810346786,
    0.93337844629002564,
    0.93474928126641843,
    0.93609994218723414,
    0.93743056044683204,
    0.93874126962352233,
    0.94003220542350197,
    0.94130350562457277,
    0.94255531001966064,
    0.94378776036016265,
    0.94500100029914091,
    0.94619517533438791,
    0.94737043275138466,
    0.94852692156617524,
    0.94966479246817825,
    0.95078419776295875,
    0.95188529131498179,
    0.95296822849037066,
    0.95403316609968936,
    0.95508026234077414,
    0.95610967674163272,
    0.95712157010343346,
    0.95811610444360695,
    0.95909344293907883,
    0.96005374986965608,
    0.96099719056158728,
    0.96192393133131637,
    0.96283413942945117,
    0.96372798298496587,
    0.96460563094965779,
    0.9654672530428775,
    0.9663130196965517,
    0.96714310200051767,
    0.96795767164818891,
    0.96875690088256916,
    0.96954096244263399,
    0.97031002951009737,
    0.97106427565658171,
    0.97180387479120667,
    0.97252900110861729,
    0.97323982903746387,
    0.97393653318935414,
    0.97461928830829059,
    0.97528826922061118,
    0.97594365078544698,
    0.97658560784571313,
    0.97721431517964741,
    0.97782994745291096,
    0.97843267917126509,
    0.97902268463383779,
    0.97960013788699374,
    0.98016521267882095,
    0.98071808241424507,
    0.98125892011078619,
    0.9817878983549676,
    0.98230518925938992,
    0.98281096442047977,
    0.98330539487692559,
    0.98378865106880986,
    0.98426090279744749,
    0.98472231918594078,
    0.98517306864045973,
    0.98561331881225533,
    0.98604323656041659,
    0.98646298791537668,
    0.98687273804317721,
    0.98727265121049756,
    0.98766289075045532,
    0.98804361902918558,
    0.98841499741320271,
    0.98877718623755229,
    0.98913034477475759,
    0.9894746312045638,
    0.98981020258448726,
    0.99013721482117034,
    0.99045582264254772,
    0.99076617957082591,
    0.99106843789627874,
    0.99136274865186225,
    0.99164926158864841,
    0.99192812515208162,
    0.9921994864590572,
    0.99246349127582378,
    0.99272028399670831,
    0.99297000762366494,
    0.99321280374664689,
    0.99344881252479955,
    0.99367817266847402,
    0.99390102142205972,
    0.99411749454763376,
    0.99432772630942357,
    0.99453184945908191,
    0.99472999522176897,
    0.99492229328304005,
    0.9951088717765324,
    0.99528985727245001,
    0.99546537476683816,
    0.99563554767164619,
    0.99580049780557101,
    0.99596034538567557,
    0.99611520901977835,
    0.99626520569960519,
    0.99641045079469781,
    0.996551058047073,
    0.99668713956662303,
    0.99681880582725224,
    0.99694616566373995,
    0.99706932626932276,
    0.99718839319398689,
    0.99730347034346301,
    0.99741465997891354,
    0.99752206271730359,
    0.99762577753244641,
    0.99772590175671305,
    0.99782253108339625,
    0.9979157595697189,
    0.9980056796404766,
    0.99809238209230278,
    0.99817595609854604,
    0.99825648921474985,
    0.99833406738472041,
    0.99840877494717506,
    0.99848069464295552,
    0.99854990762279761,
    0.99861649345564341,
    0.99868053013748437,
    0.99874209410072323,
    0.9988012602240417,
    0.99885810184276236,
    0.99891269075969069,
    0.99896509725642624,
    0.99901539010512774,
    0.99906363658072128,
    0.99910990247353715,
    0.99915425210236264,
    0.99919674832789707,
    0.99923745256659635,
    0.99927642480489309,
    0.99931372361377913,
    0.99934940616373646,
    0.99938352824000409,
    0.99941614425816594,
    0.99944730728004705,
    0.99947706902990419,
    0.99950547991089711,
    0.9995325890218274,
    0.99955844417413076,
    0.99958309190910888,
    0.99960657751538873,
    0.99962894504659383,
    0.99965023733921599,
    0.99967049603067204,
    0.99968976157753409,
    0.99970807327391908,
    0.99972546927002426,
    0.99974198659079561,
    0.99975766115471676,
    0.99977252779270476,
    0.99978662026709941,
    0.99979997129073539,
    0.99981261254608134,
    0.99982457470443731,
    0.99983588744517427,
    0.99984657947500666,
    0.99985667854728399,
    0.99986621148128996,
    0.99987520418153797,
    0.99988368165705055,
    0.99989166804061125,
    0.9998991866079795,
    0.99990625979705372,
    0.99991290922697607,
    0.99991915571716494,
    0.99992501930626565,
    0.99993051927100962,
    0.99993567414497153,
    0.99994050173721349,
    0.99994501915080891,
    0.99994924280123432,
    0.99995318843462067,
    0.99995687114585607,
    0.9999603053965298,
    0.99996350503270992,
    0.99996648330254556,
    0.99996925287368688,
    0.99997182585051392,
    0.99997421379116691,
    0.99997642772437167,
    0.99997847816605234,
    0.99998037513572369,
    0.99998212817265986,
    0.99998374635182863,
    0.99998523829958996,
    0.99998661220914919,
    0.9999878758557621,
    0.99998903661168603,
    0.99999010146087164,
    0.99999107701339052,
    0.9999919695195969,
    0.99999278488401511,
    0.99999352867895352,
    0.99999420615783852,
    0.99999482226826619,
    0.99999538166476987,
    0.99999588872129952,
    0.99999634754341082,
    0.99999676198016385,
    0.99999713563572601,
    0.99999747188068178,
    0.99999777386304545,
    0.99999804451897623,
    0.99999828658319645,
    0.99999850259911138,
    0.99999869492863036,
    0.99999886576169139,
    0.99999901712548633,
    0.99999915089339153,
    0.99999926879360135,
    0.99999937241746761,
    0.99999946322754729,
    0.99999954256535806,
    0.99999961165884532,
    0.99999967162956371,
    0.99999972349957289,
    0.99999976819805436,
    0.99999980656764842,
    0.99999983937051817,
    0.99999986729414081,
    0.99999989095683361,
    0.99999991091301477,
    0.99999992765820667,
    0.99999994163378403,
    0.99999995323147273,
    0.99999996279760339,
    0.99999997063712542,
    0.99999997701738708,
    0.99999998217168562,
    0.99999998630259579,
    0.99999998958507963,
    0.99999999216938651,
    0.99999999418374674,
    0.99999999573686837,
    0.99999999692024122,
    0.99999999781025561,
    0.99999999847014376,
    0.99999999895174951,
    0.99999999929713446,
    0.99999999954002716,
    0.99999999970712405,
    0.9999999998192477,
    0.99999999989237232,
    0.99999999993852307,
    0.99999999996655697,
    0.9999999999828344,
    0.9999999999917879,
    0.99999999999639866,
    0.99999999999858546,
    0.99999999999951916,
    0.99999999999986577,
    0.9999999999999718,
    0.99999999999999623,
    0.99999999999999978,
    1,
    0.5,
    0.50388815719586022,
    0.50777597447216072,
    0.51166311198858683,
    0.51554923006328679,
    0.51943398925203521,
    0.5233170504273158,
    0.52719807485729464,
    0.53107672428465891,
    0.53495266100529282,
    0.53882554794676429,
    0.5426950487465958,
    0.54656082783029158,
    0.55042255048909627,
    0.55427988295745712,
    0.55813249249016372,
    0.56198004743913998,
    0.56582221732985971,
    0.56965867293736316,
    0.57348908636184648,
    0.577313131103799,
    0.58113048213866281,
    0.58494081599099057,
    0.5887438108080747,
    0.59253914643302452,
    0.59632650447726621,
    0.60010556839244156,
    0.60387602354168124,
    0.60763755727022895,
    0.61138985897539233,
    0.61513262017579906,
    0.61886553457993243,
    0.6225882981539268,
    0.62630060918859787,
    0.6300021683656879,
    0.6336926788233026,
    0.63737184622051968,
    0.64103937880114581,
    0.6446949874566047,
    0.6483383857879329,
    0.65196929016686478,
    0.65558741979598678,
    0.65919249676794267,
    0.66278424612366904,
    0.66636239590964452,
    0.66992667723413424,
    0.67347682432241118,
    0.67701257457093866,
    0.68053366860049525,
    0.68403985030822845,
    0.68753086691861864,
    0.69100646903334018,
    0.6944664106800027,
    0.6979104493597601,
    0.70133834609377121,
    0.70474986546850049,
    0.70814477567984391,
    0.71152284857606907,
    0.71488385969955637,
    0.71822758832732969,
    0.72155381751036596,
    0.72486233411167289,
    0.72815292884312355,
    0.73142539630104086,
    0.73467953500051941,
    0.73791514740847919,
    0.74113203997544053,
    0.74433002316601471,
    0.74750891148810206,
    0.75066852352079116,
    0.75380868194095296,
    0.7569292135485256,
    0.76002994929048229,
    0.76311072428348115,
    0.76617137783519007,
    0.7692117534642855,
    0.77223169891912047,
    0.77523106619506055,
    0.77820971155048646,
    0.78116749552146014,
    0.78410428293505541,
    0.78701994292135236,
    0.78991434892409496,
    0.79278737871001415,
    0.79563891437681666,
    0.79846884235984172,
    0.80127705343738809,
    0.80406344273471464,
    0.80682790972671681,
    0.80957035823928469,
    0.8122906964493446,
    0.81498883688359192,
    0.81766469641591843,
    0.8203181962635413,
    0.82294926198183904,
    0.82555782345790318,
    0.82814381490281075,
    0.83070717484262768,
    0.83324784610814939,
    0.83576577582338918,
    0.83826091539282288,
    0.84073322048739907,
    0.84318265102932644,
    0.8456091711756476,
    0.84801274930061077,
    0.85039335797685223,
    0.85275097395539901,
    0.85508557814450659,
    0.8573971555873432,
    0.85968569543853446,
    0.86195119093958106,
    0.86419363939316551,
    0.86641304213635972,
    0.86860940451275093,
    0.8707827358434993,
    0.8729330493973434,
    0.8750603623595693,
    0.87716469579996059,
    0.87924607463974369,
    0.88130452761754818,
    0.8833400872543975,
    0.88535278981774745,
    0.88734267528459276,
    0.88930978730365728,
    0.89125417315668709,
    0.89317588371886647,
    0.89507497341837461,
    0.8969515001951025,
    0.89880552545855008,
    0.90063711404492275,
    0.90244633417344866,
    0.90423325740193528,
    0.90599795858158683,
    0.90774051581110415,
    0.9094610103900852,
    0.91115952677175038,
    0.91283615251501204,
    0.91449097823591041,
    0.91612409755843693,
    0.91773560706476731,
    0.91932560624492643,
    0.92089419744590584,
    0.92244148582025776,
    0.92396757927418605,
    0.92547258841515834,
    0.9269566264990593,
    0.92841980937691049,
    0.92986225544117684,
    0.93128408557168385,
    0.93268542308116709,
    0.93406639366047806,
    0.93542712532346772,
    0.93676774835157139,
    0.93808839523811716,
    0.93938920063238063,
    0.94067030128340945,
    0.94193183598363894,
    0.94317394551232214,
    0.94439677257879817,
    0.94560046176561818,
    0.94678515947155495,
    0.94795101385451619,
    0.94909817477438496,
    0.9502267937358081,
    0.95133702383095664,
    0.95242901968227911,
    0.95350293738526859,
    0.95455893445126783,
    0.95559716975033104,
    0.95661780345416614,
    0.95762099697917724,
    0.95860691292962907,
    0.95957571504095462,
    0.96052756812322471,
    0.96146263800480347,
    0.96238109147620665,
    0.96328309623418451,
    0.96416882082604904,
    0.96503843459426508,
    0.96589210762132438,
    0.96673001067492181,
    0.9675523151534533,
    0.96835919303185303,
    0.96915081680778947,
    0.96992735944823671,
    0.97068899433644085,
    0.9714358952192973,
    0.97216823615515713,
    0.97288619146207866,
    0.97358993566654228,
    0.97427964345264206,
    0.97495548961177381,
    0.97561764899283188,
    0.97626629645293139,
    0.976901606808672,
    0.97752375478795472,
    0.97813291498236965,
    0.97872926180016573,
    0.97931296941981738,
    0.97988421174420115,
    0.98044316235539453,
    0.98098999447011048,
    0.98152488089577827,
    0.98204799398728371,
    0.9825595056043791,
    0.98305958706977414,
    0.98354840912791885,
    0.98402614190448767,
    0.98449295486657473,
    0.98494901678361035,
    0.98539449568900717,
    0.98582955884254408,
    0.98625437269349647,
    0.98666910284452114,
    0.98707391401630173,
    0.9874689700129633,
    0.98785443368826109,
    0.98823046691255012,
    0.98859723054054194,
    0.98895488437985279,
    0.98930358716034894,
    0.98964349650429195,
    0.9899747688972913,
    0.99029755966006405,
    0.99061202292100725,
    0.99091831158958543,
    0.99121657733053581,
    0.99150697053889147,
    0.99178964031582728,
    0.99206473444532695,
    0.99233239937167317,
    0.99259278017776142,
    0.99284602056423732,
    0.99309226282945684,
    0.99333164785026873,
    0.99356431506361809,
    0.99379040244897043,
    0.99401004651155167,
    0.99422338226640539,
    0.99443054322326185,
    0.99463166137221692,
    0.99482686717021729,
    0.99501628952834831,
    0.99520005579992066,
    0.99537829176935,
    0.99555112164182724,
    0.99571866803377129,
    0.99588105196406151,
    0.99603839284604212,
    0.99619080848029373,
    0.99633841504816467,
    0.99648132710605508,
    0.99661965758044835,
    0.99675351776368015,
    0.99688301731043938,
    0.99700826423499156,
    0.9971293649091173,
    0.99724642406075736,
    0.99735954477335431,
    0.99746882848588325,
    0.99757437499356094,
    0.99767628244922479,
    0.99777464736537036,
    0.99786956461683984,
    0.997961127444148,
    0.99804942745743741,
    0.99813455464105172,
    0.99821659735871504,
    0.99829564235930701,
    0.99837177478322281,
    0.99844507816930528,
    0.99851563446233782,
    0.99858352402108674,
    0.99864882562688095,
    0.99871161649271523,
    0.99877197227286718,
    0.9988299670730133,
    0.99888567346083301,
    0.99893916247708614,
    0.99899050364715303,
    0.99903976499302305,
    0.99908701304571867,
    0.99913231285814141,
    0.9991757280183281,
    0.99921732066310209,
    0.99925715149210703,
    0.99929527978221022,
    0.99933176340226093,
    0.99936665882819098,
    0.99940002115844373,
    0.99943190412971794,
    0.99946236013301282,
    0.99949144022996061,
    0.99951919416943358,
    0.99954567040441089,
    0.99957091610909277,
    0.99959497719624768,
    0.9996178983347791,
    0.9996397229674987,
    0.99966049332909268,
    0.99968025046426645,
    0.99969903424605644,
    0.9997168833942941,
    0.99973383549420969,
    0.99974992701516241,
    0.99976519332948455,
    0.99977966873142532,
    0.99979338645618343,
    0.99980637869901434,
    0.99981867663439972,
    0.99983031043526782,
    0.99984130929225079,
    0.99985170143296842,
    0.99986151414132496,
    0.99987077377680755,
    0.99987950579377571,
    0.99988773476072823,
    0.99989548437953757,
    0.99990277750464074,
    0.99990963616217421,
    0.99991608156904399,
    0.99992213415191855,
    0.99992781356613514,
    0.99993313871450884,
    0.99993812776603497,
    0.99994279817447385,
    0.99994716669680983,
    0.99995124941157443,
    0.99995506173702409,
    0.99995861844916478,
    0.9999619336996135,
    0.99996502103328999,
    0.99996789340592818,
    0.99997056320140165,
    0.9999730422488543,
    0.99997534183962944,
    0.99997747274398885,
    0.99997944522761673,
    0.99998126906790119,
    0.99998295356998568,
    0.99998450758258728,
    0.99998593951357206,
    0.99998725734528593,
    0.99998846864963276,
    0.99998958060289611,
    0.99999060000030049,
    0.99999153327030554,
    0.99999238648863154,
    0.99999316539201066,
    0.99999387539166107,
    0.99999452158648017,
    0.99999510877595477,
    0.99999564147278452,
    0.99999612391521697,
    0.99999656007909099,
    0.99999695368958808,
    0.99999730823268929,
    0.99999762696633587,
    0.9999979129312937,
    0.99999816896171989,
    0.99999839769543197,
    0.99999860158387821,
    0.99999878290181032,
    0.99999894375665865,
    0.99999908609760946,
    0.99999921172438733,
    0.99999932229574129,
    0.99999941933763803,
    0.99999950425116335,
    0.99999957832013364,
    0.9999996427184189,
    0.99999969851698189,
    0.99999974669063363,
    0.99999978812451018,
    0.99999982362027207,
    0.9999998539020325,
    0.99999987962201464,
    0.99999990136594452,
    0.99999991965818325,
    0.99999993496660167,
    0.99999994770720302,
    0.99999995824849897,
    0.99999996691564264,
    0.99999997399432472,
    0.99999997973443822,
    0.99999998435351689,
    0.99999998803995371,
    0.99999999095600489,
    0.99999999324058597,
    0.99999999501186643,
    0.99999999636966874,
    0.99999999739767942,
    0.99999999816547791,
    0.99999999873039147,
    0.99999999913918214,
    0.99999999942957329,
    0.99999999963162367,
    0.9999999997689567,
    0.99999999985985044,
    0.99999999991819999,
    0.99999999995435573,
    0.99999999997584899,
    0.99999999998801126,
    0.9999999999944954,
    0.9999999999977075,
    0.9999999999991579,
    0.99999999999973843,
    0.99999999999993583,
    0.9999999999999889,
    0.999999999999999,
    1,
    1,
    1.3425829507343802e-20,
    2.9361623597863812e-17,
    1.0488216082569653e-15,
    1.1055244299683104e-14,
    6.4200772546621733e-14,
    2.6154249600072204e-13,
    8.4204998002433838e-13,
    2.2924734186359484e-12,
    5.504643429499166e-12,
    1.19887347649941e-11,
    2.4150980481190161e-11,
    4.5644270414065218e-11,
    8.1800008191850167e-11,
    1.4014953450043078e-10,
    2.3104335095604733e-10,
    3.6837629877103769e-10,
    5.7042675947319183e-10,
    8.608178513106965e-10,
    1.2696084947000413e-09,
    1.8345221132151153e-09,
    2.6023206232375837e-09,
    3.6303312455610583e-09,
    4.9881335460380928e-09,
    6.759413979857128e-09,
    9.0439950753175574e-09,
    1.1960046248120282e-08,
    1.5646483086297518e-08,
    2.0265561789062002e-08,
    2.6005675280158537e-08,
    3.3084357347850185e-08,
    4.1751500989571097e-08,
    5.2292796959635156e-08,
    6.503339833331502e-08,
    8.0341816710214084e-08,
    9.8634055484261318e-08,
    1.2037798540698762e-07,
    1.4609796746511479e-07,
    1.7637972788303318e-07,
    2.1187548984559033e-07,
    2.5330936631689468e-07,
    3.0148301810669101e-07,
    3.5728158110742733e-07,
    4.2167986639254793e-07,
    4.9574883662995798e-07,
    5.8066236202416008e-07,
    6.7770425875641056e-07,
    7.8827561264453498e-07,
    9.1390239049293531e-07,
    1.0562433413489669e-06,
    1.2170981896244283e-06,
    1.3984161217805417e-06,
    1.6023045680116818e-06,
    1.8310382800973877e-06,
    2.0870687063240461e-06,
    2.3730336641072171e-06,
    2.6917673106730786e-06,
    3.0463104118830341e-06,
    3.4399209090093638e-06,
    3.8760847829920553e-06,
    4.3585272154278101e-06,
    4.8912240452618566e-06,
    5.4784135198718099e-06,
    6.1246083389505355e-06,
    6.8346079893120366e-06,
    7.6135113684609178e-06,
    8.4667296944822075e-06,
    9.3999996995243927e-06,
    1.0419397103864658e-05,
    1.153135036726161e-05,
    1.2742654714017604e-05,
    1.4060486427890024e-05,
    1.5492417412709046e-05,
    1.7046430014278416e-05,
    1.8730932098856021e-05,
    2.0554772383232476e-05,
    2.2527256011148986e-05,
    2.4658160370520442e-05,
    2.6957751145656237e-05,
    2.9436798598399829e-05,
    3.2106594071838977e-05,
    3.4978966709971692e-05,
    3.8066300386448725e-05,
    4.1381550835251966e-05,
    4.493826297590966e-05,
    4.8750588425593555e-05,
    5.2833303190191453e-05,
    5.7201825526199317e-05,
    6.1872233965032413e-05,
    6.6861285491113317e-05,
    7.2186433864857225e-05,
    7.7865848081442015e-05,
    8.3918430956020861e-05,
    9.0363837825811226e-05,
    9.7222495359273456e-05,
    0.00010451562046237741,
    0.00011226523927174472,
    0.00012049420622424909,
    0.00012922622319245703,
    0.00013848585867509607,
    0.00014829856703154883,
    0.00015869070774918688,
    0.00016968956473218066,
    0.00018132336560024983,
    0.00019362130098565141,
    0.00020661354381654455,
    0.00022033126857471557,
    0.00023480667051550033,
    0.00025007298483760069,
    0.00026616450579035599,
    0.00028311660570590539,
    0.00030096575394355341,
    0.00031974953573354019,
    0.00033950667090730947,
    0.00036027703250126784,
    0.00038210166522093794,
    0.00040502280375232121,
    0.00042908389090721077,
    0.00045432959558912181,
    0.00048080583056644754,
    0.00050855977003939109,
    0.00053763986698717901,
    0.00056809587028201953,
    0.00059997884155624107,
    0.00063334117180901867,
    0.00066823659773908284,
    0.00070472021778979698,
    0.00074284850789298857,
    0.00078267933689792882,
    0.00082427198167187173,
    0.00086768714185858631,
    0.00091298695428135084,
    0.00096023500697691615,
    0.0010094963528469961,
    0.0010608375229138986,
    0.0011143265391669772,
    0.0011700329269866572,
    0.001228027727132867,
    0.0012883835072848063,
    0.0013511743731190662,
    0.0014164759789132347,
    0.0014843655376622306,
    0.0015549218306947286,
    0.0016282252167771746,
    0.0017043576406930261,
    0.0017834026412849953,
    0.0018654453589482378,
    0.0019505725425625776,
    0.0020388725558520442,
    0.0021304353831601637,
    0.0022253526346296417,
    0.0023237175507752595,
    0.002425625006439018,
    0.0025311715141167566,
    0.0026404552266457138,
    0.0027535759392426904,
    0.0028706350908827352,
    0.0029917357650084933,
    0.0031169826895606082,
    0.0032464822363198194,
    0.0033803424195516635,
    0.0035186728939449389,
    0.0036615849518353795,
    0.0038091915197062559,
    0.0039616071539579096,
    0.0041189480359385151,
    0.0042813319662286719,
    0.004448878358172711,
    0.0046217082306499466,
    0.0047999442000793754,
    0.0049837104716516844,
    0.0051731328297827429,
    0.0053683386277830654,
    0.0055694567767381135,
    0.0057766177335946012,
    0.0059899534884483539,
    0.0062095975510295864,
    0.0064356849363818623,
    0.0066683521497313252,
    0.0069077371705431696,
    0.0071539794357626902,
    0.0074072198222386232,
    0.0076676006283268629,
    0.007935265554673018,
    0.0082103596841726651,
    0.0084930294611085273,
    0.0087834226694642106,
    0.0090816884104145314,
    0.0093879770789928024,
    0.0097024403399359575,
    0.010025231102708658,
    0.010356503495708044,
    0.01069641283965111,
    0.01104511562014715,
    0.011402769459458091,
    0.011769533087449924,
    0.012145566311738926,
    0.012531029987036662,
    0.012926085983698271,
    0.013330897155478898,
    0.013745627306503524,
    0.014170441157455915,
    0.014605504310992785,
    0.015050983216389646,
    0.015507045133425306,
    0.015973858095512321,
    0.016451590872081107,
    0.016940412930225877,
    0.017440494395620947,
    0.017952006012716282,
    0.01847511910422172,
    0.019010005529889538,
    0.019556837644605491,
    0.020115788255798888,
    0.020687030580182598,
    0.021270738199834237,
    0.021867085017630308,
    0.022476245212045273,
    0.023098393191328028,
    0.023733703547068594,
    0.024382351007168176,
    0.025044510388226159,
    0.025720356547357926,
    0.026410064333457734,
    0.02711380853792128,
    0.027831763844842913,
    0.028564104780702725,
    0.029311005663559191,
    0.030072640551763327,
    0.030849183192210552,
    0.031640806968146945,
    0.032447684846546726,
    0.0332699893250782,
    0.034107892378675626,
    0.034961565405734904,
    0.035831179173950976,
    0.036716903765815517,
    0.037618908523793335,
    0.038537361995196501,
    0.039472431876775314,
    0.040424284959045431,
    0.041393087070370892,
    0.042379003020822814,
    0.04338219654583389,
    0.044402830249668956,
    0.045441065548732153,
    0.046497062614731376,
    0.047570980317720904,
    0.048662976169043332,
    0.049773206264191933,
    0.050901825225615031,
    0.052048986145483755,
    0.053214840528445093,
    0.054399538234381865,
    0.055603227421201842,
    0.056826054487677823,
    0.058068164016361112,
    0.059329698716590516,
    0.060610799367619365,
    0.061911604761882885,
    0.06323225164842862,
    0.064572874676532305,
    0.065933606339521986,
    0.067314576918832941,
    0.068715914428316163,
    0.070137744558823142,
    0.07158019062308954,
    0.073043373500940756,
    0.074527411584841705,
    0.076032420725813926,
    0.077558514179742286,
    0.079105802554094162,
    0.08067439375507357,
    0.082264392935232647,
    0.083875902441563052,
    0.085509021764089546,
    0.087163847484987941,
    0.088840473228249658,
    0.090538989609914827,
    0.092259484188895832,
    0.094002041418413118,
    0.095766742598064766,
    0.097553665826551311,
    0.099362885955077224,
    0.10119447454144993,
    0.10304849980489748,
    0.10492502658162541,
    0.10682411628113357,
    0.10874582684331296,
    0.11069021269634272,
    0.11265732471540718,
    0.11464721018225259,
    0.11665991274560256,
    0.11869547238245177,
    0.12075392536025634,
    0.12283530420003946,
    0.1249396376404307,
    0.12706695060265666,
    0.1292172641565007,
    0.1313905954872491,
    0.13358695786364033,
    0.13580636060683451,
    0.13804880906041889,
    0.14031430456146554,
    0.14260284441265678,
    0.14491442185549344,
    0.14724902604460102,
    0.14960664202314775,
    0.1519872506993892,
    0.15439082882435243,
    0.15681734897067351,
    0.1592667795126009,
    0.16173908460717715,
    0.16423422417661079,
    0.16675215389185064,
    0.16929282515737232,
    0.17185618509718925,
    0.17444217654209687,
    0.17705073801816099,
    0.17968180373645873,
    0.18233530358408154,
    0.18501116311640811,
    0.18770930355065546,
    0.19042964176071536,
    0.19317209027328319,
    0.19593655726528542,
    0.19872294656261191,
    0.20153115764015828,
    0.20436108562318331,
    0.20721262128998583,
    0.21008565107590504,
    0.21298005707864764,
    0.21589571706494456,
    0.21883250447853989,
    0.22179028844951354,
    0.22476893380493942,
    0.22776830108087956,
    0.23078824653571445,
    0.2338286221648099,
    0.23688927571651888,
    0.23997005070951771,
    0.24307078645147442,
    0.24619131805904698,
    0.24933147647920886,
    0.25249108851189789,
    0.25566997683398529,
    0.25886796002455947,
    0.26208485259152081,
    0.26532046499948053,
    0.26857460369895919,
    0.27184707115687645,
    0.27513766588832717,
    0.27844618248963399,
    0.28177241167267031,
    0.28511614030044363,
    0.28847715142393093,
    0.29185522432015609,
    0.29525013453149951,
    0.29866165390622873,
    0.3020895506402399,
    0.3055335893199973,
    0.30899353096665988,
    0.31246913308138136,
    0.31596014969177155,
    0.3194663313995047,
    0.32298742542906134,
    0.32652317567758876,
    0.33007332276586582,
    0.33363760409035553,
    0.33721575387633096,
    0.34080750323205727,
    0.34441258020401316,
    0.34803070983313528,
    0.3516616142120671,
    0.3553050125433953,
    0.35896062119885425,
    0.36262815377948038,
    0.36630732117669734,
    0.36999783163431216,
    0.37369939081140219,
    0.37741170184607326,
    0.38113446542006757,
    0.38486737982420094,
    0.38861014102460767,
    0.39236244272977111,
    0.39612397645831876,
    0.39989443160755844,
    0.40367349552273385,
    0.40746085356697553,
    0.4112561891919253,
    0.41505918400900949,
    0.41886951786133725,
    0.42268686889620105,
    0.42651091363815347,
    0.43034132706263684,
    0.43417778267014034,
    0.43801995256086002,
    0.44186750750983622,
    0.44572011704254294,
    0.44957744951090373,
    0.45343917216970842,
    0.45730495125340415,
    0.46117445205323565,
    0.46504733899470724,
    0.46892327571534115,
    0.47280192514270541,
    0.4766829495726842,
    0.48056601074796479,
    0.48445076993671327,
    0.48833688801141317,
    0.49222402552783928,
    0.49611184280413978,
    0.016572750131886842,
    0.017064371067705541,
    0.017567294040447375,
    0.018081689942953214,
    0.018607730236084828,
    0.01914558690733361,
    0.019695432428740375,
    0.020257439714136877,
    0.020831782075720031,
    0.02141863317997025,
    0.022018167002925651,
    0.022630557784824291,
    0.023255979984127012,
    0.023894608230933697,
    0.024546617279806286,
    0.025212181962012171,
    0.025891477137201913,
    0.026584677644535665,
    0.027291958253273001,
    0.028013493612841112,
    0.028749458202396844,
    0.0295000262798982,
    0.030265371830701375,
    0.031045668515699659,
    0.031841089619020893,
    0.032651807995300458,
    0.033477996016547031,
    0.034319825518618757,
    0.035177467747327607,
    0.036051093304190186,
    0.036940872091843283,
    0.03784697325914297,
    0.038769565145966113,
    0.039708815227733549,
    0.040664890059674351,
    0.041637955220850897,
    0.042628175257964633,
    0.043635713628962645,
    0.044660732646465487,
    0.045703393421036673,
    0.046763855804314754,
    0.047842278332028761,
    0.048938818166918231,
    0.050053631041579102,
    0.05118687120125686,
    0.05233869134660861,
    0.053509242576455777,
    0.054698674330549341,
    0.055907134332369574,
    0.057134768531982426,
    0.058381721048974745,
    0.059648134115490713,
    0.060934148019391836,
    0.062239901047562968,
    0.063565529429386952,
    0.064911167280410464,
    0.066276946546223708,
    0.067662996946576556,
    0.069069445919754002,
    0.070496418567233571,
    0.071944037598647259,
    0.073412423277071093,
    0.074901693364664609,
    0.076411963068683156,
    0.077943344987885743,
    0.079495949059360707,
    0.081069882505792154,
    0.082665249783189287,
    0.084282152529101304,
    0.085920689511339937,
    0.087580956577232028,
    0.089263046603424101,
    0.090967049446261133,
    0.092693051892761003,
    0.094441137612206821,
    0.09621138710837826,
    0.098003877672443562,
    0.099818683336533445,
    0.1016558748280178,
    0.10351551952450624,
    0.10539768140959309,
    0.10730242102936724,
    0.10922979544970715,
    0.11117985821438109,
    0.11315265930397223,
    0.11514824509564818,
    0.11716665832379437,
    0.11920793804153006,
    0.12127211958312578,
    0.12335923452734079,
    0.1254693106616985,
    0.12760237194771773,
    0.12975843848711746,
    0.13193752648901222,
    0.13413964823811481,
    0.13636481206396328,
    0.13861302231118797,
    0.14088427931083472,
    0.14317857935275957,
    0.14549591465911016,
    0.14783627335890837,
    0.15019963946374904,
    0.15258599284462776,
    0.15499530920991261,
    0.15742756008447162,
    0.15988271278996963,
    0.16236073042634658,
    0.16486157185448883,
    0.16738519168010546,
    0.16993154023882007,
    0.17250056358248897,
    0.17509220346675564,
    0.17770639733985136,
    0.18034307833265081,
    0.18300217524999191,
    0.18568361256326738,
    0.18838731040429668,
    0.19111318456048479,
    0.19386114647127511,
    0.19663110322590277,
    0.19942295756245379,
    0.20223660786823586,
    0.20507194818146512,
    0.20792886819427381,
    0.21080725325704161,
    0.21370698438405522,
    0.21662793826049784,
    0.21956998725077134,
    0.22253299940815277,
    0.22551683848578644,
    0.22852136394901201,
    0.23154643098902919,
    0.2345918905378982,
    0.23765758928487554,
    0.24074336969408328,
    0.24384907002351019,
    0.24697452434534212,
    0.25011956256761869,
    0.25328401045721249,
    0.25646768966412736,
    0.25967041774711047,
    0.26289200820057346,
    0.26613227048281701,
    0.2693910100455526,
    0.27266802836471482,
    0.27596312297255687,
    0.27927608749102206,
    0.28260671166638224,
    0.28595478140513525,
    0.28932007881115174,
    0.29270238222406164,
    0.29610146625886974,
    0.29951710184679065,
    0.30294905627729024,
    0.30639709324132364,
    0.30986097287575598,
    0.31334045180895426,
    0.31683528320753651,
    0.32034521682426503,
    0.32386999904706937,
    0.32740937294918393,
    0.33096307834038641,
    0.33453085181932035,
    0.33811242682688636,
    0.34170753370068563,
    0.34531589973049903,
    0.34893724921478381,
    0.35257130351817156,
    0.35621778112994801,
    0.35987639772349683,
    0.36354686621668869,
    0.36722889683319571,
    0.37092219716471247,
    0.37462647223406265,
    0.37834142455917136,
    0.38206675421788255,
    0.3858021589136002,
    0.38954733404173136,
    0.39330197275691048,
    0.39706576604098132,
    0.40083840277171534,
    0.40461956979224334,
    0.40840895198117699,
    0.41220623232339754,
    0.41601109198148745,
    0.41982321036778208,
    0.42364226521701581,
    0.42746793265953992,
    0.43129988729508606,
    0.43513780226705179,
    0.43898134933728195,
    0.44283019896132164,
    0.44668402036411475,
    0.45054248161612226,
    0.45440524970983498,
    0.45827199063665469,
    0.46214236946411674,
    0.46601605041342925,
    0.46989269693730079,
    0.47377197179803165,
    0.47765353714584086,
    0.48153705459740265,
    0.48542218531456566,
    0.4893085900832278,
    0.49319592939233947,
    0.49708386351300904,
    0.50097205257768229,
    0.50486015665937001,
    0.50874783585089489,
    0.51263475034413319,
    0.51652056050922068,
    0.52040492697369922,
    0.52428751070157398,
    0.52816797307225616,
    0.53204597595936332,
    0.53592118180935044,
    0.53979325371994524,
    0.54366185551836044,
    0.54752665183925686,
    0.55138730820242965,
    0.55524349109019322,
    0.55909486802443553,
    0.56294110764331917,
    0.56678187977759931,
    0.57061685552653596,
    0.57444570733337252,
    0.57826810906035631,
    0.58208373606327524,
    0.5858922652654851,
    0.58969337523140375,
    0.59348674623944586,
    0.59727206035437519,
    0.60104900149904872,
    0.60481725552553023,
    0.60857651028554827,
    0.61232645570027566,
    0.61606678382940705,
    0.6197971889395123,
    0.62351736757164222,
    0.62722701860816499,
    0.63092584333881097,
    0.63461354552590488,
    0.63828983146876261,
    0.6419544100672343,
    0.64560699288436985,
    0.6492472942081895,
    0.65287503111253797,
    0.65648992351700319,
    0.66009169424588066,
    0.66368006908616417,
    0.66725477684454604,
    0.67081554940340649,
    0.67436212177577759,
    0.67789423215926248,
    0.68141162198889427,
    0.68491403598891853,
    0.68840122222348332,
    0.69187293214622114,
    0.69532892064870921,
    0.69876894610779183,
    0.7021927704317531,
    0.70560015910532448,
    0.70899088123351606,
    0.71236470958425746,
    0.71572142062983757,
    0.71906079458713135,
    0.72238261545660187,
    0.72568667106006868,
    0.72897275307723064,
    0.732240657080935,
    0.73549018257118393,
    0.73872113300786857,
    0.74193331584222377,
    0.74512654254699662,
    0.74830062864531965,
    0.75145539373828496,
    0.75459066153121124,
    0.75770625985859974,
    0.76080202070777314,
    0.76387778024119468,
    0.76693337881746149,
    0.76996866101097083,
    0.7729834756302556,
    0.77597767573498744,
    0.77895111865164435,
    0.78190366598784378,
    0.78483518364533889,
    0.78774554183167866,
    0.7906346150705319,
    0.79350228221067665,
    0.79634842643365578,
    0.79917293526010058,
    0.80197570055472689,
    0.80475661853000324,
    0.80751558974849791,
    0.81025251912390739,
    0.81296731592077009,
    0.81565989375287262,
    0.81833017058035162,
    0.82097806870549883,
    0.82360351476727578,
    0.82620643973454455,
    0.82878677889802232,
    0.83134447186096816,
    0.83387946252861012,
    0.83639169909632116,
    0.83888113403655484,
    0.84134772408454861,
    0.84379143022280667,
    0.84621221766437227,
    0.84861005583490123,
    0.85098491835354784,
    0.85333678301267568,
    0.85566563175640509,
    0.85797145065801206,
    0.86025422989618894,
    0.86251396373018396,
    0.86475065047383115,
    0.8669642924684865,
    0.86915489605488538,
    0.87132247154393516,
    0.87346703318646035,
    0.87558859914191589,
    0.87768719144608354,
    0.87976283597777005,
    0.88181556242452308,
    0.883845404247382,
    0.88585239864468257,
    0.88783658651493158,
    0.88979801241877188,
    0.89173672454005393,
    0.89365277464603554,
    0.89554621804672652,
    0.89741711355339882,
    0.89926552343628163,
    0.90109151338146143,
    0.90289515244700702,
    0.90467651301833918,
    0.90643567076286735,
    0.90817270458391186,
    0.90988769657393431,
    0.91158073196709677,
    0.91325189909117077,
    0.91490128931881798,
    0.91652899701826407,
    0.91813511950338789,
    0.91971975698324693,
    0.92128301251106226,
    0.92282499193268464,
    0.92434580383456355,
    0.92584555949124236,
    0.92732437281240188,
    0.92878236028947359,
    0.93021964094184761,
    0.9316363362626956,
    0.93303257016443197,
    0.93440846892383733,
    0.93576416112686511,
    0.9370997776131551,
    0.9384154514202766,
    0.93971131772772332,
    0.94098751380068335,
    0.94224417893360657,
    0.94348145439359221,
    0.94469948336361909,
    0.94589841088564086,
    0.94707838380356912,
    0.94823955070616672,
    0.9493820618698724,
    0.95050606920158098,
    0.95161172618139989,
    0.95269918780540286,
    0.95376861052840578,
    0.95482015220678318,
    0.95585397204134825,
    0.95687023052031839,
    0.95786908936238591,
    0.95885071145991663,
    0.95981526082229551,
    0.96076290251944185,
    0.9616938026255123,
    0.96260812816281427,
    0.96350604704594778,
    0.96438772802619699,
    0.96525334063618995,
    0.96610305513484673,
    0.96693704245263412,
    0.96775547413714658,
    0.96855852229903172,
    0.96934635955827819,
    0.97011915899088419,
    0.97087709407592515,
    0.97162033864303632,
    0.97234906682032929,
    0.97306345298275754,
    0.97376367170094891,
    0.97444989769052015,
    0.97512230576189041,
    0.97578107077060827,
    0.97642636756820755,
    0.97705837095360826,
    0.97767725562507513,
    0.97828319613274872,
    0.97887636683176382,
    0.97945694183596743,
    0.9800250949722491,
    0.98058099973549839,
    0.98112482924419919,
    0.98165675619667436,
    0.98217695282799278,
    0.98268559086754825,
    0.98318284149732249,
    0.98366887531084235,
    0.98414386227284067,
    0.98460797167963165,
    0.98506137212020817,
    0.98550423143807164,
    0.98593671669380256,
    0.98635899412837769,
    0.98677122912724513,
    0.98717358618516127,
    0.98756622887179801,
    0.98794931979812639,
    0.98832302058358257,
    0.98868749182402205,
    0.98904289306046611,
    0.98938938274864718,
    0.98972711822935566,
    0.99005625569959366,
    0.99037695018453775,
    0.99068935551031501,
    0.99099362427759452,
    0.99128990783599669,
    0.99157835625932256,
    0.99185911832160467,
    0.99213234147397933,
    0.99239817182238343,
    0.9926567541060739,
    0.9929082316769704,
    0.99315274647982155,
    0.99339043903319291,
    0.99362144841127553,
    0.99384591222651419,
    0.99406396661305219,
    0.99427574621099157,
    0.99448138415146503,
    0.9946810120425168,
    0.99487475995578867,
    0.9950627564140081,
    0.9952451283792737,
    0.99542200124213276,
    0.99559349881144743,
    0.99575974330504358,
    0.99592085534113617,
    0.99607695393052664,
    0.99622815646956431,
    0.99637457873386703,
    0.99651633487279256,
    0.99665353740465534,
    0.99678629721267908,
    0.99691472354167898,
    0.99703892399546479,
    0.99715900453495698,
    0.99727506947700628,
    0.9973872214939089,
    0.99749556161360764,
    0.99760018922056903,
    0.9977012020573286,
    0.99779869622669137,
    0.997892766194581,
    0.99798350479352416,
    0.99807100322676123,
    0.99815535107297182,
    0.99823663629160431,
    0.99831494522879871,
    0.99839036262388992,
    0.99846297161648179,
    0.99853285375407852,
    0.99860008900026176,
    0.9986647557434033,
    0.99872693080589736,
    0.9987866894539037,
    0.9988441054075865,
    0.9988992508518385,
    0.99895219644747513,
    0.99900301134288783,
    0.99905176318614275,
    0.99909851813751149,
    0.99914334088242118,
    0.99918629464481001,
    0.99922744120087625,
    0.99926684089320505,
    0.99930455264526197,
    0.99934063397623818,
    0.99937514101623448,
    0.99940812852176997,
    0.99943964989160283,
    0.99946975718284847,
    0.99949850112738181,
    0.99952593114851063,
    0.99955209537790568,
    0.9995770406727742,
    0.99960081263326295,
    0.9996234556200787,
    0.9996450127723101,
    0.99966552602544034,
    0.99968503612953474,
    0.99970358266759185,
    0.9997212040740433,
    0.99973793765339125,
    0.99975381959896725,
    0.99976888501180317,
    0.99978316791959831,
    0.99979670129577147,
    0.99980951707858534,
    0.99982164619032965,
    0.99983311855655199,
    0.99984396312532309,
    0.99985420788652424,
    0.99986387989114633,
    0.9998730052705872,
    0.99988160925593572,
    0.99988971619723244,
    0.99989734958269383,
    0.99990453205789009,
    0.99991128544486529,
    0.99991763076118867,
    0.99992358823892691,
    0.99992917734352749,
    0.99993441679260153,
    0.999939324574598,
    0.99994391796735793,
    0.99994821355654029,
    0.99995222725390931,
    0.99995597431547534,
    0.9999594693594791,
    0.99996272638421158,
    0.9999657587856613,
    0.9999685793749804,
    0.99997120039576204,
    0.99997363354112079,
    0.99997588997057041,
    0.9999779803266895,
    0.999979914751571,
    0.99998170290304589,
    0.99998335397067761,
    0.99998487669151948,
    0.99998627936563023,
    0.99998756987134152,
    0.99998875568027279,
    0.99998984387208811,
    0.99999084114899106,
    0.9999917538499522,
    0.99999258796466639,
    0.99999334914723492,
    0.99999404272957004,
    0.99999467373451723,
    0.99999524688869446,
    0.99999576663504308,
    0.99999623714509089,
    0.99999666233092221,
    0.99999704585685589,
    0.9999973911508282,
    0.99999770141547972,
    0.9999979796389451,
    0.9999982286053456,
    0.99999845090498318,
    0.99999864894423685,
    0.99999882495516068,
    0.99999898100478457,
    0.99999911900411764,
    0.99999924071685631,
    0.99999934776779753,
    0.99999944165095833,
    0.99999952373740486,
    0.99999959528279148,
    0.99999965743461261,
    0.99999971123917075,
    0.99999975764826188,
    0.99999979752558243,
    0.99999983165286122,
    0.99999986073571878,
    0.99999988540925877,
    0.99999990624339508,
    0.99999992374791902,
    0.99999993837731083,
    0.99999995053530044,
    0.9999999605791815,
    0.99999996882388542,
    0.99999997554581899,
    0.99999998098647158,
    0.9999999853557987,
    0.99999998883538554,
    0.99999999158139907,
    0.99999999372733284,
    0.99999999538655204,
    0.99999999665464523,
    0.99999999761158875,
    0.99999999832373188,
    0.99999999884560875,
    0.99999999922158467,
    0.99999999948734386,
    0.99999999967122666,
    0.99999999979542298,
    0.99999999987702992,
    0.99999999992898236,
    0.9999999999608622,
    0.99999999997959665,
    0.99999999999005229,
    0.99999999999553324,
    0.99999999999819178,
    0.99999999999936007,
    0.99999999999981115,
    0.99999999999995692,
    0.99999999999999345,
    0.99999999999999944,
    1,
    1,
    2.2939207142021998e-19,
    8.6378089713222088e-17,
    2.0438262008197158e-15,
    1.7918513846202511e-14,
    9.3733868867020101e-14,
    3.5699695125733563e-13,
    1.0966157497684462e-12,
    2.8838231061971761e-12,
    6.7426692558912817e-12,
    1.4378604479583848e-11,
    2.8473625886593605e-11,
    5.3056407921765022e-11,
    9.395577484159768e-11,
    1.5934580517639727e-10,
    2.6039078312676167e-10,
    4.1199812998028195e-10,
    6.3368736002815251e-10,
    9.505830101657488e-10,
    1.3945393904345676e-09,
    2.0054048942677319e-09,
    2.8324334922040856e-09,
    3.9358509114073013e-09,
    5.3885828755409225e-09,
    7.2781526454863142e-09,
    9.7087549611304297e-09,
    1.2803513338080325e-08,
    1.6706927520771386e-08,
    2.1587517735120029e-08,
    2.7640672219726551e-08,
    3.5091704344760904e-08,
    4.4199125452168019e-08,
    5.5258139369817985e-08,
    6.8604364365811752e-08,
    8.4617788117449452e-08,
    1.0372696107249501e-07,
    1.2641343337844833e-07,
    1.5321644034869011e-07,
    1.8473784122272215e-07,
    2.2164731576141867e-07,
    2.6468782299736564e-07,
    3.146813262351319e-07,
    3.7253478816682993e-07,
    4.3924643973472153e-07,
    5.1591232613505997e-07,
    6.0373313311597178e-07,
    7.0402129647712573e-07,
    8.1820839743036593e-07,
    9.4785284622854856e-07,
    1.0946478562146873e-06,
    1.2604297101853247e-06,
    1.447186320700981e-06,
    1.6570660857127451e-06,
    1.8923870406077229e-06,
    2.1556463075073398e-06,
    2.4495298423815261e-06,
    2.776922480268832e-06,
    3.1409182786176326e-06,
    3.5448311584870042e-06,
    3.9922058430677497e-06,
    4.4868290927045866e-06,
    5.0327412353198692e-06,
    5.634247990857587e-06,
    6.2959325880839251e-06,
    7.0226681717975725e-06,
    7.8196304982194187e-06,
    8.6923109160474406e-06,
    9.6465296303786576e-06,
    1.0688449246416182e-05,
    1.1824588589595846e-05,
    1.3061836798483708e-05,
    1.4407467686513319e-05,
    1.5869154368349873e-05,
    1.7454984146387782e-05,
    1.9173473652608625e-05,
    2.103358424074834e-05,
    2.3044737623445951e-05,
    2.5216831748771173e-05,
    2.7560256910255358e-05,
    3.008591208427935e-05,
    3.2805221488403154e-05,
    3.5730151353956195e-05,
    3.8873226905943388e-05,
    4.2247549543061458e-05,
    4.5866814210362114e-05,
    4.974532695684418e-05,
    5.389802267000516e-05,
    5.8340482979135242e-05,
    6.3088954318892156e-05,
    6.8160366144455153e-05,
    7.3572349289319873e-05,
    7.9343254456563624e-05,
    8.5492170834182711e-05,
    9.2038944824879754e-05,
    9.9004198880460294e-05,
    0.00010640935043078379,
    0.00011427663089700501,
    0.0001226291047786375,
    0.00013149068880377246,
    0.00014088617113159195,
    0.00015084123059612889,
    0.00016138245598004185,
    0.00017253736530699833,
    0.00018433442514108792,
    0.0001968030698815228,
    0.00020997372104072492,
    0.00022387780649374612,
    0.00023854777968682311,
    0.00025401713879272866,
    0.00027032044580044875,
    0.00028749334552658943,
    0.00030557258453579801,
    0.00032459602995737178,
    0.00034460268818512142,
    0.00036563272344746075,
    0.00038772747623460084,
    0.00041092948156964693,
    0.00043528248711031752,
    0.00046083147106793936,
    0.00048762265993031,
    0.00051570354597496712,
    0.0005451229045593585,
    0.00057593081117436802,
    0.00060817865824762645,
    0.00064191917168300919,
    0.00067720642712271358,
    0.00071409586591830146,
    0.00075264431079709207,
    0.00079290998121030469,
    0.00083495250834936697,
    0.00087883294981682953,
    0.00092461380393836588,
    0.00097235902370237525,
    0.0010221340303137609,
    0.0010740057263485129,
    0.0011280425084957912,
    0.0011843142798742842,
    0.0012428924619096978,
    0.0013038500057603218,
    0.0013672614032777255,
    0.0014332026974897365,
    0.0015017514925929755,
    0.0015729869634423476,
    0.0016469898645250166,
    0.0017238425384065359,
    0.0018036289236369528,
    0.0018864345621048652,
    0.0019723466058275681,
    0.0020614538231656042,
    0.002153846604450213,
    0.002249616967012355,
    0.0023488585596021911,
    0.0024516666661880955,
    0.0025581382091244918,
    0.0026683717516780203,
    0.0027824674999017722,
    0.002900527303847553,
    0.0030226546581063863,
    0.003148954701667706,
    0.0032795342170879494,
    0.0034145016289595142,
    0.0035539670016713188,
    0.0036980420364524733,
    0.003846840067690851,
    0.0040004760585186407,
    0.0041590665956572487,
    0.0043227298835142236,
    0.0044915857375251731,
    0.0046657555767339711,
    0.0048453624156048498,
    0.0050305308550603134,
    0.0052213870727391195,
    0.0054180588124689271,
    0.005620675372948537,
    0.0058293675956349869,
    0.0060442678518311385,
    0.0062655100289697174,
    0.0064932295160901425,
    0.0067275631885048339,
    0.0069686493916520679,
    0.0072166279241327987,
    0.0074716400199292578,
    0.0077338283298035082,
    0.0080033369018745158,
    0.0082803111613726985,
    0.0085648978895712519,
    0.008857245201894029,
    0.0091575025252000446,
    0.0094658205742451321,
    0.0097823513273216801,
    0.010107248001077745,
    0.010440665024517237,
    0.010782758012183338,
    0.011133683736527622,
    0.011493600099467855,
    0.011862666103137765,
    0.01224104181983255,
    0.01262888836115429,
    0.013026367846361782,
    0.013433643369929806,
    0.013850878968323203,
    0.014278239585991526,
    0.014715891040590495,
    0.015163999987436861,
    0.01562273388320368,
    0.016092260948863436,
    0.98342724986811314,
    0.9839077390511366,
    0.98437726611679632,
    0.98483600001256311,
    0.98528410895940954,
    0.9857217604140085,
    0.98614912103167685,
    0.9865663566300702,
    0.9869736321536382,
    0.98737111163884572,
    0.98775895818016746,
    0.9881373338968622,
    0.98850639990053213,
    0.98886631626347232,
    0.98921724198781669,
    0.98955933497548276,
    0.98989275199892224,
    0.99021764867267836,
    0.99053417942575483,
    0.99084249747479991,
    0.99114275479810598,
    0.99143510211042873,
    0.99171968883862727,
    0.99199666309812551,
    0.99226617167019648,
    0.99252835998007072,
    0.9927833720758672,
    0.99303135060834791,
    0.99327243681149513,
    0.9935067704839099,
    0.99373448997103031,
    0.9939557321481689,
    0.99417063240436498,
    0.99437932462705148,
    0.99458194118753107,
    0.99477861292726089,
    0.99496946914493967,
    0.99515463758439515,
    0.99533424442326601,
    0.99550841426247483,
    0.99567727011648577,
    0.99584093340434277,
    0.99599952394148139,
    0.99615315993230913,
    0.99630195796354748,
    0.99644603299832868,
    0.99658549837104049,
    0.99672046578291207,
    0.99685104529833224,
    0.99697734534189364,
    0.9970994726961524,
    0.99721753250009826,
    0.997331628248322,
    0.99744186179087546,
    0.99754833333381188,
    0.99765114144039779,
    0.99775038303298769,
    0.99784615339554983,
    0.99793854617683442,
    0.99802765339417243,
    0.99811356543789509,
    0.99819637107636305,
    0.99827615746159348,
    0.99835301013547495,
    0.99842701303655768,
    0.99849824850740698,
    0.9985667973025103,
    0.9986327385967223,
    0.99869614999423972,
    0.99875710753809033,
    0.99881568572012569,
    0.99887195749150426,
    0.99892599427365147,
    0.99897786596968619,
    0.99902764097629759,
    0.99907538619606162,
    0.99912116705018317,
    0.99916504749165058,
    0.99920709001878971,
    0.99924735568920287,
    0.99928590413408169,
    0.99932279357287723,
    0.99935808082831701,
    0.99939182134175242,
    0.99942406918882565,
    0.99945487709544067,
    0.99948429645402503,
    0.9995123773400697,
    0.9995391685289321,
    0.99956471751288967,
    0.99958907051843038,
    0.99961227252376539,
    0.99963436727655253,
    0.99965539731181485,
    0.99967540397004262,
    0.99969442741546422,
    0.99971250665447342,
    0.99972967955419956,
    0.99974598286120731,
    0.99976145222031321,
    0.9997761221935062,
    0.9997900262789593,
    0.99980319693011843,
    0.9998156655748589,
    0.99982746263469302,
    0.99983861754401993,
    0.99984915876940383,
    0.99985911382886838,
    0.9998685093111962,
    0.99987737089522133,
    0.99988572336910297,
    0.99989359064956917,
    0.99990099580111957,
    0.9999079610551751,
    0.9999145078291658,
    0.99992065674554342,
    0.99992642765071071,
    0.99993183963385557,
    0.99993691104568105,
    0.99994165951702085,
    0.99994610197733003,
    0.9999502546730431,
    0.99995413318578963,
    0.99995775245045693,
    0.99996112677309401,
    0.99996426984864606,
    0.99996719477851159,
    0.99996991408791569,
    0.99997243974308969,
    0.99997478316825128,
    0.99997695526237651,
    0.99997896641575923,
    0.99998082652634734,
    0.9999825450158536,
    0.99998413084563165,
    0.99998559253231345,
    0.9999869381632015,
    0.99998817541141038,
    0.99998931155075355,
    0.99999035347036958,
    0.99999130768908395,
    0.99999218036950177,
    0.99999297733182824,
    0.99999370406741195,
    0.99999436575200917,
    0.99999496725876469,
    0.99999551317090729,
    0.99999600779415698,
    0.9999964551688415,
    0.99999685908172142,
    0.9999972230775197,
    0.99999755047015759,
    0.99999784435369254,
    0.99999810761295937,
    0.99999834293391432,
    0.99999855281367933,
    0.99999873957028984,
    0.99999890535214375,
    0.99999905214715379,
    0.99999918179160252,
    0.99999929597870352,
    0.99999939626686685,
    0.99999948408767392,
    0.99999956075356022,
    0.99999962746521187,
    0.99999968531867378,
    0.99999973531217701,
    0.99999977835268428,
    0.99999981526215875,
    0.99999984678355969,
    0.99999987358656661,
    0.99999989627303898,
    0.99999991538221189,
    0.99999993139563559,
    0.9999999447418606,
    0.99999995580087453,
    0.99999996490829568,
    0.9999999723593278,
    0.99999997841248223,
    0.99999998329307249,
    0.99999998719648664,
    0.99999999029124509,
    0.99999999272184736,
    0.99999999461141709,
    0.99999999606414913,
    0.99999999716756649,
    0.99999999799459516,
    0.99999999860546063,
    0.99999999904941694,
    0.99999999936631268,
    0.9999999995880019,
    0.99999999973960918,
    0.99999999984065424,
    0.99999999990604427,
    0.99999999994694355,
    0.99999999997152633,
    0.99999999998562139,
    0.99999999999325728,
    0.9999999999971162,
    0.99999999999890343,
    0.99999999999964295,
    0.9999999999999063,
    0.99999999999998213,
    0.999999999999998,
    0.99999999999999989,
    1,
    1.0488951591505954e-22,
    8.1943541262962082e-18,
    5.0165487525131381e-16,
    6.5808784042419197e-15,
    4.3031445487046332e-14,
    1.888564358677416e-13,
    6.3991189591077653e-13,
    1.8082549430697723e-12,
    4.4667841166126366e-12,
    9.9477145907819227e-12,
    2.0403386878955009e-11,
    3.9137856702099753e-11,
    7.1017655803096113e-11,
    1.2297004869576301e-10,
    2.0457703868337489e-10,
    3.2877329800996646e-10,
    5.1265611176688437e-10,
    7.7841533321908799e-10,
    1.1543912495969941e-09,
    1.6762681521806647e-09,
    2.3884112927483839e-09,
    3.3453547902397102e-09,
    4.6134479268664782e-09,
    6.2726671419704494e-09,
    8.4186008947537601e-09,
    1.1164614423682312e-08,
    1.4644201280970755e-08,
    1.9013528365192732e-08,
    2.4454181013816967e-08,
    3.1176114550447925e-08,
    3.9420818508852053e-08,
    4.9464699577582939e-08,
    6.1622689125291822e-08,
    7.6252080977736059e-08,
    9.3756604923194619e-08,
    1.1459074122358652e-07,
    1.3926428120418802e-07,
    1.6834713878558369e-07,
    2.0247441760749565e-07,
    2.4235173817554477e-07,
    2.887608292389456e-07,
    3.4256538737975772e-07,
    4.0471720856275601e-07,
    4.7626259515938004e-07,
    5.58349041719727e-07,
    6.5223220252331358e-07,
    7.5928314369249965e-07,
    8.8099588240219009e-07,
    1.0189952154658806e-06,
    1.1750448393214271e-06,
    1.3510557631802745e-06,
    1.5490950168414316e-06,
    1.7713946544063933e-06,
    2.0203610548636514e-06,
    2.2985845202415756e-06,
    2.6088491717564454e-06,
    2.9541431441084643e-06,
    3.3376690778028252e-06,
    3.7628549090955388e-06,
    4.2333649568849279e-06,
    4.7531113055896147e-06,
    5.3262654827726788e-06,
    5.9572704299895792e-06,
    6.6508527650546529e-06,
    7.4120353336376486e-06,
    8.2461500478180395e-06,
    9.1588510089409518e-06,
    1.0156127911834645e-05,
    1.1244319727165756e-05,
    1.2430128658425115e-05,
    1.3720634369754154e-05,
    1.5123308480539789e-05,
    1.6646029322424469e-05,
    1.829709695409798e-05,
    2.0085248428958745e-05,
    2.2019673310454948e-05,
    2.411002942964016e-05,
    2.6366458879204015e-05,
    2.879960423796679e-05,
    3.1420625019556714e-05,
    3.4241214338721643e-05,
    3.7273615788461686e-05,
    4.0530640520907276e-05,
    4.4025684524607813e-05,
    4.7772746090639881e-05,
    5.1786443459690818e-05,
    5.6082032642024028e-05,
    6.0675425401986184e-05,
    6.5583207398474221e-05,
    7.0822656472541699e-05,
    7.641176107308953e-05,
    8.2369238811356144e-05,
    8.8714555134696287e-05,
    9.5467942109916704e-05,
    0.00010265041730621988,
    0.00011028380276759619,
    0.00011839074406429733,
    0.00012699472941282312,
    0.00013612010885365704,
    0.00014579211347579533,
    0.00015603687467692946,
    0.00016688144344796185,
    0.00017835380967036189,
    0.00019048292141470044,
    0.00020329870422854113,
    0.00021683208040170984,
    0.00023111498819681597,
    0.00024618040103275619,
    0.00026206234660879551,
    0.00027879592595669124,
    0.00029641733240820393,
    0.00031496387046522196,
    0.00033447397455962084,
    0.00035498722768987403,
    0.00037654437992134041,
    0.0003991873667370651,
    0.00042295932722585171,
    0.00044790462209429193,
    0.00047406885148937355,
    0.0005014988726182323,
    0.00053024281715156191,
    0.00056035010839715772,
    0.0005918714782300325,
    0.00062485898376552179,
    0.00065936602376177314,
    0.00069544735473800717,
    0.00073315910679493741,
    0.00077255879912373681,
    0.00081370535518996009,
    0.00085665911757884777,
    0.00090148186248847114,
    0.0009482368138572163,
    0.00099698865711214928,
    0.0010478035525248643,
    0.0011007491481614745,
    0.0011558945924134817,
    0.0012133105460963374,
    0.0012730691941025955,
    0.0013352442565966562,
    0.0013999109997382018,
    0.0014671462459215356,
    0.0015370283835181636,
    0.0016096373761100762,
    0.0016850547712013338,
    0.0017633637083956963,
    0.0018446489270281981,
    0.0019289967732387199,
    0.0020164952064757884,
    0.0021072338054190019,
    0.0022013037733086705,
    0.0022987979426714457,
    0.0023998107794309183,
    0.0025044383863923689,
    0.0026127785060910663,
    0.0027249305229937381,
    0.0028409954650430577,
    0.0029610760045352376,
    0.0030852764583210557,
    0.0032137027873208918,
    0.0033464625953446162,
    0.0034836651272074277,
    0.0036254212661330104,
    0.0037718435304356659,
    0.0039230460694733467,
    0.0040791446588638184,
    0.0042402566949564695,
    0.0044065011885525921,
    0.0045779987578672618,
    0.0047548716207262672,
    0.0049372435859918531,
    0.0051252400442113634,
    0.0053189879574832106,
    0.0055186158485349352,
    0.0057242537890084388,
    0.0059360333869478551,
    0.0061540877734858413,
    0.0063785515887244508,
    0.0066095609668070905,
    0.0068472535201784474,
    0.0070917683230296274,
    0.0073432458939261128,
    0.0076018281776165427,
    0.0078676585260206884,
    0.0081408816783953616,
    0.0084216437406774147,
    0.0087100921640033621,
    0.0090063757224055126,
    0.0093106444896849817,
    0.0096230498154622392,
    0.0099437443004063448,
    0.01027288177064436,
    0.010610617251352863,
    0.01095710693953387,
    0.011312508175977915,
    0.011676979416417379,
    0.012050680201873646,
    0.012433771128202013,
    0.012826413814838721,
    0.013228770872754857,
    0.013641005871622328,
    0.014063283306197468,
    0.014495768561928307,
    0.014938627879791869,
    0.015392028320368333,
    0.015856137727159287,
    0.016331124689157685,
    0.016817158502677537,
    0.017314409132451799,
    0.017823047172007257,
    0.018343243803325682,
    0.01887517075580086,
    0.019419000264501569,
    0.019974905027750876,
    0.020543058164032599,
    0.021123633168236153,
    0.021716803867251309,
    0.022322744374924885,
    0.022941629046391698,
    0.023573632431792467,
    0.024218929229391777,
    0.024877694238109555,
    0.025550102309479863,
    0.026236328299051141,
    0.026936547017242476,
    0.027650933179670732,
    0.028379661356963699,
    0.029122905924074901,
    0.029880841009115824,
    0.030653640441721832,
    0.031441477700968244,
    0.032244525862853403,
    0.033062957547365886,
    0.033896944865153246,
    0.034746659363810009,
    0.035612271973803011,
    0.036493952954052229,
    0.037391871837185744,
    0.038306197374487684,
    0.039237097480558147,
    0.040184739177704466,
    0.041149288540083422,
    0.042130910637614115,
    0.04312976947968164,
    0.044146027958651732,
    0.0451798477932168,
    0.046231389471594171,
    0.047300812194597142,
    0.048388273818600142,
    0.049493930798418968,
    0.050617938130127639,
    0.051760449293833291,
    0.052921616196430828,
    0.054101589114359167,
    0.055300516636380935,
    0.056518545606407802,
    0.057755821066393445,
    0.059012486199316649,
    0.060288682272276656,
    0.061584548579723362,
    0.062900222386844895,
    0.064235838873134929,
    0.06559153107616271,
    0.066967429835568054,
    0.068363663737304417,
    0.069780359058152344,
    0.071217639710526442,
    0.072675627187598166,
    0.074154440508757594,
    0.075654196165436449,
    0.077175008067315337,
    0.078716987488937712,
    0.080280243016753072,
    0.081864880496612086,
    0.08347100298173589,
    0.085098710681182022,
    0.086748100908829215,
    0.088419268032903203,
    0.090112303426065687,
    0.091827295416088153,
    0.093564329237132635,
    0.095323486981660807,
    0.097104847552993007,
    0.098908486618538557,
    0.10073447656371841,
    0.1025828864466012,
    0.10445378195327346,
    0.10634722535396443,
    0.10826327545994607,
    0.11020198758122815,
    0.11216341348506839,
    0.11414760135531746,
    0.11615459575261801,
    0.11818443757547693,
    0.12023716402222989,
    0.12231280855391642,
    0.12441140085808407,
    0.12653296681353962,
    0.12867752845606487,
    0.13084510394511459,
    0.13303570753151345,
    0.13524934952616888,
    0.13748603626981606,
    0.13974577010381109,
    0.14202854934198797,
    0.14433436824359488,
    0.14666321698732435,
    0.14901508164645211,
    0.15138994416509877,
    0.15378778233562776,
    0.15620856977719338,
    0.15865227591545142,
    0.16111886596344516,
    0.16360830090367884,
    0.16612053747138988,
    0.16865552813903179,
    0.17121322110197768,
    0.17379356026545548,
    0.17639648523272422,
    0.17902193129450122,
    0.1816698294196484,
    0.18434010624712735,
    0.18703268407922988,
    0.18974748087609261,
    0.19248441025150204,
    0.19524338146999676,
    0.19802429944527308,
    0.20082706473989939,
    0.20365157356634428,
    0.20649771778932333,
    0.20936538492946813,
    0.21225445816832139,
    0.21516481635466111,
    0.21809633401215622,
    0.22104888134835562,
    0.22402232426501253,
    0.22701652436974437,
    0.23003133898902922,
    0.23306662118253851,
    0.23612221975880529,
    0.2391979792922268,
    0.24229374014140029,
    0.24540933846878871,
    0.24854460626171501,
    0.25169937135468029,
    0.25487345745300333,
    0.25806668415777617,
    0.26127886699213143,
    0.26450981742881602,
    0.267759342919065,
    0.27102724692276936,
    0.27431332893993127,
    0.27761738454339813,
    0.28093920541286871,
    0.28427857937016243,
    0.28763529041574254,
    0.29100911876648389,
    0.29439984089467547,
    0.2978072295682469,
    0.30123105389220811,
    0.30467107935129079,
    0.3081270678537788,
    0.31159877777651668,
    0.31508596401108147,
    0.31858837801110579,
    0.32210576784073752,
    0.32563787822422235,
    0.32918445059659351,
    0.33274522315545396,
    0.33631993091383583,
    0.3399083057541194,
    0.34351007648299681,
    0.34712496888746203,
    0.3507527057918105,
    0.35439300711563015,
    0.3580455899327657,
    0.36171016853123739,
    0.36538645447409518,
    0.36907415666118898,
    0.37277298139183501,
    0.37648263242835778,
    0.3802028110604877,
    0.38393321617059295,
    0.38767354429972434,
    0.39142348971445168,
    0.39518274447446977,
    0.39895099850095128,
    0.40272793964562481,
    0.40651325376055408,
    0.41030662476859625,
    0.4141077347345149,
    0.41791626393672476,
    0.42173189093964369,
    0.42555429266662748,
    0.42938314447346404,
    0.43321812022240069,
    0.43705889235668083,
    0.44090513197556441,
    0.44475650890980684,
    0.44861269179757035,
    0.45247334816074319,
    0.45633814448163956,
    0.46020674628005481,
    0.46407881819064961,
    0.46795402404063668,
    0.47183202692774384,
    0.47571248929842602,
    0.47959507302630078,
    0.48347943949077932,
    0.48736524965586681,
    0.49125216414910505,
    0.49513984334062999,
    0.49902794742231765,
    0.50291613648699096,
    0.50680407060766053,
    0.5106914099167722,
    0.51457781468543429,
    0.51846294540259741,
    0.5223464628541592,
    0.52622802820196835,
    0.53010730306269926,
    0.53398394958657081,
    0.5378576305358832,
    0.54172800936334531,
    0.54559475029016502,
    0.5494575183838778,
    0.55331597963588519,
    0.5571698010386783,
    0.56101865066271805,
    0.56486219773294821,
    0.56870011270491394,
    0.57253206734046014,
    0.57635773478298424,
    0.58017678963221797,
    0.58398890801851255,
    0.58779376767660252,
    0.59159104801882301,
    0.59538043020775666,
    0.59916159722828466,
    0.60293423395901868,
    0.60669802724308952,
    0.61045266595826864,
    0.61419784108639985,
    0.6179332457821175,
    0.6216585754408287,
    0.62537352776593735,
    0.62907780283528747,
    0.63277110316680429,
    0.63645313378331136,
    0.64012360227650311,
    0.64378221887005194,
    0.64742869648182844,
    0.65106275078521614,
    0.65468410026950097,
    0.65829246629931437,
    0.66188757317311364,
    0.6654691481806797,
    0.66903692165961359,
    0.67259062705081607,
    0.67613000095293063,
    0.67965478317573491,
    0.68316471679246349,
    0.6866595481910458,
    0.69013902712424402,
    0.69360290675867631,
    0.6970509437227097,
    0.70048289815320941,
    0.70389853374113021,
    0.70729761777593836,
    0.71067992118884826,
    0.71404521859486481,
    0.71739328833361782,
    0.72072391250897794,
    0.72403687702744313,
    0.72733197163528518,
    0.73060898995444745,
    0.73386772951718304,
    0.73710799179942654,
    0.74032958225288958,
    0.74353231033587264,
    0.74671598954278751,
    0.74988043743238131,
    0.75302547565465783,
    0.75615092997648981,
    0.75925663030591672,
    0.76234241071512443,
    0.7654081094621018,
    0.76845356901097084,
    0.77147863605098799,
    0.77448316151421359,
    0.77746700059184726,
    0.78043001274922863,
    0.78337206173950213,
    0.78629301561594478,
    0.78919274674295836,
    0.79207113180572619,
    0.79492805181853488,
    0.79776339213176417,
    0.80057704243754624,
    0.80336889677409729,
    0.80613885352872494,
    0.80888681543951524,
    0.81161268959570332,
    0.81431638743673262,
    0.81699782475000815,
    0.81965692166734916,
    0.82229360266014861,
    0.82490779653324431,
    0.827499436417511,
    0.83006845976117993,
    0.83261480831989454,
    0.8351384281455112,
    0.83763926957365342,
    0.84011728721003032,
    0.84257243991552833,
    0.84500469079008733,
    0.84741400715537218,
    0.84980036053625096,
    0.85216372664109163,
    0.85450408534088984,
    0.85682142064724043,
    0.85911572068916531,
    0.86138697768881201,
    0.86363518793603677,
    0.86586035176188514,
    0.86806247351098775,
    0.87024156151288257,
    0.87239762805228227,
    0.87453068933830147,
    0.87664076547265923,
    0.87872788041687422,
    0.88079206195846993,
    0.88283334167620564,
    0.88485175490435186,
    0.88684734069602777,
    0.88882014178561886,
    0.89077020455029288,
    0.89269757897063273,
    0.89460231859040695,
    0.89648448047549378,
    0.89834412517198226,
    0.90018131666346657,
    0.90199612232755644,
    0.90378861289162171,
    0.90555886238779315,
    0.907306948107239,
    0.90903295055373889,
    0.9107369533965759,
    0.91241904342276803,
    0.91407931048866009,
    0.91571784747089868,
    0.91733475021681077,
    0.91893011749420783,
    0.92050405094063925,
    0.92205665501211431,
    0.92358803693131686,
    0.92509830663533543,
    0.92658757672292891,
    0.9280559624013528,
    0.92950358143276646,
    0.93093055408024594,
    0.93233700305342349,
    0.93372305345377626,
    0.93508883271958954,
    0.93643447057061302,
    0.93776009895243706,
    0.93906585198060821,
    0.94035186588450925,
    0.94161827895102523,
    0.94286523146801759,
    0.9440928656676304,
    0.94530132566945069,
    0.94649075742354427,
    0.94766130865339138,
    0.94881312879874313,
    0.94994636895842088,
    0.95106118183308175,
    0.95215772166797119,
    0.95323614419568525,
    0.95429660657896331,
    0.95533926735353447,
    0.95636428637103732,
    0.95737182474203542,
    0.9583620447791491,
    0.95933510994032567,
    0.96029118477226649,
    0.96123043485403392,
    0.96215302674085701,
    0.96305912790815673,
    0.96394890669580979,
    0.96482253225267245,
    0.96568017448138121,
    0.966522003983453,
    0.96734819200469957,
    0.96815891038097912,
    0.96895433148430032,
    0.96973462816929867,
    0.97049997372010177,
    0.97125054179760317,
    0.97198650638715889,
    0.97270804174672698,
    0.97341532235546435,
    0.97410852286279814,
    0.97478781803798786,
    0.97545338272019366,
    0.9761053917690663,
    0.97674402001587302,
    0.97736944221517574,
    0.97798183299707431,
    0.97858136682002972,
    0.97916821792427999,
    0.97974256028586315,
    0.98030456757125961,
    0.98085441309266641,
    0.98139226976391514,
    0.98191831005704677,
    0.98243270595955268,
    0.98293562893229447,
    0.00018585566412327074,
    0.00019841037553619727,
    0.00021167096642244775,
    0.00022566899180397626,
    0.00024043703467150463,
    0.00025600872513234914,
    0.00027241875949100521,
    0.00028970291924987866,
    0.00030789809001743434,
    0.00032704228031092052,
    0.00034717464024072557,
    0.00036833548006332472,
    0.00039056628858968591,
    0.00041390975143592191,
    0.00043840976910290025,
    0.0004641114748714563,
    0.00049106125249979342,
    0.0005193067537096048,
    0.00054889691544740343,
    0.00057988197690751525,
    0.0006123134963031562,
    0.00064624436737199711,
    0.00068172883560260647,
    0.00071882251416815557,
    0.00075758239955377453,
    0.00079806688686395743,
    0.00084033578479643736,
    0.00088445033026897531,
    0.00093047320268554708,
    0.00097846853782845096,
    0.0010285019413629171,
    0.0010806405019408523,
    0.0011349528038904313,
    0.0011915089394783126,
    0.0012503805207313484,
    0.0013116406908047498,
    0.0013753641348837661,
    0.0014416270906060531,
    0.0015105073579920135,
    0.0015820843088705276,
    0.0016564388957876182,
    0.0017336536603857417,
    0.0018138127412415403,
    0.0018970018811500559,
    0.0019833084338435609,
    0.0020728213701333496,
    0.0021656312834629952,
    0.0022618303948617881,
    0.0023615125572872494,
    0.0024647732593458282,
    0.0025717096283810969,
    0.0026824204329189858,
    0.0027970060844598123,
    0.0029155686386071068,
    0.003038211795523469,
    0.0031650408997039377,
    0.0032961629390576195,
    0.003431686543288568,
    0.0035717219815671916,
    0.0037163811594837313,
    0.0038657776152756325,
    0.004020026515320926,
    0.0041792446488900271,
    0.004343550422148664,
    0.0045130638514049391,
    0.0046879065555938673,
    0.0048682017479930199,
    0.0050540742271632539,
    0.0052456503671088166,
    0.0054430581066514626,
    0.0056464269380135518,
    0.0058558878946054389,
    0.0060715735380128307,
    0.0062936179441801183,
    0.0065221566887860606,
    0.0067573268318085692,
    0.0069992669012756909,
    0.0072481168762002588,
    0.0075040181686960861,
    0.0077671136052738995,
    0.0080375474073156473,
    0.0083154651707261704,
    0.0086010138447616159,
    0.0088943417100343647,
    0.0091955983556946518,
    0.0095049346557894258,
    0.009822502744799429,
    0.010148455992355832,
    0.010482948977138215,
    0.010826137459956038,
    0.011178178356016189,
    0.011539229706379581,
    0.011909450648610178,
    0.012289001386620266,
    0.01267804315971614,
    0.013076738210848855,
    0.013485249754075035,
    0.013903741941233191,
    0.014332379827841387,
    0.014771329338222493,
    0.015220757229863715,
    0.015680831057017421,
    0.016151719133550804,
    0.016633590495052208,
    0.017126614860202408,
    0.01763096259141959,
    0.018146804654787022,
    0.018674312579272989,
    0.019213658415252834,
    0.019765014692343385,
    0.020328554376560455,
    0.020904450826810441,
    0.021492877750727477,
    0.022094009159868009,
    0.022708019324274881,
    0.023335082726423657,
    0.023975374014563986,
    0.02462906795546942,
    0.025296339386609315,
    0.02597736316775685,
    0.026672314132047577,
    0.027381367036503183,
    0.028104696512035614,
    0.028842477012946895,
    0.02959488276594046,
    0.030362087718659999,
    0.031144265487772296,
    0.03194158930661066,
    0.032754231972396068,
    0.033582365793053272,
    0.034426162533639476,
    0.035285793362403547,
    0.036161428796493866,
    0.037053238647333314,
    0.03796139196568013,
    0.038886056986393545,
    0.039827401072923582,
    0.040785590661544288,
    0.041760791205350339,
    0.042753167118036749,
    0.043762881717482036,
    0.04479009716915508,
    0.045834974429366349,
    0.046897673188384198,
    0.047978351813437251,
    0.049077167291624008,
    0.050194275172750949,
    0.051329829512120603,
    0.05248398281329128,
    0.053656885970830147,
    0.054848688213081517,
    0.056059537044972525,
    0.057289578190878092,
    0.058538955537567612,
    0.0598078110772556,
    0.061096284850778677,
    0.062404514890921484,
    0.063732637165913944,
    0.065080785523122631,
    0.066449091632958743,
    0.067837684933025458,
    0.069246692572527344,
    0.07067623935696464,
    0.072126447693134865,
    0.073597437534464891,
    0.075089326326695793,
    0.076602228953943349,
    0.078136257685156862,
    0.079691522120998767,
    0.081268129141167586,
    0.082866182852186832,
    0.084485784535682118,
    0.086127032597168668,
    0.087790022515371796,
    0.08947484679210202,
    0.091181594902706997,
    0.092910353247122213,
    0.094661205101541801,
    0.096434230570731472,
    0.098229506541004499,
    0.10004710663388232,
    0.10188710116046065,
    0.10374955707650189,
    0.10563453793827465,
    0.1075421038591606,
    0.10947231146704914,
    0.11142521386253951,
    0.11340086057797051,
    0.11539929753729697,
    0.11742056701683229,
    0.11946470760687623,
    0.12153175417424637,
    0.12362173782573173,
    0.12573468587248676,
    0.12787062179538333,
    0.13002956521133838,
    0.13221153184063425,
    0.13441653347524865,
    0.13664457794821069,
    0.13889566910399909,
    0.14116980676999843,
    0.14346698672902905,
    0.14578720069296511,
    0.14813043627745606,
    0.15049667697776573,
    0.15288590214574241,
    0.15529808696793421,
    0.15773320244486216,
    0.16019121537146414,
    0.16267208831872132,
    0.16517577961647995,
    0.16770224333747863,
    0.17025142928259299,
    0.1728232829673079,
    0.17541774560942713,
    0.17803475411803041,
    0.18067424108368649,
    0.18333613476993157,
    0.18602035910602058,
    0.18872683368095938,
    0.19145547373882518,
    0.19420619017538149,
    0.19697888953599438,
    0.19977347401485535,
    0.20258984145551634,
    0.20542788535274148,
    0.20828749485567977,
    0.21116855477236257,
    0.21407094557552908,
    0.21699454340978236,
    0.21993922010007819,
    0.22290484316154854,
    0.22589127581066051,
    0.22889837697771159,
    0.2319260013206613,
    0.23497399924029871,
    0.23804221689674485,
    0.24113049622728874,
    0.24423867496555476,
    0.24736658666199896,
    0.25051406070573123,
    0.25368092234765988,
    0.25686699272495406,
    0.26007208888681993,
    0.26329602382158512,
    0.26653860648508565,
    0.26979964183034949,
    0.27307893083856949,
    0.2763762705513591,
    0.27969145410428248,
    0.28302427076165088,
    0.2863745059525768,
    0.28974194130827624,
    0.29312635470060927,
    0.29652752028184909,
    0.29994520852566769,
    0.30337918626932836,
    0.30682921675707148,
    0.31029505968468274,
    0.31377647124523034,
    0.31727320417595789,
    0.32078500780631969,
    0.32431162810714365,
    0.32785280774090769,
    0.33140828611311424,
    0.33497779942474748,
    0.33856108072579671,
    0.34215785996983061,
    0.34576786406960414,
    0.34939081695368174,
    0.35302643962405861,
    0.3566744502147623,
    0.36033456405141573,
    0.36400649371174276,
    0.36768994908699737,
    0.37138463744429612,
    0.37509026348983426,
    0.37880652943296511,
    0.38253313505112113,
    0.38626977775555682,
    0.39001615265789069,
    0.39377195263742515,
    0.39753686840922209,
    0.40131058859291152,
    0.40509279978221058,
    0.40888318661513001,
    0.41268143184484435,
    0.41648721641120245,
    0.42030021951285462,
    0.42412011867997201,
    0.42794658984753348,
    0.43177930742915599,
    0.43561794439144352,
    0.43946217232882867,
    0.44331166153888274,
    0.44716608109806821,
    0.45102509893790765,
    0.45488838192154413,
    0.45875559592066623,
    0.46262640589277182,
    0.46650047595874461,
    0.47037746948071663,
    0.47425704914018979,
    0.47813887701639107,
    0.48202261466483315,
    0.48590792319605447,
    0.48979446335451154,
    0.49368189559759679,
    0.49756988017475434,
    0.50145807720666724,
    0.50534614676448852,
    0.5092337489490899,
    0.51312054397029927,
    0.51700619222610145,
    0.52089035438177467,
    0.52477269144893557,
    0.52865286486446494,
    0.5325305365692885,
    0.53640536908698533,
    0.54027702560219637,
    0.54414517003880725,
    0.54800946713787779,
    0.55186958253529284,
    0.55572518283910644,
    0.55957593570655406,
    0.56342150992070661,
    0.56726157546673961,
    0.57109580360779244,
    0.57492386696039099,
    0.57874543956940916,
    0.58256019698254302,
    0.58636781632427337,
    0.59016797636929064,
    0.5939603576153587,
    0.59774464235559188,
    0.60152051475012258,
    0.60528766089713293,
    0.60904576890323026,
    0.61279452895313946,
    0.61653363337869171,
    0.6202627767270853,
    0.62398165582839671,
    0.62768996986231895,
    0.63138742042410634,
    0.63507371158970283,
    0.63874854998003416,
    0.64241164482444146,
    0.6460627080232364,
    0.64970145420935843,
    0.65332760080911279,
    0.65694086810197083,
    0.66054097927941424,
    0.66412766050280259,
    0.66770064096024784,
    0.67125965292247747,
    0.67480443179766747,
    0.67833471618523111,
    0.68185024792854354,
    0.68535077216658846,
    0.68883603738451082,
    0.69230579546305937,
    0.6957598017269051,
    0.6991978149918211,
    0.7026195976107098,
    0.70602491551846502,
    0.70941353827565434,
    0.71278523911101066,
    0.7161397949627214,
    0.71947698651850234,
    0.72279659825444598,
    0.72609841847263512,
    0.72938223933750912,
    0.732647856910976,
    0.73589507118626007,
    0.7391236861204763,
    0.74233350966592571,
    0.74552435380010074,
    0.7486960345543977,
    0.75184837204152755,
    0.75498119048161894,
    0.7580943182270099,
    0.7611875877857226,
    0.7642608358436157,
    0.76731390328521343,
    0.77034663521320446,
    0.77335888096661176,
    0.77635049413762791,
    0.77932133258711722,
    0.78227125845878065,
    0.78520013819198553,
    0.78810784253325883,
    0.79099424654644357,
    0.79385922962152222,
    0.79670267548210483,
    0.79952447219158773,
    0.80232451215798206,
    0.80510269213741759,
    0.80785891323632453,
    0.81059308091229698,
    0.81330510497364339,
    0.81599489957762927,
    0.81866238322741658,
    0.82130747876770727,
    0.82393011337909672,
    0.82653021857114495,
    0.82910773017417239,
    0.83166258832978779,
    0.83419473748015971,
    0.83670412635603597,
    0.83919070796352502,
    0.84165443956964636,
    0.84409528268666079,
    0.84651320305519329,
    0.84890817062615631,
    0.85128015954148872,
    0.85362914811372081,
    0.85595511880437669,
    0.8582580582012308,
    0.86053795699442748,
    0.86279480995148095,
    0.86502861589116786,
    0.86723937765632764,
    0.86942710208558593,
    0.8715917999840157,
    0.87373348609275248,
    0.87585217905757895,
    0.8779479013964967,
    0.88002067946630003,
    0.88207054342817004,
    0.88409752721230717,
    0.88610166848161742,
    0.88808300859447342,
    0.89004159256656556,
    0.89197746903186481,
    0.8938906902027135,
    0.89578131182906506,
    0.89764939315689118,
    0.89949499688577639,
    0.90131818912572004,
    0.90311903935316573,
    0.90489762036627874,
    0.90665400823949138,
    0.90838828227733848,
    0.91010052496760219,
    0.91179082193378869,
    0.91345926188695781,
    0.91510593657692652,
    0.91673094074286832,
    0.91833437206333113,
    0.91991633110569337,
    0.92147692127508385,
    0.92301624876278332,
    0.92453442249413376,
    0.92603155407597604,
    0.92750775774363781,
    0.92896315030749521,
    0.93039785109913165,
    0.93181198191711356,
    0.93320566697240925,
    0.93457903283347032,
    0.93593220837100133,
    0.93726532470243773,
    0.93857851513615687,
    0.93987191511544377,
    0.94114566216223439,
    0.94239989582065897,
    0.9436347576004096,
    0.94485039091995127,
    0.94604694104960263,
    0.9472245550545062,
    0.94838338173751224,
    0.94952357158199652,
    0.95064527669463617,
    0.95174865074816462,
    0.95283384892412748,
    0.95390102785566144,
    0.9549503455703181,
    0.95598196143295333,
    0.95699603608870465,
    0.95799273140607721,
    0.95897221042015912,
    0.95993463727598771,
    0.9608801771720864,
    0.96180899630419336,
    0.96272126180920248,
    0.9636171417093351,
    0.96449680485656408,
    0.96536042087730844,
    0.96620816011741895,
    0.96704019358747251,
    0.96785669290839527,
    0.96865783025743213,
    0.96944377831448136,
    0.97021471020881245,
    0.97097079946618403,
    0.97171221995638013,
    0.97243914584118196,
    0.97315175152279076,
    0.97385021159271978,
    0.97453470078117022,
    0.9752053939069083,
    0.97586246582765701,
    0.97650609139102029,
    0.97713644538595179,
    0.97775370249478444,
    0.97835803724583514,
    0.97894962396659624,
    0.97952863673752977,
    0.98009524934647563,
    0.98064963524368798,
    0.9811919674975097,
    0.98172241875070021,
    0.98224116117742488,
    0.98274836644091934,
    0.98324420565183857,
    0.98372884932730176,
    0.98420246735064165,
    0.9846652289318697,
    0.98511730256886476,
    0.98555885600929505,
    0.98599005621328073,
    0.986411069316807,
    0.9868220605958925,
    0.98722319443152318,
    0.98761463427535567,
    0.98799654261619863,
    0.98836908094727716,
    0.988732409734285,
    0.98908668838423153,
    0.98943207521508658,
    0.98976872742622812,
    0.99009680106969655,
    0.99041645102225939,
    0.99072783095828898,
    0.99103109332345596,
    0.99132638930924111,
    0.99161386882826741,
    0.99189368049045301,
    0.99216597157998743,
    0.99243088803313029,
    0.99268857441683411,
    0.99293917390818998,
    0.99318282827469662,
    0.99341967785535057,
    0.99364986154255808,
    0.99387351676486468,
    0.99409077947050273,
    0.99430178411175285,
    0.9945066636301173,
    0.99470554944230138,
    0.99489857142700078,
    0.9950858579124886,
    0.99526753566500048,
    0.9954437298779113,
    0.99561456416169902,
    0.99578016053469121,
    0.99594063941458733,
    0.99609611961075117,
    0.99624671831726885,
    0.9963925511067625,
    0.9965337319249562,
    0.99667037308598405,
    0.99680258526843535,
    0.99693047751212693,
    0.99705415721559598,
    0.99717373013430444,
    0.997289300379546,
    0.99740097041804787,
    0.99750884107225635,
    0.99761301152129866,
    0.99771357930261029,
    0.99781064031421718,
    0.99790428881766491,
    0.9979946174415818,
    0.99808171718586702,
    0.99816567742649276,
    0.99824658592090787,
    0.99832452881403422,
    0.99839959064484207,
    0.99847185435349406,
    0.99854140128904567,
    0.99860831121769034,
    0.99867266233153695,
    0.99873453125790734,
    0.99879399306914174,
    0.99885112129289944,
    0.99890598792294161,
    0.99895866343038409,
    0.99900921677540699,
    0.99905771541940835,
    0.99910422533758736,
    0.99914881103194653,
    0.99919153554469675,
    0.9992324604720545,
    0.999271645978415,
    0.99930915081089067,
    0.99934503231419902,
    0.99937934644588855,
    0.99941214779188692,
    0.99944348958235951,
    0.99947342370786385,
    0.99950200073578688,
    0.99952926992705082,
    0.9995552792530743,
    0.99958007541297578,
    0.99960370385100406,
    0.99962620877418507,
    0.99964763317016803,
    0.99966801882526057,
    0.99968740634263809,
    0.99970583516071354,
    0.9997233435716556,
    0.99973996874004167,
    0.99975574672163203,
    0.99977071248225247,
    0.99978489991677388,
    0.9997983418681734,
    0.99981107014666681,
    0.9998231155488988,
    0.99983450787717798,
    0.99984527595874628,
    0.99985544766506829,
    0.99986504993113068,
    0.99987410877473815,
    0.99988264931579618,
    0.99989069579556655,
    0.9998982715958874,
    0.99990539925834399,
    0.99991210050338031,
    0.9999183962493422,
    0.99992430663143828,
    0.99992985102061116,
    0.99993504804230815,
    0.99993991559513984,
    0.99994447086941962,
    0.99994873036557264,
    0.9999527099124047,
    0.99995642468522483,
    0.99995988922380885,
    0.99996311745019872,
    0.99996612268632701,
    0.9999689176714599,
    0.99997151457945088,
    0.99997392503579563,
    0.99997616013448387,
    0.99997823045463885,
    0.99998014607693786,
    0.99998191659980851,
    0.99998355115539383,
    0.99998505842527929,
    0.99998644665597802,
    0.99998772367416655,
    0.99998889690166837,
    0.9999899733701777,
    0.99999095973572127,
    0.99999186229285286,
    0.99999268698857524,
    0.99999343943598895,
    0.99999412492766093,
    0.99999474844871206,
    0.99999531468962011,
    0.99999582805873521,
    0.99999629269450607,
    0.99999671247741362,
    0.99999709104161261,
    0.99999743178627698,
    0.99999773788665014,
    0.99999801230479735,
    0.99999825780006102,
    0.99999847693921784,
    0.99999867210633731,
    0.99999884551234297,
    0.99999899920427571,
    0.99999913507426053,
    0.99999925486817731,
    0.99999936019403746,
    0.99999945253006695,
    0.99999953323249902,
    0.99999960354307738,
    0.99999966459627232,
    0.99999971742621285,
    0.99999976297333804,
    0.99999980209076866,
    0.99999983555040506,
    0.99999986404875274,
    0.99999988821248065,
    0.99999990860371579,
    0.99999992572507745,
    0.9999999400244578,
    0.99999995189955126,
    0.99999996170213834,
    0.99999996974212979,
    0.99999997629137505,
    0.99999998158724146,
    0.99999998583596938,
    0.99999998921580968,
    0.99999999187994926,
    0.99999999395923045,
    0.99999999556467167,
    0.99999999678979523,
    0.99999999771276926,
    0.99999999839837039,
    0.9999999988997752,
    0.99999999926018535,
    0.9999999995142973,
    0.99999999968961983,
    0.9999999998076512,
    0.99999999988492005,
    0.9999999999339001,
    0.9999999999638054,
    0.99999999998127542,
    0.99999999999095579,
    0.99999999999598588,
    0.99999999999839895,
    0.99999999999944444,
    0.99999999999984046,
    0.99999999999996503,
    0.999999999999995,
    0.99999999999999967,
    1,
    1,
    6.7484639267479648e-19,
    1.4000512336177896e-16,
    2.7898225984973458e-15,
    2.2541373189961688e-14,
    1.1242365331126275e-13,
    4.1498991121415065e-13,
    1.2468913262561542e-12,
    3.2255295396392943e-12,
    7.4463453461731081e-12,
    1.5719259489043674e-11,
    3.0872733153744752e-11,
    5.7133860462846746e-11,
    1.0059284306400181e-10,
    1.6976012781760695e-10,
    2.7622441267018772e-10,
    4.3541964309133445e-10,
    6.6750903200660102e-10,
    9.9839404003098361e-10,
    1.460856150056366e-09,
    2.0958391609346546e-09,
    2.9538796091932804e-09,
    4.0966928054923799e-09,
    5.5989218439582746e-09,
    7.5500568076848451e-09,
    1.0056531252665779e-08,
    1.3244002905293218e-08,
    1.7259825355422361e-08,
    2.2275717367951396e-08,
    2.849063627099944e-08,
    3.6133861708183571e-08,
    4.546829586630454e-08,
    5.6793986108057834e-08,
    7.0451875752304526e-08,
    8.6827788552080286e-08,
    1.0635665222300937e-07,
    1.2952696617224582e-07,
    1.5688551837061124e-07,
    1.8904235609836615e-07,
    2.2667601507817043e-07,
    2.7053901128739314e-07,
    3.2146359951616012e-07,
    3.8036780250751782e-07,
    4.4826171428198816e-07,
    5.2625408101073754e-07,
    6.1555916255972887e-07,
    7.1750387758172285e-07,
    8.3353523478399173e-07,
    9.6522805274726877e-07,
    1.1142929704159245e-06,
    1.2825847501208106e-06,
    1.4721108747347938e-06,
    1.6850404402968881e-06,
    1.923713345174238e-06,
    2.1906497765621918e-06,
    2.4885599948514906e-06,
    2.820354416118344e-06,
    3.1891539927180801e-06,
    3.5983008916862643e-06,
    4.0513694703729049e-06,
    4.552177548455733e-06,
    5.1047979751977837e-06,
    5.7135704905327483e-06,
    6.3831138782790343e-06,
    7.1183384095002831e-06,
    7.9244585737465389e-06,
    8.8070060956263592e-06,
    9.771843233876274e-06,
    1.0825176359810147e-05,
    1.19735698117475e-05,
    1.3223960021736754e-05,
    1.4583669910607014e-05,
    1.6060423547100392e-05,
    1.7662361066556348e-05,
    1.9398053844340238e-05,
    2.1276519918930202e-05,
    2.3307239659300261e-05,
    2.5500171670962734e-05,
    2.7865768934760467e-05,
    3.0414995172228719e-05,
    3.315934143107822e-05,
    3.611084288408507e-05,
    3.9282095834409962e-05,
    4.268627492010875e-05,
    4.6337150510339049e-05,
    5.0249106285513173e-05,
    5.4437156993396996e-05,
    5.8916966372906721e-05,
    6.3704865237111896e-05,
    6.8817869706713129e-05,
    7.4273699585026979e-05,
    8.0090796865278855e-05,
    8.6288344360777332e-05,
    9.2886284448320182e-05,
    9.9905337914964551e-05,
    0.00010736702289807981,
    0.00011529367390839325,
    0.00012370846092553544,
    0.00013263540855539357,
    0.00014209941523838881,
    0.00015212627249760566,
    0.00016274268421552046,
    0.00017397628592789996,
    0.8453071101910139,
    0.84771355765449319,
    0.85009703886874399,
    0.85245753006362812,
    0.85479501162894456,
    0.85710946809225674,
    0.85940088809557946,
    0.86166926437093661,
    0.86391459371480561,
    0.86613687696146213,
    0.86833611895523999,
    0.87051232852172167,
    0.87266551843787432,
    0.87479570540114759,
    0.87690290999755005,
    0.87898715666871929,
    0.88104847367800454,
    0.88308689307557697,
    0.88510245066258719,
    0.88709518595438674,
    0.88906514214283172,
    0.89101236605768741,
    0.89293690812715276,
    0.8948388223375231,
    0.89671816619201139,
    0.89857500066874652,
    0.90040939017796873,
    0.90222140251844352,
    0.90401110883311175,
    0.90577858356399898,
    0.90752390440640296,
    0.90924715226238129,
    0.91094841119355952,
    0.91262776837328174,
    0.91428531403812396,
    0.91592114143879355,
    0.91753534679043436,
    0.91912802922236181,
    0.92069929072724743,
    0.92224923610977694,
    0.92377797293480313,
    0.92528561147501642,
    0.92677226465815465,
    0.92823804801377519,
    0.92968307961961261,
    0.93110748004754262,
    0.93251137230917669,
    0.9338948818011098,
    0.93525813624984266,
    0.93660126565640245,
    0.9379244022406853,
    0.93922768038554105,
    0.9405112365806253,
    0.94177520936603976,
    0.94301973927578497,
    0.94424496878104669,
    0.94545104223333909,
    0.94663810580752739,
    0.94780630744475236,
    0.94895579679527764,
    0.95008672516128456,
    0.95119924543963474,
    0.95229351206462209,
    0.95336968095073804,
    0.95442790943547029,
    0.95546835622215631,
    0.95649118132291444,
    0.95749654600167255,
    0.95848461271731655,
    0.95945554506697805,
    0.96040950772948386,
    0.96134666640898592,
    0.96226718777879372,
    0.96317123942542793,
    0.96405898979291627,
    0.96493060812735021,
    0.96578626442172255,
    0.96662612936106507,
    0.96745037426790537,
    0.96825917104806036,
    0.96905269213678624,
    0.96983111044530257,
    0.97059459930770786,
    0.97134333242830473,
    0.97207748382935144,
    0.97279722779925726,
    0.97350273884123828,
    0.9741941916224488,
    0.97487176092360595,
    0.97553562158912244,
    0.97618594847776274,
    0.97682291641383701,
    0.97744670013894941,
    0.978057474264313,
    0.97865541322364591,
    0.97924069122666313,
    0.97981348221317555,
    0.98037395980781106,
    0.98092229727536828,
    0.98145866747681643,
    0.98198324282595217,
    0.98249619524672549,
    0.98299769613124477,
    0.98348791629847265,
    0.98396702595362207,
    0.98443519464826179,
    0.98489259124114292,
    0.98533938385975239,
    0.98577573986260514,
    0.98620182580228,
    0.98661780738921023,
    0.98702384945623345,
    0.98742011592390944,
    0.98780676976661208,
    0.98818397297940086,
    0.98855188654567927,
    0.98891067040564318,
    0.98926048342552664,
    0.98960148336764697,
    0.98993382686125619,
    0.99025766937419957,
    0.99057316518538607,
    0.99088046735807456,
    0.99117972771397556,
    0.99147109680817325,
    0.99175472390486752,
    0.99203075695393883,
    0.9922993425683353,
    0.99256062600228379,
    0.99281475113032391,
    0.9930618604271656,
    0.99330209494836896,
    0.99353559431184502,
    0.99376249668017735,
    0.99398293874376065,
    0.99419705570475525,
    0.9944049812618555,
    0.99460684759586693,
    0.99480278535609157,
    0.99499292364751546,
    0.99517739001879524,
    0.99535631045103978,
    0.99552980934738089,
    0.99569800952332965,
    0.99586103219791056,
    0.99601899698557117,
    0.99617202188885645,
    0.99632022329184688,
    0.99646371595434813,
    0.99660261300682951,
    0.99673702594610158,
    0.99686706463172547,
    0.99699283728314714,
    0.99711445047754621,
    0.99723200914839361,
    0.99734561658470666,
    0.99745537443099386,
    0.99756138268787997,
    0.99766373971340172,
    0.99776254222496397,
    0.9978578853019463,
    0.997949862388951,
    0.99803856529967971,
    0.99812408422143006,
    0.9982065077201997,
    0.99828592274638805,
    0.99836241464108333,
    0.99843606714292399,
    0.99850696239552228,
    0.9985751809554384,
    0.99864080180069359,
    0.99870390233980888,
    0.99876455842135847,
    0.99882284434402402,
    0.9988788328671383,
    0.99893259522170419,
    0.99898420112187825,
    0.99903371877690306,
    0.99908121490347757,
    0.99912675473855217,
    0.99917040205253305,
    0.99921221916288538,
    0.99925226694812019,
    0.99929060486215182,
    0.99932729094901285,
    0.99936238185791271,
    0.99939593285862605,
    0.99942799785719838,
    0.99945862941195374,
    0.99948787874979272,
    0.99951579578276528,
    0.99954242912490643,
    0.99956782610931949,
    0.99959203280549569,
    0.99961509403685378,
    0.99963705339848818,
    0.99965795327511198,
    0.99967783485918016,
    0.99969673816918092,
    0.99971470206808166,
    0.99973176428191601,
    0.99974796141849886,
    0.99976332898625653,
    0.99977790141315959,
    0.99979171206574469,
    0.99980479326821281,
    0.99981717632159217,
    0.99982889152295273,
    0.99983996818465959,
    0.99985043465365386,
    0.99986031833074873,
    0.99986964568992842,
    0.99987844229763922,
    0.99988673283205964,
    0.99989454110234,
    0.99990189006779895,
    0.99990880185706643,
    0.99991529778716193,
    0.99992139838249749,
    0.9999271233937963,
    0.99993249181691357,
    0.99993752191155316,
    0.99994223121986769,
    0.99994663658493366,
    0.99995075416909185,
    0.99995459947214438,
    0.99995818734939801,
    0.99996153202954752,
    0.99996464713238831,
    0.99996754568635104,
    0.99997024014585101,
    0.99997274240844347,
    0.99997506383177792,
    0.99997721525034433,
    0.99997920699200415,
    0.99998104889429973,
    0.99998275032053552,
    0.99998432017562477,
    0.99998576692169661,
    0.99998709859345714,
    0.99998832281329952,
    0.99998944680615831,
    0.99999047741410307,
    0.99999142111066652,
    0.99999228401490359,
    0.9999930719051775,
    0.99999379023266832,
    0.99999444413460159,
    0.9999950384471944,
    0.99999557771831415,
    0.99999606621985038,
    0.99999650795979422,
    0.99999690669402685,
    0.99999726593781291,
    0.99999758897699842,
    0.99999787887891223,
    0.99999813850296981,
    0.99999837051097973,
    0.99999857737715148,
    0.99999876139780486,
    0.99999892470078289,
    0.99999906925456605,
    0.9999991968770916,
    0.99999930924427627,
    0.9999994078982466,
    0.99999949425527634,
    0.99999956961343395,
    0.99999963515994228,
    0.99999969197825211,
    0.99999974105483358,
    0.99999978328568739,
    0.99999981948257888,
    0.99999985037899963,
    0.99999987663585876,
    0.99999989884690865,
    0.99999991754390916,
    0.99999993320153424,
    0.99999994624202648,
    0.99999995703960309,
    0.99999996592461893,
    0.99999997318749245,
    0.9999999790823989,
    0.99999998383073629,
    0.9999999876243717,
    0.99999999062867129,
    0.99999999298532216,
    0.99999999481495183,
    0.9999999962195516,
    0.99999999728471034,
    0.99999999808166584,
    0.99999999866918143,
    0.99999999909525294,
    0.99999999939865536,
    0.99999999961033548,
    0.9999999997546587,
    0.99999999985051624,
    0.99999999991230248,
    0.99999999995076905,
    0.99999999997376388,
    0.99999999998686284,
    0.99999999999390321,
    0.99999999999742639,
    0.99999999999903788,
    0.99999999999969391,
    0.99999999999992217,
    0.9999999999999859,
    0.99999999999999856,
    1,
    1,
    1.7921341265513171e-21,
    1.5968418626961903e-17,
    7.3243630434786802e-16,
    8.5706291501317543e-15,
    5.2711396165872979e-14,
    2.2266882047348308e-13,
    7.3504447665101103e-13,
    2.0380685598782038e-12,
    4.9625032550959404e-12,
    1.0927445845911843e-11,
    2.2209524202852791e-11,
    4.2283925466445602e-11,
    7.6245639984230511e-11,
    1.3131923690815882e-10,
    2.1746541535010566e-10,
    3.4809249567285945e-10,
    5.4088092941052732e-10,
    8.1872881351858075e-10,
    1.2108262952066291e-09,
    1.7538666475674337e-09,
    2.4934016836486901e-09,
    3.4853490575887633e-09,
    4.7976588760269168e-09,
    6.5121469112796023e-09,
    8.7265015698279611e-09,
    1.1556471625572594e-08,
    1.513824157716839e-08,
    1.9631001332647136e-08,
    2.521971676256474e-08,
    3.2118107495172828e-08,
    4.0571838153714578e-08,
    5.0861929056988677e-08,
    6.3308392219923508e-08,
    7.8274098301169832e-08,
    9.6168879949771496e-08,
    1.1745387680292991e-07,
    1.4264612718186401e-07,
    1.7232341132290784e-07,
    2.0712935076641495e-07,
    2.4777876830688331e-07,
    2.9506331268411367e-07,
    3.498573519673047e-07,
    4.1312413935191579e-07,
    4.8592225485302954e-07,
    5.6941232613897633e-07,
    6.6486403150528506e-07,
    7.7366338774175264e-07,
    8.9732032539473356e-07,
    1.0374765536727934e-06,
    1.1959137169868127e-06,
    1.3745618448556193e-06,
    1.5755080966454405e-06,
    1.8010058023460643e-06,
    2.0534838003187398e-06,
    2.3355560726807126e-06,
    2.6500316787190365e-06,
    2.9999249864521124e-06,
    3.3884662021814474e-06,
    3.8191121975985597e-06,
    4.2955576337329952e-06,
    4.8217463807471974e-06,
    5.401883232302678e-06,
    6.0404459129397961e-06,
    6.7421973766305375e-06,
    7.5121983943803311e-06,
    8.3558204284711389e-06,
    9.2787587906541904e-06,
    1.0287046081316797e-05,
    1.138706590636402e-05,
    1.2585566868272624e-05,
    1.3889676827492017e-05,
    1.5306917430084842e-05,
    1.6845218897218882e-05,
    1.8512935071841885e-05,
    2.0318858717592325e-05,
    2.2272237064721863e-05,
    2.4382787597529822e-05,
    2.6660714077536144e-05,
    2.9116722796347816e-05,
    3.176203905190405e-05,
    3.4608423841518595e-05,
    3.7668190764872803e-05,
    4.095422312985147e-05,
    4.4479991253854266e-05,
    4.8259569952959991e-05,
    5.2307656211068075e-05,
    5.6639587020892715e-05,
    6.1271357388439308e-05,
    6.621963849235111e-05,
    7.1501795989276044e-05,
    7.7135908456169841e-05,
    8.3140785960221961e-05,
    8.9535988746865757e-05,
    9.6341846036113513e-05,
    0.00010357947491724125,
    0.00011127079933163708,
    0.00011943856913342087,
    0.00012810637921724228,
    0.00013729868870246845,
    0.00014704084016278269,
    0.0001573590788900314,
    0.00016828057218097679,
    0.00017983342863544125,
    0.00019204671745416143,
    0.00020495048772450967,
    0.00021857578768208643,
    0.00023295468393603796,
    0.0002481202806458135,
    0.00026410673863693954,
    0.00028094929444326215,
    0.00029868427926298482,
    0.00031734913781571704,
    0.00033698244708763882,
    0.00035762393495178879,
    0.00037931449865038672,
    0.00040209622312601935,
    0.00042601239918843667,
    0.00045110754150363671,
    0.00047742740639185256,
    0.00050501900942099957,
    0.00053393064278209269,
    0.00056421189243310366,
    0.00059591365499769443,
    0.00062908815440523846,
    0.00066378895825852755,
    0.00070007099391554601,
    0.00073799056427170358,
    0.00077760536322891418,
    0.00081897449083793267,
    0.00086215846810037879,
    0.00090721925141691226,
    0.00095422024666806128,
    0.0010032263229142555,
    0.0010543038257016693,
    0.0011075205899605469,
    0.0011629459524827525,
    0.0012206507639653697,
    0.0012807074006072629,
    0.0013431897752456104,
    0.0014081733480195235,
    0.0014757351365479838,
    0.0015459537256094424,
    0.0016189092763105673,
    0.0016946835347317493,
    0.0017733598400371357,
    0.0018550231320371027,
    0.0019397599581912472,
    0.0020276584800401447,
    0.0021188084790542986,
    0.0022133013618888852,
    0.0023112301650331079,
    0.0024126895588431471,
    0.0025177758509479318,
    0.002626586989017147,
    0.0027392225628811318,
    0.0028557838059925411,
    0.0029763735962198938,
    0.0031010964559633571,
    0.0032300585515833897,
    0.0033633676921331021,
    0.0035011333273854747,
    0.0036434665451468414,
    0.0037904800678483165,
    0.0039422882484071407,
    0.0040990070653501983,
    0.004260754117192277,
    0.0044276486160619135,
    0.0045998113805680117,
    0.0047773648279007029,
    0.0049604329651602693,
    0.0051491413799082519,
    0.0053436172299352134,
    0.0055439892322399478,
    0.0057503876512152944,
    0.0059629442860360269,
    0.0061817924572446755,
    0.006407066992531461,
    0.0066389042117049237,
    0.0068774419108501329,
    0.0071228193456718053,
    0.0073751772140199711,
    0.0076346576375962288,
    0.0079014041428390244,
    0.0081755616409867407,
    0.0084572764073177816,
    0.0087466960595672643,
    0.009043969535520233,
    0.009349247069781812,
    0.0096626801697250238,
    0.009984421590617458,
    0.010314625309928332,
    0.010653446500817914,
    0.011001041504811697,
    0.011357567803662061,
    0.011723183990400647,
    0.012098049739584997,
    0.012482325776743488,
    0.012876173847022944,
    0.013279756683043768,
    0.013693237971967793,
    0.014116782321784527,
    0.014550555226821799,
    0.014994723032487294,
    0.015449452899247823,
    0.01591491276585361,
    0.016391271311815267,
    0.016878697919141549,
    0.017377362633346365,
    0.017887436123733939,
    0.018409089642971412,
    0.018942494985958565,
    0.019487824448004742,
    0.020045250782323464,
    0.020614947156855551,
    0.021197087110432077,
    0.021791844508288697,
    0.022399393496943462,
    0.023019908458450401,
    0.023653563964041734,
    0.024300534727171787,
    0.024960995555976098,
    0.025635121305159613,
    0.026323086827328153,
    0.027025066923777701,
    0.027741236294756447,
    0.028471769489214798,
    0.029216840854058981,
    0.029976624482924086,
    0.03075129416448286,
    0.031541023330306714,
    0.032345985002295866,
    0.033166351739695801,
    0.034002295585717435,
    0.034853988013778779,
    0.035721599873386224,
    0.0366053013356736,
    0.037505261838617757,
    0.038421650031949417,
    0.039354633721778484,
    0.040304379814953092,
    0.041271054263172055,
    0.042254822006870502,
    0.043255846918898812,
    0.044274291748015039,
    0.04531031806221137,
    0.04636408619189529,
    0.047435755172946292,
    0.048525482689669219,
    0.049633425017665413,
    0.050759736966643154,
    0.051904571823188826,
    0.053068081293520593,
    0.054250415446246335,
    0.055451722655147896,
    0.056672149542013585,
    0.057911840919541255,
    0.059170939734334155,
    0.060449587010011963,
    0.061747921790459376,
    0.063066081083234915,
    0.064404199803162421,
    0.065762410716127845,
    0.067140844383104081,
    0.068539629104426597,
    0.069958890864342316,
    0.071398753275854793,
    0.072859337525888204,
    0.074340762320792966,
    0.075843143832215576,
    0.077366595643355462,
    0.078911228695631344,
    0.080477151235779726,
    0.082064468763408074,
    0.083673283979024876,
    0.085303696732569351,
    0.086955803972462553,
    0.088629699695202471,
    0.090325474895524782,
    0.092043217517151485,
    0.093783012404148894,
    0.09554494125291689,
    0.097329082564830652,
    0.099135511599556358,
    0.10096430032906194,
    0.10281551739234367,
    0.10468922805088964,
    0.10658549414490036,
    0.10850437405028703,
    0.11044592263646742,
    0.11241019122497943,
    0.11439722754893158,
    0.1164070757133103,
    0.11843977615616263,
    0.12049536561067364,
    0.12257387706815656,
    0.12467533974197439,
    0.12679977903241074,
    0.1289472164925074,
    0.13111766979488629,
    0.13331115269957244,
    0.13552767502283494,
    0.13776724260706222,
    0.14002985729168713,
    0.14231551688517824,
    0.14462421513811191,
    0.14695594171734025,
    0.14931068218126967,
    0.15168841795626362,
    0.15408912631418392,
    0.15651278035108307,
    0.1589593489670614,
    0.16142879684730074,
    0.16392108444428696,
    0.1664361679612332,
    0.16897399933671459,
    0.1715345262305252,
    0.17411769201076793,
    0.17672343574218655,
    0.17935169217574981,
    0.18200239173949603,
    0.18467546053064698,
    0.18737082030899865,
    0.19008838849159682,
    0.1928280781487037,
    0.19558979800106308,
    0.19837345241846899,
    0.2011789414196441,
    0.20400616067343244,
    0.20685500150131134,
    0.20972535088122585,
    0.21261709145274996,
    0.21553010152357699,
    0.21846425507734196,
    0.22141942178277729,
    0.22439546700420421,
    0.22739225181335948,
    0.23040963300255923,
    0.23344746309919834,
    0.23650559038158561,
    0.23958385889611339,
    0.24268210847575925,
    0.24580017475991844,
    0.24893788921556345,
    0.25209507915972784,
    0.25527156778331028,
    0.25846717417619453,
    0.2616817133536804,
    0.26491499628422011,
    0.26816682991845447,
    0.27143701721954194,
    0.27472535719477392,
    0.27803164492846855,
    0.28135567161613512,
    0.2846972245999006,
    0.28805608740518873,
    0.29143203977864302,
    0.29482485772728267,
    0.29823431355888136,
    0.30166017592355782,
    0.30510220985656644,
    0.30856017682227582,
    0.31203383475932323,
    0.31552293812693133,
    0.3190272379523737,
    0.3225464818795763,
    0.32608041421883832,
    0.32962877599766027,
    0.3331913050126612,
    0.33676773588257131,
    0.34035780010228295,
    0.34396122609794311,
    0.34757773928307073,
    0.35120706211568137,
    0.35484891415640069,
    0.35850301212754909,
    0.3621690699731786,
    0.36584679892004268,
    0.36953590753947918,
    0.37323610181018768,
    0.37694708518187942,
    0.38066855863978016,
    0.38440022076996533,
    0.3881417678255043,
    0.39189289379339493,
    0.39565329046226355,
    0.39942264749081002,
    0.40320065247697462,
    0.40698699102780345,
    0.41078134682998974,
    0.41458340172106728,
    0.41839283576123221,
    0.42220932730576932,
    0.4260325530780581,
    0.42986218824313471,
    0.43369790648178463,
    0.43753938006514109,
    0.44138627992976409,
    0.44523827575317471,
    0.44909503602981932,
    0.45295622814743736,
    0.45682151846380714,
    0.46069057238384381,
    0.46456305443702262,
    0.46843862835510175,
    0.47231695715011773,
    0.47619770319262722,
    0.48008052829016812,
    0.48396509376591329,
    0.48785106053749017,
    0.49173808919593931,
    0.49562584008478433,
    0.49951397337918707,
    0.50340214916516002,
    0.50729002751880958,
    0.51117726858558221,
    0.51506353265948779,
    0.51894848026227103,
    0.52283177222250532,
    0.52671306975458199,
    0.53059203453756709,
    0.53446832879389872,
    0.53834161536789937,
    0.54221155780407448,
    0.54607782042517261,
    0.54994006840997833,
    0.55379796787081292,
    0.55765118593071661,
    0.5614993908002841,
    0.56534225185413012,
    0.56917943970695639,
    0.57301062628919597,
    0.57683548492220826,
    0.58065369039300063,
    0.58446491902844899,
    0.58826884876899532,
    0.59206515924179504,
    0.59585353183329015,
    0.59963364976118472,
    0.60340519814579741,
    0.60716786408076828,
    0.61092133670309523,
    0.61466530726247781,
    0.61839946918994437,
    0.6221235181657413,
    0.62583715218646008,
    0.6295400716313817,
    0.63323197932801611,
    0.63691258061681499,
    0.64058158341503668,
    0.644238698279744,
    0.64788363846991248,
    0.65151612000763004,
    0.65513586173836946,
    0.65874258539031227,
    0.66233601563270739,
    0.6659158801332441,
    0.66948190961442311,
    0.67303383790890681,
    0.67657140201383259,
    0.68009434214407083,
    0.68360240178441334,
    0.68709532774067439,
    0.69057287018969027,
    0.69403478272820207,
    0.69748082242060583,
    0.70091074984555857,
    0.70432432914142418,
    0.70772132805054699,
    0.71110151796234078,
    0.71446467395518032,
    0.71781057483708421,
    0.72113900318517754,
    0.72444974538392404,
    0.72774259166211697,
    0.73101733612862008,
    0.73427377680684791,
    0.73751171566797791,
    0.74073095866288685,
    0.74393131575280114,
    0.74711260093865761,
    0.75027463228916502,
    0.75341723196756216,
    0.75654022625706552,
    0.75964344558500352,
    0.76272672454563006,
    0.76578990192161611,
    0.76883282070421377,
    0.7718553281120909,
    0.77485727560883422,
    0.77783851891911793,
    0.78079891804353763,
    0.78373833727210707,
    0.78665664519642042,
    0.78955371472047653,
    0.79242942307016861,
    0.79528365180144012,
    0.79811628680710756,
    0.80092721832235392,
    0.80371634092889455,
    0.80648355355781909,
    0.80922875949111395,
    0.81195186636186845,
    0.81465278615317105,
    0.81733143519569884,
    0.81998773416400927,
    0.82262160807153728,
    0.82523298626430652,
    0.8278218024133619,
    0.83038799450593115,
    0.83293150483532219,
    0.83545227998956839,
    0.83795027083882756,
    0.84042543252154633,
    0.84287772442939912,
    0.15469288980898607,
    0.15712227557060085,
    0.15957456747845369,
    0.16204972916117238,
    0.16454772001043158,
    0.16706849516467784,
    0.16961200549406891,
    0.17217819758663805,
    0.17476701373569353,
    0.17737839192846272,
    0.18001226583599067,
    0.18266856480430113,
    0.18534721384682898,
    0.1880481336381315,
    0.19077124050888608,
    0.19351644644218094,
    0.19628365907110551,
    0.19907278167764608,
    0.20188371319289247,
    0.20471634819855994,
    0.20757057692983141,
    0.2104462852795235,
    0.21334335480357955,
    0.21626166272789291,
    0.21920108195646243,
    0.22216148108088202,
    0.22514272439116581,
    0.22814467188790913,
    0.23116717929578626,
    0.23421009807838386,
    0.23727327545436994,
    0.24035655441499654,
    0.24345977374293451,
    0.24658276803243789,
    0.24972536771083492,
    0.25288739906134233,
    0.25606868424719886,
    0.25926904133711315,
    0.26248828433202204,
    0.26572622319315214,
    0.26898266387137987,
    0.27225740833788298,
    0.27555025461607596,
    0.27886099681482246,
    0.28218942516291579,
    0.28553532604481963,
    0.28889848203765917,
    0.29227867194945301,
    0.29567567085857582,
    0.29908925015444143,
    0.30251917757939423,
    0.30596521727179798,
    0.30942712981030968,
    0.31290467225932561,
    0.31639759821558666,
    0.31990565785592912,
    0.32342859798616741,
    0.32696616209109319,
    0.33051809038557695,
    0.3340841198667559,
    0.33766398436729256,
    0.34125741460968767,
    0.34486413826163059,
    0.34848387999237002,
    0.35211636153008757,
    0.35576130172025594,
    0.35941841658496332,
    0.36308741938318506,
    0.36676802067198383,
    0.3704599283686183,
    0.37416284781353992,
    0.37787648183425865,
    0.38160053081005557,
    0.38533469273752219,
    0.38907866329690477,
    0.39283213591923172,
    0.39659480185420259,
    0.40036635023881534,
    0.40414646816670985,
    0.40793484075820491,
    0.41173115123100462,
    0.41553508097155101,
    0.41934630960699937,
    0.42316451507779168,
    0.42698937371080403,
    0.43082056029304355,
    0.43465774814586983,
    0.4385006091997159,
    0.44234881406928345,
    0.44620203212918708,
    0.45005993159002172,
    0.45392217957482733,
    0.45778844219592552,
    0.46165838463210068,
    0.46553167120610128,
    0.46940796546243291,
    0.47328693024541801,
    0.47716822777749474,
    0.48105151973772903,
    0.48493646734051216,
    0.48882273141441773,
    0.49270997248119042,
    0.49659785083483993,
    0.50048602662081287,
    0.50437415991521561,
    0.50826191080406069,
    0.51214893946250983,
    0.51603490623408677,
    0.51991947170983188,
    0.52380229680737278,
    0.52768304284988232,
    0.53156137164489825,
    0.53543694556297738,
    0.53930942761615619,
    0.54317848153619286,
    0.5470437718525627,
    0.55090496397018063,
    0.55476172424682535,
    0.55861372007023591,
    0.56246061993485885,
    0.56630209351821537,
    0.57013781175686529,
    0.5739674469219419,
    0.57779067269423068,
    0.58160716423876779,
    0.58541659827893278,
    0.58921865317001021,
    0.59301300897219655,
    0.59679934752302533,
    0.60057735250918998,
    0.6043467095377365,
    0.60810710620660502,
    0.61185823217449564,
    0.61559977923003473,
    0.61933144136021978,
    0.62305291481812064,
    0.62676389818981226,
    0.63046409246052082,
    0.63415320107995732,
    0.63783093002682134,
    0.64149698787245091,
    0.64515108584359937,
    0.64879293788431858,
    0.65242226071692921,
    0.65603877390205689,
    0.659642199897717,
    0.66323226411742864,
    0.66680869498733875,
    0.67037122400233973,
    0.67391958578116162,
    0.67745351812042376,
    0.68097276204762625,
    0.68447706187306867,
    0.68796616524067677,
    0.69143982317772423,
    0.69489779014343356,
    0.69833982407644213,
    0.70176568644111859,
    0.70517514227271727,
    0.70856796022135693,
    0.71194391259481127,
    0.71530277540009946,
    0.71864432838386494,
    0.72196835507153145,
    0.72527464280522613,
    0.72856298278045806,
    0.73183317008154547,
    0.73508500371577989,
    0.7383182866463196,
    0.74153282582380553,
    0.74472843221668972,
    0.74790492084027216,
    0.75106211078443652,
    0.75419982524008156,
    0.75731789152424078,
    0.76041614110388667,
    0.76349440961841442,
    0.76655253690080172,
    0.76959036699744077,
    0.77260774818664046,
    0.77560453299579579,
    0.77858057821722271,
    0.78153574492265809,
    0.78446989847642301,
    0.78738290854725002,
    0.79027464911877421,
    0.79314499849868869,
    0.79599383932656753,
    0.7988210585803559,
    0.80162654758153096,
    0.80441020199893687,
    0.80717192185129627,
    0.80991161150840318,
    0.81262917969100135,
    0.81532453946935302,
    0.81799760826050394,
    0.82064830782425013,
    0.82327656425781348,
    0.82588230798923212,
    0.82846547376947477,
    0.83102600066328536,
    0.83356383203876683,
    0.83607891555571301,
    0.83857120315269928,
    0.84104065103293857,
    0.84348721964891693,
    0.84591087368581608,
    0.84831158204373636,
    0.85068931781873036,
    0.85304405828265972,
    0.85537578486188803,
    0.85768448311482171,
    0.85997014270831285,
    0.86223275739293781,
    0.86447232497716509,
    0.86668884730042761,
    0.86888233020511374,
    0.8710527835074926,
    0.87320022096758931,
    0.87532466025802558,
    0.87742612293184341,
    0.87950463438932636,
    0.88156022384383737,
    0.88359292428668967,
    0.88560277245106844,
    0.88758980877502058,
    0.88955407736353254,
    0.89149562594971299,
    0.89341450585509963,
    0.89531077194911035,
    0.89718448260765638,
    0.8990356996709381,
    0.90086448840044364,
    0.90267091743516936,
    0.9044550587470831,
    0.90621698759585112,
    0.9079567824828485,
    0.90967452510447522,
    0.91137030030479749,
    0.91304419602753739,
    0.91469630326743068,
    0.91632671602097515,
    0.91793553123659188,
    0.9195228487642203,
    0.92108877130436861,
    0.92263340435664454,
    0.92415685616778442,
    0.92565923767920699,
    0.92714066247411175,
    0.92860124672414524,
    0.93004110913565763,
    0.93146037089557343,
    0.93285915561689592,
    0.93423758928387213,
    0.93559580019683752,
    0.93693391891676503,
    0.9382520782095406,
    0.93955041298998809,
    0.94082906026566582,
    0.94208815908045873,
    0.94332785045798639,
    0.94454827734485214,
    0.94574958455375369,
    0.94693191870647941,
    0.94809542817681114,
    0.94924026303335685,
    0.95036657498233457,
    0.95147451731033073,
    0.95256424482705371,
    0.95363591380810475,
    0.95468968193778858,
    0.955725708251985,
    0.95674415308110117,
    0.95774517799312953,
    0.95872894573682799,
    0.95969562018504695,
    0.9606453662782215,
    0.96157834996805058,
    0.9624947381613822,
    0.96339469866432637,
    0.96427840012661381,
    0.96514601198622119,
    0.96599770441428257,
    0.96683364826030416,
    0.96765401499770409,
    0.96845897666969327,
    0.96924870583551714,
    0.97002337551707596,
    0.97078315914594104,
    0.97152823051078518,
    0.97225876370524356,
    0.97297493307622229,
    0.9736769131726718,
    0.97436487869484034,
    0.97503900444402392,
    0.97569946527282825,
    0.97634643603595828,
    0.97698009154154963,
    0.97760060650305658,
    0.97820815549171125,
    0.97880291288956789,
    0.97938505284314448,
    0.97995474921767656,
    0.98051217555199521,
    0.98105750501404143,
    0.98159091035702861,
    0.98211256387626611,
    0.98262263736665367,
    0.98312130208085846,
    0.98360872868818472,
    0.98408508723414634,
    0.98455054710075218,
    0.98500527696751272,
    0.9854494447731782,
    0.98588321767821552,
    0.98630676202803225,
    0.98672024331695618,
    0.987123826152977,
    0.98751767422325654,
    0.98790195026041505,
    0.98827681600959938,
    0.98864243219633796,
    0.98899895849518826,
    0.9893465534991821,
    0.98968537469007167,
    0.99001557840938259,
    0.99033731983027495,
    0.9906507529302182,
    0.99095603046447978,
    0.99125330394043276,
    0.99154272359268225,
    0.99182443835901324,
    0.99209859585716098,
    0.99236534236240381,
    0.99262482278598008,
    0.99287718065432817,
    0.99312255808914984,
    0.99336109578829512,
    0.99359293300746854,
    0.99381820754275529,
    0.99403705571396395,
    0.99424961234878473,
    0.99445601076776002,
    0.99465638277006474,
    0.9948508586200917,
    0.99503956703483976,
    0.9952226351720993,
    0.99540018861943202,
    0.99557235138393807,
    0.99573924588280771,
    0.99590099293464984,
    0.99605771175159286,
    0.99620951993215168,
    0.9963565334548532,
    0.99649886667261456,
    0.99663663230786692,
    0.99676994144841657,
    0.99689890354403665,
    0.99702362640378006,
    0.9971442161940075,
    0.99726077743711883,
    0.99737341301098281,
    0.99748222414905208,
    0.99758731044115689,
    0.9976887698349669,
    0.99778669863811109,
    0.99788119152094568,
    0.99797234151995984,
    0.99806024004180871,
    0.99814497686796289,
    0.99822664015996287,
    0.99830531646526821,
    0.99838109072368941,
    0.99845404627439061,
    0.99852426486345203,
    0.99859182665198043,
    0.99865681022475439,
    0.99871929259939274,
    0.99877934923603462,
    0.99883705404751721,
    0.99889247941003945,
    0.99894569617429829,
    0.99899677367708573,
    0.99904577975333198,
    0.9990927807485831,
    0.99913784153189966,
    0.9991810255091621,
    0.99922239463677109,
    0.99926200943572829,
    0.99929992900608444,
    0.99933621104174142,
    0.99937091184559479,
    0.99940408634500233,
    0.99943578810756695,
    0.99946606935721793,
    0.99949498099057899,
    0.99952257259360811,
    0.99954889245849632,
    0.99957398760081151,
    0.99959790377687396,
    0.99962068550134964,
    0.99964237606504824,
    0.99966301755291231,
    0.99968265086218433,
    0.99970131572073706,
    0.99971905070555678,
    0.99973589326136303,
    0.9997518797193542,
    0.99976704531606397,
    0.99978142421231786,
    0.99979504951227549,
    0.99980795328254579,
    0.99982016657136452,
    0.99983171942781901,
    0.99984264092110997,
    0.99985295915983718,
    0.99986270131129751,
    0.99987189362078277,
    0.99988056143086657,
    0.9998887292006684,
    0.99989642052508276,
    0.99990365815396387,
    0.99991046401125316,
    0.99991685921403983,
    0.99992286409154385,
    0.99992849820401075,
    0.9999337803615076,
    0.99993872864261157,
    0.99994336041297915,
    0.99994769234378889,
    0.99995174043004709,
    0.9999555200087461,
    0.99995904577687011,
    0.99996233180923511,
    0.99996539157615849,
    0.99996823796094814,
    0.99997088327720363,
    0.99997333928592247,
    0.99997561721240247,
    0.99997772776293525,
    0.99997968114128244,
    0.99998148706492818,
    0.99998315478110278,
    0.99998469308256988,
    0.99998611032317253,
    0.99998741443313177,
    0.99998861293409369,
    0.9999897129539187,
    0.99999072124120936,
    0.99999164417957154,
    0.99999248780160566,
    0.99999325780262338,
    0.9999939595540871,
    0.99999459811676772,
    0.99999517825361928,
    0.9999957044423663,
    0.9999961808878024,
    0.99999661153379782,
    0.99999700007501358,
    0.99999734996832124,
    0.99999766444392735,
    0.99999794651619967,
    0.99999819899419762,
    0.99999842449190335,
    0.9999986254381551,
    0.99999880408628306,
    0.99999896252344633,
    0.99999910267967462,
    0.99999922633661231,
    0.9999993351359685,
    0.99999943058767382,
    0.99999951407774512,
    0.99999958687586066,
    0.99999965014264802,
    0.99999970493668733,
    0.99999975222123172,
    0.99999979287064922,
    0.99999982767658868,
    0.99999985735387287,
    0.99999988254612315,
    0.99999990383112003,
    0.99999992172590169,
    0.9999999366916078,
    0.999999949138071,
    0.99999995942816189,
    0.99999996788189249,
    0.99999997478028324,
    0.99999998036899862,
    0.99999998486175845,
    0.99999998844352833,
    0.99999999127349848,
    0.99999999348785307,
    0.99999999520234117,
    0.99999999651465099,
    0.99999999750659829,
    0.99999999824613339,
    0.99999999878917367,
    0.99999999918127114,
    0.9999999994591191,
    0.99999999965190756,
    0.99999999978253462,
    0.99999999986868071,
    0.99999999992375432,
    0.99999999995771605,
    0.99999999997779043,
    0.99999999998907252,
    0.99999999999503753,
    0.99999999999796196,
    0.99999999999926492,
    0.99999999999977729,
    0.99999999999994726,
    0.99999999999999145,
    0.99999999999999922,
    1,
    1,
    6.4019241857743263e-20,
    5.1417728658284795e-17,
    1.4757866134410007e-15,
    1.4133304840773797e-14,
    7.7773062922292888e-14,
    3.0609362200748802e-13,
    9.6213824113713251e-13,
    2.5736239851331434e-12,
    6.096771271156868e-12,
    1.3137160793084063e-11,
    2.6236098158974609e-11,
    4.9230913016590165e-11,
    8.7697558957928455e-11,
    1.4948381312313835e-10,
    2.4534130020821956e-10,
    3.8966448456505846e-10,
    6.0134468631625063e-10,
    9.0474708294459364e-10,
    1.3308185567160172e-09,
    1.9183341406170298e-09,
    2.7152897012995602e-09,
    3.780448376897401e-09,
    5.1850481605833634e-09,
    7.0146778874526811e-09,
    9.3713287428280262e-09,
    1.2375628264470414e-08,
    1.6169263659537749e-08,
    2.0917601099552108e-08,
    2.6812507493214902e-08,
    3.407538106774654e-08,
    4.2960396914628045e-08,
    5.3757973475294213e-08,
    6.6798465756582421e-08,
    8.2456090874692255e-08,
    1.011530913301776e-07,
    1.2336414121519444e-07,
    1.4962100034799024e-07,
    1.8051742111857095e-07,
    2.1671431261375003e-07,
    2.5894516636950432e-07,
    3.0802174787387088e-07,
    3.6484005771465517e-07,
    4.3038656603312721e-07,
    5.0574472370780414e-07,
    5.9210175345149918e-07,
    6.9075572376021357e-07,
    8.0312290840430436e-07,
    9.3074543390084071e-07,
    1.0752992171513139e-06,
    1.2386021951710529e-06,
    1.4226228485759833e-06,
    1.6294890202289088e-06,
    1.8614970301814625e-06,
    2.1211210877794319e-06,
    2.4110230015284818e-06,
    2.734062187044545e-06,
    3.093305973138498e-06,
    3.4920402058083674e-06,
    3.9337801496343839e-06,
    4.422281685792891e-06,
    4.9615528056246316e-06,
    5.5558653984113976e-06,
    6.2097673317326729e-06,
    6.9280948224908783e-06,
    7.715985096410321e-06,
    8.5788893335311255e-06,
    9.5225858969355341e-06,
    1.0553193841660068e-05,
    1.1677186700463434e-05,
    1.2901406542836869e-05,
    1.4233078303361068e-05,
    1.5679824375231954e-05,
    1.7249679464496891e-05,
    1.8951105700263119e-05,
    2.0793007995862018e-05,
    2.2784749655675885e-05,
    2.4936168222058936e-05,
    2.7257591556510943e-05,
    2.9759854148990795e-05,
    3.2454313648988398e-05,
    3.5352867611706772e-05,
    3.8467970452442387e-05,
    4.1812650601990568e-05,
    4.5400527855644752e-05,
    4.9245830908103119e-05,
    5.3363415066344654e-05,
    5.7768780132288047e-05,
    6.2478088446802434e-05,
    6.7508183086398003e-05,
    7.2876606203687533e-05,
    7.860161750247724e-05,
    8.4702212838116695e-05,
    9.1198142933513688e-05,
    9.8109932201000168e-05,
    0.000105458897660021,
    0.00011326716794040747,
    0.00012155770236079225,
    0.00013035431007152383,
    0.00013968166925124349,
    0.0001495653463460998,
    0.0001600318153403919,
    0.00017110847704725652,
    0.00018282367840784109,
    0.00019520673178724114,
    0.00020828793425532009,
    0.00022209858684037716,
    0.00023667101374348225,
    0.00025203858150115691,
    0.00026823571808394731,
    0.00028529793191830761,
    0.0003032618308190939,
    0.00032216514081985427,
    0.0003420467248879964,
    0.00036294660151181437,
    0.00038490596314626549,
    0.00040796719450430385,
    0.00043217389068050022,
    0.00045757087509360996,
    0.0004842042172346868,
    0.0005121212502072895,
    0.00054137058804627877,
    0.00057200214280166617,
    0.00060406714137394414,
    0.00063761814208730498,
    0.00067270905098714014,
    0.00070939513784820641,
    0.0007477330518798446,
    0.00078778083711464634,
    0.00082959794746698438,
    0.00087324526144784106,
    0.00091878509652241103,
    0.00096628122309698938,
    0.0010157988781217106,
    0.0010674047782957587,
    0.0011211671328617396,
    0.0011771556559759745,
    0.0012354415786415629,
    0.0012960976601911519,
    0.0013591981993064429,
    0.0014248190445615846,
    0.0014930376044777055,
    0.00156393285707597,
    0.0016375853589166687,
    0.0017140772536120004,
    0.0017934922798003396,
    0.0018759157785699517,
    0.0019614347003202703,
    0.0020501376110490322,
    0.0021421146980537347,
    0.0022374577750360757,
    0.0023362602865982301,
    0.0024386173121200084,
    0.002544625569006169,
    0.0026543834152933615,
    0.002767990851606404,
    0.0028855495224538352,
    0.0030071627168529133,
    0.0031329353682744889,
    0.0032629740538984188,
    0.0033973869931704648,
    0.0035362840456518731,
    0.0036797767081531091,
    0.0038279781111435099,
    0.0039810030144288829,
    0.0041389678020893987,
    0.0043019904766704023,
    0.0044701906526190809,
    0.0046436895489602378,
    0.0048226099812047427,
    0.0050070763524845338,
    0.0051972146439083957,
    0.0053931524041330595,
    0.0055950187381445058,
    0.0058029442952447054,
    0.0060170612562393672,
    0.0062375033198226321,
    0.0064644056881549916,
    0.0066979050516310801,
    0.0069381395728343584,
    0.0071852488696760655,
    0.0074393739977162091,
    0.0077006574316647045,
    0.0079692430460612043,
    0.0082452760951324949,
    0.0085289031918267706,
    0.00882027228602442,
    0.0091195326419254465,
    0.0094268348146139408,
    0.0097423306258004865,
    0.010066173138743771,
    0.010398516632353026,
    0.010739516574473416,
    0.011089329594356791,
    0.01144811345432073,
    0.011816027020599109,
    0.012193230233387944,
    0.01257988407609055,
    0.012976150543766574,
    0.013382192610789795,
    0.013798174197720035,
    0.014224260137394914,
    0.014660616140247603,
    0.015107408758857131,
    0.015564805351738211,
    0.016032974046377966,
    0.016512083701527319,
    0.017002303868755266,
    0.017503804753274548,
    0.018016757174047816,
    0.01854133252318357,
    0.019077702724631732,
    0.019626040192188991,
    0.020186517786824502,
    0.020759308773336916,
    0.021344586776354053,
    0.021942525735686974,
    0.022553299861050546,
    0.023177083586162994,
    0.023814051522237274,
    0.024464378410877511,
    0.025128239076394072,
    0.025805808377551226,
    0.026497261158761692,
    0.027202772200742707,
    0.027922516170648604,
    0.028656667571695268,
    0.029405400692292086,
    0.030168889554697393,
    0.030947307863213744,
    0.03174082895193963,
    0.032549625732094592,
    0.033373870638934912,
    0.034213735578277497,
    0.035069391872649791,
    0.0359410102070837,
    0.036828760574572079,
    0.037732812221206329,
    0.03865333359101409,
    0.039590492270516128,
    0.040544454933021903,
    0.041515387282683447,
    0.042503453998327438,
    0.043508818677085594,
    0.044531643777843674,
    0.045572090564529683,
    0.046630319049261931,
    0.047706487935377948,
    0.048800754560365277,
    0.04991327483871541,
    0.051044203204722416,
    0.052193692555247674,
    0.05336189419247258,
    0.054548957766660959,
    0.055755031218953345,
    0.05698026072421504,
    0.058224790633960265,
    0.059488763419374729,
    0.060772319614458926,
    0.062075597759314698,
    0.063398734343597551,
    0.064741863750157391,
    0.066105118198890145,
    0.067488627690823269,
    0.068892519952457418,
    0.070316920380387365,
    0.071761951986224773,
    0.07322773534184536,
    0.074714388524983549,
    0.076222027065196873,
    0.077750763890223104,
    0.079300709272752581,
    0.080871970777638164,
    0.082464653209565628,
    0.08407885856120649,
    0.085714685961876036,
    0.087372231626718269,
    0.089051588806440457,
    0.090752847737618714,
    0.092476095593597041,
    0.094221416436001029,
    0.095988891166888232,
    0.097778597481556484,
    0.099590609822031259,
    0.10142499933125351,
    0.10328183380798857,
    0.10516117766247689,
    0.10706309187284728,
    0.1089876339423126,
    0.11093485785716829,
    0.11290481404561324,
    0.11489754933741281,
    0.11691310692442305,
    0.11895152632199545,
    0.12101284333128065,
    0.12309709000244995,
    0.12520429459885238,
    0.12733448156212568,
    0.12948767147827828,
    0.13166388104475998,
    0.1338631230385379,
    0.13608540628519442,
    0.13833073562906339,
    0.14059911190442054,
    0.14289053190774326,
    0.14520498837105547,
    0.14754246993637185,
    0.14990296113125601,
    0.15228644234550684,
    0.99981414433587668,
    0.99982602371407214,
    0.99983725731578443,
    0.99984787372750239,
    0.99985790058476165,
    0.99986736459144465,
    0.99987629153907442,
    0.99988470632609161,
    0.99989263297710196,
    0.99990009466208507,
    0.99990711371555163,
    0.99991371165563925,
    0.99991990920313467,
    0.99992572630041499,
    0.99993118213029331,
    0.9999362951347629,
    0.99994108303362705,
    0.99994556284300662,
    0.99994975089371452,
    0.99995366284948961,
    0.9999573137250799,
    0.99996071790416563,
    0.99996388915711587,
    0.99996684065856889,
    0.9999695850048278,
    0.99997213423106523,
    0.99997449982832909,
    0.99997669276034074,
    0.9999787234800811,
    0.99998060194615568,
    0.99998233763893341,
    0.99998393957645293,
    0.99998541633008942,
    0.99998677603997821,
    0.99998802643018825,
    0.99998917482364014,
    0.99999022815676608,
    0.99999119299390438,
    0.99999207554142622,
    0.9999928816615905,
    0.99999361688612176,
    0.99999428642950949,
    0.99999489520202478,
    0.99999544782245153,
    0.99999594863052965,
    0.99999640169910831,
    0.99999681084600733,
    0.99999717964558388,
    0.99999751144000515,
    0.99999780935022342,
    0.99999807628665482,
    0.99999831495955971,
    0.99999852788912524,
    0.99999871741524993,
    0.99999888570702955,
    0.99999903477194729,
    0.99999916646476517,
    0.99999928249612247,
    0.99999938444083747,
    0.99999947374591902,
    0.9999995517382857,
    0.99999961963219752,
    0.99999967853640048,
    0.9999997294609887,
    0.99999977332398493,
    0.9999998109576439,
    0.99999984311448165,
    0.99999987047303385,
    0.99999989364334774,
    0.99999991317221149,
    0.99999992954812422,
    0.99999994320601393,
    0.99999995453170409,
    0.99999996386613832,
    0.99999997150936371,
    0.99999997772428262,
    0.99999998274017465,
    0.99999998675599711,
    0.99999998994346873,
    0.9999999924499432,
    0.99999999440107812,
    0.99999999590330724,
    0.99999999704612041,
    0.99999999790416083,
    0.9999999985391439,
    0.99999999900160597,
    0.99999999933249095,
    0.99999999956458041,
    0.99999999972377562,
    0.99999999983023991,
    0.99999999989940713,
    0.99999999994286615,
    0.99999999996912725,
    0.99999999998428069,
    0.99999999999255362,
    0.99999999999677447,
    0.99999999999875311,
    0.999999999999585,
    0.99999999999988753,
    0.99999999999997746,
    0.99999999999999722,
    0.99999999999999989,
    1,
    8.1944977841692359e-25,
    3.9193473629997056e-18,
    3.3623478632757054e-16,
    5.0009718906043362e-15,
    3.4916945694391895e-14,
    1.595438753099194e-13,
    5.5553359501213202e-13,
    1.6010191994196156e-12,
    4.0141258631291412e-12,
    9.0442509581335297e-12,
    1.872461797767619e-11,
    3.6194558689264783e-11,
    6.6099920913046731e-11,
    1.1507993538956022e-10,
    1.9234880440217076e-10,
    3.1038019719765258e-10,
    4.857027528509321e-10,
    7.3981460009869533e-10,
    1.100224805877225e-09,
    1.6016295599024281e-09,
    2.2872307916885612e-09,
    3.2102047989847225e-09,
    4.4353283427867908e-09,
    6.0407695339331642e-09,
    8.1200506998953175e-09,
    1.0784190277812596e-08,
    1.4164030631182456e-08,
    1.8412758532993871e-08,
    2.3708624897577977e-08,
    3.0257870177145e-08,
    3.8297861666982385e-08,
    4.8100448785711968e-08,
    5.9975542213953045e-08,
    7.4274922586326678e-08,
    9.139628423808024e-08,
    1.1178751930883002e-07,
    1.3595124730213588e-07,
    1.6444959499096259e-07,
    1.9790923134567464e-07,
    2.3702666194318946e-07,
    2.8257378709341148e-07,
    3.3540372769222383e-07,
    3.9645692257926921e-07,
    4.6676750094364624e-07,
    5.4746993308163072e-07,
    6.3980596256775171e-07,
    7.4513182265416015e-07,
    8.6492573946337462e-07,
    1.0007957242863375e-06,
    1.154487657041412e-06,
    1.3278936626906685e-06,
    1.5230607821477013e-06,
    1.7421999389464488e-06,
    1.9876952026732377e-06,
    2.2621133498946764e-06,
    2.5682137230423052e-06,
    2.9089583874411584e-06,
    3.2875225863938892e-06,
    3.7073054939549089e-06,
    4.1719412647503742e-06,
    4.6853103799199188e-06,
    5.2515512879749983e-06,
    5.8750723390867654e-06,
    6.5605640110336419e-06,
    7.3130114247555145e-06,
    8.137707147177759e-06,
    9.0402642786844262e-06,
    1.0026629822336025e-05,
    1.1103098331643526e-05,
    1.227632583342683e-05,
    1.3553344022003022e-05,
    1.494157472066751e-05,
    1.6448844606149829e-05,
    1.8083400191445665e-05,
    1.9853923062147589e-05,
    2.1769545361119485e-05,
    2.3839865516083598e-05,
    2.6074964204415042e-05,
    2.8485420549166349e-05,
    3.1082328540074626e-05,
    3.3877313673036107e-05,
    3.6882549801267769e-05,
    4.0110776191113039e-05,
    4.3575314775189071e-05,
    4.7290087595316271e-05,
    5.1269634427417536e-05,
    5.5529130580324409e-05,
    6.0084404860180912e-05,
    6.4951957691893164e-05,
    7.0148979388833657e-05,
    7.5693368561774573e-05,
    8.1603750657793559e-05,
    8.7899496619669196e-05,
    9.4600741656061647e-05,
    0.00010172840411255681,
    0.00010930420443344012,
    0.00011735068420385871,
    0.00012589122526182874,
    0.00013495006886934756,
    0.00014455233493167917,
    0.00015472404125369541,
    0.00016549212282197508,
    0.00017688445110118926,
    0.00018892985333313178,
    0.00020165813182659287,
    0.00021510008322611645,
    0.00022928751774753327,
    0.00024425327836801722,
    0.00026003125995827738,
    0.00027665642834436629,
    0.00029416483928646349,
    0.00031259365736187599,
    0.00033198117473938829,
    0.00035236682983199273,
    0.00037379122581493554,
    0.00039629614899592587,
    0.00041992458702427615,
    0.00044472074692566694,
    0.00047073007294916693,
    0.00049799926421307769,
    0.0005265762921361259,
    0.00055651041764048054,
    0.00058785220811304062,
    0.00062065355411140952,
    0.00065496768580095596,
    0.00069084918910934851,
    0.00072835402158494928,
    0.00076753952794545669,
    0.00080846445530320036,
    0.00085118896805351442,
    0.00089577466241264113,
    0.00094228458059165921,
    0.00099078322459297406,
    0.0010413365696159596,
    0.0010940120770584103,
    0.0011488787071005221,
    0.0012060069308582112,
    0.0012654687420926558,
    0.0013273376684630491,
    0.0013916887823096484,
    0.0014585987109543238,
    0.0015281456465059213,
    0.0016004093551578891,
    0.0016754711859657471,
    0.0017534140790921274,
    0.0018343225735072616,
    0.001918282814132951,
    0.0020053825584182257,
    0.0020957111823350703,
    0.0021893596857827792,
    0.0022864206973896945,
    0.0023869884787012805,
    0.0024911589277436873,
    0.0025990295819521779,
    0.0027106996204540146,
    0.0028262698656956135,
    0.0029458427844040345,
    0.003069522487873099,
    0.0031974147315646812,
    0.0033296269140159827,
    0.0034662680750438542,
    0.0036074488932374992,
    0.003753281682731183,
    0.0039038803892488334,
    0.0040593605854127267,
    0.0042198394653087407,
    0.0043854358383009591,
    0.0045562701220887162,
    0.0047324643349994948,
    0.0049141420875113912,
    0.0051014285729992148,
    0.0052944505576985809,
    0.0054933363698827361,
    0.0056982158882471498,
    0.0059092205294972939,
    0.0061264832351353472,
    0.0063501384574419445,
    0.0065803221446494274,
    0.0068171717253034357,
    0.0070608260918100241,
    0.0073114255831658948,
    0.0075691119668696658,
    0.0078340284200125246,
    0.0081063195095469505,
    0.0083861311717326123,
    0.0086736106907589205,
    0.0089689066765440955,
    0.0092721690417110238,
    0.009583548977740582,
    0.0099031989303034421,
    0.010231272573771879,
    0.010567924784913392,
    0.010913311615768454,
    0.011267590265715005,
    0.011630919052722842,
    0.012003457383801321,
    0.012385365724644333,
    0.012776805568476803,
    0.013177939404107464,
    0.013588930683193029,
    0.014009943786719272,
    0.014441143990704994,
    0.014882697431135206,
    0.015334771068130295,
    0.015797532649358358,
    0.016271150672698267,
    0.016755794348161435,
    0.017251633559080701,
    0.017758838822575118,
    0.018277581249299766,
    0.018808032502490295,
    0.01935036475631207,
    0.019904750653524355,
    0.020471363262470284,
    0.021050376033403755,
    0.021641962754164835,
    0.022246297505215531,
    0.022863554614048261,
    0.023493908608979687,
    0.024137534172342958,
    0.024794606093091726,
    0.02546529921882976,
    0.026149788407280241,
    0.026848248477209211,
    0.027560854158818,
    0.028287780043619818,
    0.029029200533815969,
    0.029785289791187521,
    0.030556221685518629,
    0.031342169742567927,
    0.032143307091604768,
    0.032959806412527513,
    0.033791839882581075,
    0.034639579122691555,
    0.035503195143435917,
    0.03638285829066485,
    0.037278738190797474,
    0.038191003695806601,
    0.039119822827913611,
    0.040065362724012239,
    0.04102778957984083,
    0.042007268593922807,
    0.043003963911295393,
    0.044018038567046724,
    0.045049654429681903,
    0.046098972144338529,
    0.047166151075872534,
    0.048251349251835399,
    0.049354723305363869,
    0.050476428418003524,
    0.051616618262487764,
    0.052775444945493778,
    0.053953058950397405,
    0.055149609080048713,
    0.056365242399590364,
    0.057600104179340972,
    0.058854337837765669,
    0.060128084884556211,
    0.06142148486384312,
    0.062734675297562298,
    0.064067791628998672,
    0.065420967166529648,
    0.06679433302759076,
    0.068188018082886409,
    0.069602148900868369,
    0.071036849692504775,
    0.072492242256362235,
    0.073968445924023921,
    0.075465577505866202,
    0.076983751237216735,
    0.078523078724916151,
    0.080083668894306584,
    0.081665627936668928,
    0.083269059257131656,
    0.084894063423073499,
    0.086540738113042179,
    0.088209178066211313,
    0.089899475032397813,
    0.091611717722661481,
    0.093345991760508562,
    0.095102379633721243,
    0.096880960646834227,
    0.098681810874279963,
    0.10050500311422361,
    0.1023506068431088,
    0.10421868817093492,
    0.1061093097972865,
    0.10802253096813522,
    0.10995840743343448,
    0.11191699140552663,
    0.11389833151838256,
    0.11590247278769286,
    0.11792945657182996,
    0.11997932053370002,
    0.12205209860350327,
    0.12414782094242105,
    0.12626651390724758,
    0.1284082000159843,
    0.13057289791441409,
    0.13276062234367236,
    0.13497138410883217,
    0.13720519004851903,
    0.13946204300557249,
    0.14174194179876917,
    0.14404488119562325,
    0.14637085188627921,
    0.14871984045851122,
    0.15109182937384374,
    0.15348679694480674,
    0.15590471731333916,
    0.15834556043035367,
    0.16080929203647493,
    0.16329587364396403,
    0.16580526251984032,
    0.16833741167021218,
    0.17089226982582767,
    0.173469781428855,
    0.17606988662090331,
    0.17869252123229279,
    0.18133761677258339,
    0.1840051004223707,
    0.18669489502635658,
    0.18940691908770305,
    0.19214108676367547,
    0.19489730786258239,
    0.19767548784201794,
    0.20047552780841224,
    0.20329732451789514,
    0.20614077037847781,
    0.2090057534535564,
    0.2118921574667412,
    0.21479986180801441,
    0.21772874154121938,
    0.22067866741288281,
    0.22364950586237209,
    0.22664111903338829,
    0.22965336478679552,
    0.23268609671478657,
    0.23573916415638424,
    0.23881241221427743,
    0.24190568177299007,
    0.24501880951838109,
    0.24815162795847243,
    0.25130396544560224,
    0.25447564619989932,
    0.25766649033407429,
    0.26087631387952365,
    0.26410492881373998,
    0.267352143089024,
    0.27061776066249088,
    0.27390158152736488,
    0.27720340174555402,
    0.28052301348149772,
    0.28386020503727855,
    0.28721476088898934,
    0.29058646172434566,
    0.29397508448153492,
    0.2973804023892902,
    0.30080218500817896,
    0.3042401982730949,
    0.30769420453694063,
    0.31116396261548918,
    0.31464922783341159,
    0.31814975207145652,
    0.32166528381476883,
    0.32519556820233247,
    0.32874034707752253,
    0.33229935903975211,
    0.33587233949719741,
    0.33945902072058576,
    0.34305913189802917,
    0.34667239919088721,
    0.35029854579064151,
    0.35393729197676355,
    0.35758835517555854,
    0.36125145001996584,
    0.36492628841029717,
    0.36861257957589372,
    0.37231003013768105,
    0.37601834417160329,
    0.3797372232729147,
    0.38346636662130834,
    0.38720547104686059,
    0.39095423109676974,
    0.39471233910286707,
    0.39847948524987747,
    0.40225535764440806,
    0.40603964238464135,
    0.40983202363070936,
    0.41363218367572663,
    0.41743980301745692,
    0.42125456043059084,
    0.42507613303960901,
    0.42890419639220756,
    0.43273842453326039,
    0.43657849007929339,
    0.44042406429344594,
    0.44427481716089356,
    0.44813041746470711,
    0.45199053286212221,
    0.45585482996119275,
    0.45972297439780363,
    0.46359463091301467,
    0.4674694634307115,
    0.47134713513553506,
    0.47522730855106438,
    0.47910964561822528,
    0.48299380777389861,
    0.48687945602970073,
    0.49076625105091004,
    0.49465385323551142,
    0.49854192279333281,
    0.50243011982524566,
    0.50631810440240321,
    0.5102055366454884,
    0.51409207680394553,
    0.5179773853351668,
    0.52186112298360887,
    0.52574295085981027,
    0.52962253051928343,
    0.53349952404125534,
    0.53737359410722818,
    0.54124440407933383,
    0.54511161807845587,
    0.54897490106209235,
    0.55283391890193179,
    0.5566883384611172,
    0.56053782767117133,
    0.56438205560855648,
    0.56822069257084395,
    0.57205341015246647,
    0.57587988132002799,
    0.57969978048714532,
    0.58351278358879755,
    0.58731856815515571,
    0.59111681338486999,
    0.59490720021778942,
    0.59868941140708853,
    0.60246313159077791,
    0.60622804736257485,
    0.60998384734210931,
    0.61373022224444318,
    0.61746686494887892,
    0.62119347056703489,
    0.62490973651016568,
    0.62861536255570394,
    0.63231005091300263,
    0.63599350628825724,
    0.63966543594858427,
    0.6433255497852377,
    0.64697356037594145,
    0.65060918304631832,
    0.65423213593039586,
    0.65784214003016939,
    0.66143891927420329,
    0.66502220057525252,
    0.66859171388688576,
    0.67214719225909236,
    0.67568837189285635,
    0.67921499219368031,
    0.68272679582404217,
    0.68622352875476966,
    0.68970494031531726,
    0.69317078324292847,
    0.69662081373067164,
    0.70005479147433225,
    0.70347247971815097,
    0.70687364529939067,
    0.71025805869172376,
    0.71362549404742315,
    0.71697572923834907,
    0.72030854589571747,
    0.72362372944864084,
    0.72692106916143051,
    0.73020035816965057,
    0.73346139351491435,
    0.73670397617841488,
    0.73992791111318001,
    0.74313300727504594,
    0.74631907765234018,
    0.74948593929426877,
    0.75263341333800104,
    0.75576132503444526,
    0.75886950377271123,
    0.76195778310325513,
    0.76502600075970129,
    0.76807399867933868,
    0.77110162302228846,
    0.77410872418933951,
    0.77709515683845143,
    0.78006077989992184,
    0.7830054565902177,
    0.78592905442447092,
    0.78883144522763748,
    0.79171250514432023,
    0.79457211464725852,
    0.7974101585444836,
    0.80022652598514465,
    0.80302111046400559,
    0.80579380982461846,
    0.80854452626117479,
    0.81127316631904056,
    0.81397964089397945,
    0.8166638652300684,
    0.81932575891631354,
    0.82196524588196962,
    0.82458225439057287,
    0.82717671703269213,
    0.82974857071740704,
    0.83229775666252137,
    0.83482422038352,
    0.83732791168127862,
    0.83980878462853592,
    0.84226679755513778,
    0.84470191303206577,
    0.84711409785425762,
    0.8495033230222343,
    0.85186956372254397,
    0.85421279930703486,
    0.85653301327097098,
    0.85883019323000154,
    0.86110433089600091,
    0.86335542205178928,
    0.86558346652475138,
    0.86778846815936572,
    0.86997043478866165,
    0.87212937820461667,
    0.87426531412751329,
    0.87637826217426829,
    0.87846824582575367,
    0.88053529239312378,
    0.88257943298316777,
    0.88460070246270306,
    0.8865991394220295,
    0.88857478613746055,
    0.89052768853295083,
    0.89245789614083937,
    0.8943654620617254,
    0.89625044292349809,
    0.89811289883953938,
    0.89995289336611772,
    0.90177049345899551,
    0.90356576942926858,
    0.90533879489845814,
    0.90708964675287784,
    0.90881840509729295,
    0.91052515320789795,
    0.91220997748462818,
    0.9138729674028313,
    0.91551421546431788,
    0.91713381714781317,
    0.91873187085883246,
    0.92030847787900127,
    0.92186374231484314,
    0.92339777104605669,
    0.92491067367330426,
    0.9264025624655351,
    0.92787355230686519,
    0.92932376064303535,
    0.93075330742747264,
    0.93216231506697456,
    0.93355090836704124,
    0.93491921447687731,
    0.93626736283408607,
    0.93759548510907853,
    0.93890371514922133,
    0.94019218892274437,
    0.94146104446243239,
    0.9427104218091219,
    0.94394046295502743,
    0.94515131178691847,
    0.94634311402916982,
    0.94751601718670875,
    0.94867017048787938,
    0.94980572482724901,
    0.95092283270837596,
    0.95202164818656276,
    0.95310232681161577,
    0.95416502557063365,
    0.95520990283084495,
    0.95623711828251801,
    0.95724683288196322,
    0.95823920879464963,
    0.9592144093384557,
    0.9601725989270764,
    0.96111394301360642,
    0.96203860803431984,
    0.96294676135266666,
    0.96383857120350613,
    0.96471420663759644,
    0.9655738374663605,
    0.96641763420694671,
    0.96724576802760398,
    0.96805841069338938,
    0.96885573451222773,
    0.96963791228134,
    0.97040511723405953,
    0.97115752298705316,
    0.97189530348796438,
    0.97261863296349682,
    0.97332768586795237,
    0.9740226368322431,
    0.97470366061339064,
    0.97537093204453063,
    0.97602462598543605,
    0.97666491727357629,
    0.97729198067572509,
    0.97790599084013197,
    0.9785071222492725,
    0.97909554917318953,
    0.97967144562343955,
    0.98023498530765663,
    0.9807863415847472,
    0.98132568742072701,
    0.98185319534521298,
    0.98236903740858039,
    0.98287338513979761,
    0.9833664095049478,
    0.98384828086644915,
    0.98431916894298255,
    0.98477924277013629,
    0.98522867066177755,
    0.98566762017215859,
    0.98609625805876677,
    0.98651475024592494,
    0.98692326178915113,
    0.9873219568402839,
    0.98771099861337974,
    0.98809054935138985,
    0.98846077029362045,
    0.9888218216439838,
    0.98917386254004391,
    0.9895170510228618,
    0.98985154400764419,
    0.99017749725520054,
    0.99049506534421061,
    0.99080440164430539,
    0.99110565828996566,
    0.99139898615523836,
    0.99168453482927388,
    0.99196245259268434,
    0.99223288639472607,
    0.99249598183130394,
    0.99275188312379969,
    0.99300073309872428,
    0.99324267316819148,
    0.99347784331121392,
    0.99370638205581985,
    0.9939284264619872,
    0.99414411210539455,
    0.99435357306198646,
    0.99455694189334853,
    0.99475434963289122,
    0.99494592577283669,
    0.99513179825200693,
    0.99531209344440619,
    0.99548693614859507,
    0.99565644957785138,
    0.99582075535110992,
    0.99597997348467904,
    0.99613422238472438,
    0.99628361884051631,
    0.99642827801843281,
    0.99656831345671149,
    0.99670383706094234,
    0.99683495910029607,
    0.99696178820447656,
    0.99708443136139291,
    0.99720299391554024,
    0.997317579567081,
    0.99742829037161895,
    0.9975352267406542,
    0.99763848744271277,
    0.99773816960513817,
    0.99783436871653697,
    0.99792717862986668,
    0.99801669156615647,
    0.99810299811884995,
    0.99818618725875841,
    0.99826634633961431,
    0.99834356110421241,
    0.99841791569112948,
    0.99848949264200804,
    0.99855837290939398,
    0.99862463586511618,
    0.99868835930919519,
    0.99874961947926866,
    0.99880849106052172,
    0.99886504719610958,
    0.9989193594980591,
    0.99897149805863705,
    0.99902153146217154,
    0.99906952679731442,
    0.99911554966973104,
    0.99915966421520352,
    0.99920193311313599,
    0.99924241760044619,
    0.9992811774858319,
    0.99931827116439742,
    0.99935375563262796,
    0.9993876865036968,
    0.9994201180230925,
    0.99945110308455265,
    0.99948069324629041,
    0.99950893874750024,
    0.99953588852512854,
    0.99956159023089708,
    0.99958609024856404,
    0.99960943371141031,
    0.99963166451993668,
    0.9996528253597593,
    0.99967295771968911,
    0.99969210190998259,
    0.9997102970807501,
    0.99972758124050898,
    0.99974399127486768,
    0.99975956296532853,
    0.99977433100819602,
    0.99978832903357751,
    0.99980158962446375
};

// Precomputed weights
static const double weights[] = {
    0,
    9.9001978756848123e-15,
    6.3358496678809455e-13,
    7.2164029758348085e-12,
    4.0542348100157951e-11,
    1.5463650332915445e-10,
    4.6166811817952852e-10,
    1.1639339904962804e-09,
    2.5928960365412975e-09,
    5.2552411622711736e-09,
    9.8859257972273447e-09,
    1.7508161724025829e-08,
    2.9500294134551125e-08,
    4.7669517172124736e-08,
    7.4332366580612006e-08,
    1.1240192348011164e-07,
    1.6548165772797982e-07,
    2.3796583380689135e-07,
    3.3514639671182794e-07,
    4.6332624988867166e-07,
    6.2993883191180983e-07,
    8.436738932801332e-07,
    1.1146093694633064e-06,
    1.4543492411464503e-06,
    1.876167267504613e-06,
    2.3951564732917668e-06,
    3.0283842655557202e-06,
    3.7950530508933246e-06,
    4.7166662193427456e-06,
    5.817199356274391e-06,
    7.1232765389922308e-06,
    8.6643515701956823e-06,
    1.0472893995981784e-05,
    1.2584579751690872e-05,
    1.5038486274619134e-05,
    1.7877291918440994e-05,
    2.1147479500105833e-05,
    2.4899543805999817e-05,
    2.9188202880296922e-05,
    3.4072612914666179e-05,
    3.9616586554857326e-05,
    4.5888814436156128e-05,
    5.2963089756287046e-05,
    6.0918535691045371e-05,
    6.983983545476721e-05,
    7.9817464804694501e-05,
    9.0947926785366328e-05,
    0.00010333398850636909,
    0.00011708491974410797,
    0.00013231673315572292,
    0.00014915242589086584,
    0.00016772222238478247,
    0.00018816381811400602,
    0.00021062262409396893,
    0.00023525201189597886,
    0.00026221355895928231,
    0.0002916772939723604,
    0.0003238219420961626,
    0.00035883516980068968,
    0.00039691382908518851,
    0.00043826420085121495,
    0.00048310223719696493,
    0.00053165380240056113,
    0.00058415491235941992,
    0.00064085197225240901,
    0.00070200201219123778,
    0.00076787292062740933,
    0.0008387436752810923,
    0.00091490457135845547,
    0.00099665744682434182,
    0.0010843159044976397,
    0.0011782055307373446,
    0.0012786641104880864,
    0.0013860418384548361,
    0.0015007015261775786,
    0.0016230188047779864,
    0.0017533823231514949,
    0.0018921939413797271,
    0.0020398689191398773,
    0.002196836098889504,
    0.0023635380836071443,
    0.0025404314088712894,
    0.0027279867090625113,
    0.0029266888774759536,
    0.0031370372201339329,
    0.0033595456030911016,
    0.0035947425930274446,
    0.0038431715909273594,
    0.0041053909586461716,
    0.0043819741381686877,
    0.0046735097633677581,
    0.0049806017640743423,
    0.0053038694622742086,
    0.0056439476602501608,
    0.0060014867204926121,
    0.0063771526372053133,
    0.0067716270992372223,
    0.0071856075442757539,
    0.007619807204141054,
    0.0080749551410254135,
    0.0085517962745266102,
    0.0090510913993286524,
    0.0095736171933882615,
    0.010120166216490385,
    0.010691546899041069,
    0.011288583520971203,
    0.011912116180629876,
    0.012563000753551441,
    0.01324210884098587,
    0.013950327708087464,
    0.014688560211662637,
    0.015457724717383221,
    0.016258755006377482,
    0.017092600171116964,
    0.017960224500523186,
    0.018862607354224268,
    0.01980074302589762,
    0.020775640595640996,
    0.021788323771320492,
    0.022839830718850179,
    0.023931213881364662,
    0.02506353978725194,
    0.026237888847020645,
    0.027455355138982035,
    0.028717046183733733,
    0.03002408270743874,
    0.031377598393899871,
    0.032778739625436329,
    0.034228665212575872,
    0.035728546112582658,
    0.037279565136847541,
    0.038882916647174394,
    0.040539806241002618,
    0.04225145042561293,
    0.044019076281370063,
    0.045843921114062963,
    0.047727232096409569,
    0.049670265898800338,
    0.051674288309361015,
    0.053740573843422207,
    0.055870405342489833,
    0.058065073562817184,
    0.060325876753686176,
    0.06265412022551177,
    0.0650511159078902,
    0.067518181897718424,
    0.070056641997518349,
    0.072667825244106116,
    0.075353065427753149,
    0.078113700601992006,
    0.080951072584226319,
    0.083866526447310463,
    0.086861410002270809,
    0.089937073272346596,
    0.093094867958534264,
    0.096336146896825556,
    0.099662263507335008,
    0.10307457123551904,
    0.10657442298569381,
    0.11016317054706536,
    0.11384216401249074,
    0.11761275119019446,
    0.12147627700866999,
    0.12543408291500127,
    0.1294875062668446,
    0.13363787971831592,
    0.13788653060003403,
    0.14223478029357486,
    0.1466839436005965,
    0.15123532810689966,
    0.15589023354169279,
    0.16064995113233499,
    0.16551576295483464,
    0.17048894128038561,
    0.17557074791822688,
    0.18076243355511518,
    0.18606523709170458,
    0.19148038497612949,
    0.19700909053509225,
    0.20265255330275866,
    0.20841195834776882,
    0.21428847559867334,
    0.22028325916810743,
    0.22639744667601938,
    0.23263215857227057,
    0.23898849745892869,
    0.24546754741257643,
    0.25207037330696069,
    0.25879802013630981,
    0.26565151233964657,
    0.27263185312642835,
    0.27974002380384544,
    0.28697698310611092,
    0.29434366652607635,
    0.30184098564950801,
    0.30946982749236007,
    0.31723105384138112,
    0.32512550059839063,
    0.33315397712856293,
    0.34131726561305614,
    0.34961612040632278,
    0.35805126739843984,
    0.36662340338279376,
    0.37533319542945742,
    0.38418128026459397,
    0.3931682636562211,
    0.40229471980666975,
    0.41156119075206909,
    0.42096818576918693,
    0.43051618078995624,
    0.44020561782401257,
    0.45003690438956939,
    0.46001041295295203,
    0.47012648037711235,
    0.48038540737944124,
    0.49078745799919421,
    0.50133285907484249,
    0.51202179973165984,
    0.5228544308798494,
    0.53383086472351582,
    0.5449511742807801,
    0.55621539291533384,
    0.56762351387972421,
    0.57917548987065848,
    0.59087123259660979,
    0.60271061235800616,
    0.61469345764027539,
    0.62681955472001749,
    0.63908864728456927,
    0.65150043606522312,
    0.66405457848435301,
    0.67675068831670082,
    0.68958833536506592,
    0.70256704515063828,
    0.71568629861820876,
    0.72894553185648436,
    0.74234413583373104,
    0.75588145614896018,
    0.76955679279886724,
    0.78336939996072796,
    0.79731848579144893,
    0.81140321224296186,
    0.82562269489414719,
    0.83997600279946294,
    0.85446215835445072,
    0.86908013717827992,
    0.88382886801348881,
    0.89870723264307006,
    0.91371406582504255,
    0.92884815524464492,
    0.94410824148427641,
    0.95949301801130515,
    0.97500113118385578,
    0.99063118027467978,
    1.0063817175132046,
    1.0222512481458508,
    1.0382382305146973,
    1.054341076154566,
    1.0705581499085906,
    1.0868877700623274,
    1.1033282084964544,
    1.119877690858097,
    1.1365343967508155,
    1.1532964599432742,
    1.1701619685966098,
    1.1871289655105015,
    1.2041954483879458,
    1.2213593701187202,
    1.2386186390815221,
    1.2559711194647512,
    1.2734146316059014,
    1.2909469523495185,
    1.3085658154236683,
    1.3262689118348567,
    1.3440538902813286,
    1.3619183575846698,
    1.3798598791396244,
    1.3978759793820288,
    1.4159641422747664,
    1.4341218118116197,
    1.4523463925389097,
    1.4706352500947861,
    1.4889857117660339,
    1.5073950670622513,
    1.5258605683072421,
    1.5443794312474637,
    1.5629488356773573,
    1.5815659260813861,
    1.6002278122925884,
    1.6189315701674583,
    1.6376742422769472,
    1.6564528386133781,
    1.6752643373130529,
    1.6941056853943293,
    1.712973799510932,
    1.7318655667202616,
    1.7507778452664469,
    1.769707465377891,
    1.7886512300790465,
    1.8076059160161499,
    1.8265682742966389,
    1.8455350313419703,
    1.8645028897535458,
    1.8834685291914526,
    1.9024286072657111,
    1.9213797604397236,
    1.940318604945606,
    1.9592417377110809,
    1.9781457372976037,
    1.9970271648493885,
    2.0158825650529928,
    2.0347084671071167,
    2.0535013857022659,
    2.0722578220099197,
    2.0909742646808493,
    2.1096471908522121,
    2.1282730671630561,
    2.1468483507778604,
    2.1653694904177243,
    2.1838329273988286,
    2.2022350966777764,
    2.22057242790342,
    2.2388413464747798,
    2.2570382746046564,
    2.2751596323885246,
    2.2932018388783142,
    2.3111613131606554,
    2.329034475439189,
    2.3468177481205088,
    2.3645075569033387,
    2.3821003318705039,
    2.3995925085832863,
    2.416980529177736,
    2.4342608434625093,
    2.4514299100178096,
    2.4684841972949969,
    2.4854201847164377,
    2.5022343637751652,
    2.51892323913391,
    2.5354833297230779,
    2.5519111698372297,
    2.5682033102296398,
    2.5843563192044883,
    2.6003667837062641,
    2.6162313104059334,
    2.6319465267834503,
    2.6475090822061715,
    2.6629156490027426,
    2.6781629235320299,
    2.6932476272466652,
    2.7081665077507759,
    2.7229163398514777,
    2.7374939266036988,
    2.7518961003479223,
    2.7661197237404198,
    2.7801616907755649,
    2.794018927799804,
    2.8076883945168838,
    2.8211670849839141,
    2.8344520285978727,
    2.8475402910721419,
    2.8604289754026784,
    2.8731152228234298,
    2.8855962137505977,
    2.8978691687153657,
    2.9099313492847099,
    2.921780058969913,
    2.9334126441224093,
    2.9448264948165952,
    2.9560190457192355,
    2.966987776945115,
    2.9777302148985791,
    2.9882439331006165,
    2.998526553001144,
    3.0085757447761545,
    3.0183892281094056,
    3.0279647729583172,
    3.0373002003037675,
    3.0463933828834731,
    3.0552422459086515,
    3.0638447677636673,
    3.0721989806883729,
    3.0803029714428578,
    3.0881548819543356,
    3.0957529099458929,
    3.1030953095468443,
    3.1101803918844384,
    3.1170065256566626,
    3.1235721376859185,
    3.1298757134533268,
    3.1359157976134489,
    3.1416909944891986,
    3.1471999685467495,
    3.1524414448502314,
    3.1574142094960305,
    3.1621171100265091,
    3.1665490558229727,
    3.1707090184777234,
    3.1745960321450366,
    3.1782091938709223,
    3.1815476639015294,
    3.1846106659700637,
    3.1873974875620976,
    3.1899074801591665,
    3.1921400594605447,
    3.1940947055831077,
    3.1957709632392013,
    3.1971684418924409,
    3.1982868158913704,
    3.1991258245809382,
    3.1996852723917262,
    3.199965028906909,
    3.199965028906909,
    3.1996852723917262,
    3.1991258245809382,
    3.1982868158913704,
    3.1971684418924409,
    3.1957709632392013,
    3.1940947055831077,
    3.1921400594605447,
    3.1899074801591665,
    3.1873974875620976,
    3.1846106659700637,
    3.1815476639015294,
    3.1782091938709223,
    3.1745960321450366,
    3.1707090184777234,
    3.1665490558229727,
    3.1621171100265091,
    3.1574142094960305,
    3.1524414448502314,
    3.1471999685467495,
    3.1416909944891986,
    3.1359157976134489,
    3.1298757134533268,
    3.1235721376859185,
    3.1170065256566626,
    3.1101803918844384,
    3.1030953095468443,
    3.0957529099458929,
    3.0881548819543356,
    3.0803029714428578,
    3.0721989806883729,
    3.0638447677636673,
    3.0552422459086515,
    3.0463933828834731,
    3.0373002003037675,
    3.0279647729583172,
    3.0183892281094056,
    3.0085757447761545,
    2.998526553001144,
    2.9882439331006165,
    2.9777302148985791,
    2.966987776945115,
    2.9560190457192355,
    2.9448264948165952,
    2.9334126441224093,
    2.921780058969913,
    2.9099313492847099,
    2.8978691687153657,
    2.8855962137505977,
    2.8731152228234298,
    2.8604289754026784,
    2.8475402910721419,
    2.8344520285978727,
    2.8211670849839141,
    2.8076883945168838,
    2.794018927799804,
    2.7801616907755649,
    2.7661197237404198,
    2.7518961003479223,
    2.7374939266036988,
    2.7229163398514777,
    2.7081665077507759,
    2.6932476272466652,
    2.6781629235320299,
    2.6629156490027426,
    2.6475090822061715,
    2.6319465267834503,
    2.6162313104059334,
    2.6003667837062641,
    2.5843563192044883,
    2.5682033102296398,
    2.5519111698372297,
    2.5354833297230779,
    2.51892323913391,
    2.5022343637751652,
    2.4854201847164377,
    2.4684841972949969,
    2.4514299100178096,
    2.4342608434625093,
    2.416980529177736,
    2.3995925085832863,
    2.3821003318705039,
    2.3645075569033387,
    2.3468177481205088,
    2.329034475439189,
    2.3111613131606554,
    2.2932018388783142,
    2.2751596323885246,
    2.2570382746046564,
    2.2388413464747798,
    2.22057242790342,
    2.2022350966777764,
    2.1838329273988286,
    2.1653694904177243,
    2.1468483507778604,
    2.1282730671630561,
    2.1096471908522121,
    2.0909742646808493,
    2.0722578220099197,
    2.0535013857022659,
    2.0347084671071167,
    2.0158825650529928,
    1.9970271648493885,
    1.9781457372976037,
    1.9592417377110809,
    1.940318604945606,
    1.9213797604397236,
    1.9024286072657111,
    1.8834685291914526,
    1.8645028897535458,
    1.8455350313419703,
    1.8265682742966389,
    1.8076059160161499,
    1.7886512300790465,
    1.769707465377891,
    1.7507778452664469,
    1.7318655667202616,
    1.712973799510932,
    1.6941056853943293,
    1.6752643373130529,
    1.6564528386133781,
    1.6376742422769472,
    1.6189315701674583,
    1.6002278122925884,
    1.5815659260813861,
    1.5629488356773573,
    1.5443794312474637,
    1.5258605683072421,
    1.5073950670622513,
    1.4889857117660339,
    1.4706352500947861,
    1.4523463925389097,
    1.4341218118116197,
    1.4159641422747664,
    1.3978759793820288,
    1.3798598791396244,
    1.3619183575846698,
    1.3440538902813286,
    1.3262689118348567,
    1.3085658154236683,
    1.2909469523495185,
    1.2734146316059014,
    1.2559711194647512,
    1.2386186390815221,
    1.2213593701187202,
    1.2041954483879458,
    1.1871289655105015,
    1.1701619685966098,
    1.1532964599432742,
    1.1365343967508155,
    1.119877690858097,
    1.1033282084964544,
    1.0868877700623274,
    1.0705581499085906,
    1.054341076154566,
    1.0382382305146973,
    1.0222512481458508,
    1.0063817175132046,
    0.99063118027467978,
    0.97500113118385578,
    0.95949301801130515,
    0.94410824148427641,
    0.92884815524464492,
    0.91371406582504255,
    0.89870723264307006,
    0.88382886801348881,
    0.86908013717827992,
    0.85446215835445072,
    0.83997600279946294,
    0.82562269489414719,
    0.81140321224296186,
    0.79731848579144893,
    0.78336939996072796,
    0.76955679279886724,
    0.75588145614896018,
    0.74234413583373104,
    0.72894553185648436,
    0.71568629861820876,
    0.70256704515063828,
    0.68958833536506592,
    0.67675068831670082,
    0.66405457848435301,
    0.65150043606522312,
    0.63908864728456927,
    0.62681955472001749,
    0.61469345764027539,
    0.60271061235800616,
    0.59087123259660979,
    0.57917548987065848,
    0.56762351387972421,
    0.55621539291533384,
    0.5449511742807801,
    0.53383086472351582,
    0.5228544308798494,
    0.51202179973165984,
    0.50133285907484249,
    0.49078745799919421,
    0.48038540737944124,
    0.47012648037711235,
    0.46001041295295203,
    0.45003690438956939,
    0.44020561782401257,
    0.43051618078995624,
    0.42096818576918693,
    0.41156119075206909,
    0.40229471980666975,
    0.3931682636562211,
    0.38418128026459397,
    0.37533319542945742,
    0.36662340338279376,
    0.35805126739843984,
    0.34961612040632278,
    0.34131726561305614,
    0.33315397712856293,
    0.32512550059839063,
    0.31723105384138112,
    0.30946982749236007,
    0.30184098564950801,
    0.29434366652607635,
    0.28697698310611092,
    0.27974002380384544,
    0.27263185312642835,
    0.26565151233964657,
    0.25879802013630981,
    0.25207037330696069,
    0.24546754741257643,
    0.23898849745892869,
    0.23263215857227057,
    0.22639744667601938,
    0.22028325916810743,
    0.21428847559867334,
    0.20841195834776882,
    0.20265255330275866,
    0.19700909053509225,
    0.19148038497612949,
    0.18606523709170458,
    0.18076243355511518,
    0.17557074791822688,
    0.17048894128038561,
    0.16551576295483464,
    0.16064995113233499,
    0.15589023354169279,
    0.15123532810689966,
    0.1466839436005965,
    0.14223478029357486,
    0.13788653060003403,
    0.13363787971831592,
    0.1294875062668446,
    0.12543408291500127,
    0.12147627700866999,
    0.11761275119019446,
    0.11384216401249074,
    0.11016317054706536,
    0.10657442298569381,
    0.10307457123551904,
    0.099662263507335008,
    0.096336146896825556,
    0.093094867958534264,
    0.089937073272346596,
    0.086861410002270809,
    0.083866526447310463,
    0.080951072584226319,
    0.078113700601992006,
    0.075353065427753149,
    0.072667825244106116,
    0.070056641997518349,
    0.067518181897718424,
    0.0650511159078902,
    0.06265412022551177,
    0.060325876753686176,
    0.058065073562817184,
    0.055870405342489833,
    0.053740573843422207,
    0.051674288309361015,
    0.049670265898800338,
    0.047727232096409569,
    0.045843921114062963,
    0.044019076281370063,
    0.04225145042561293,
    0.040539806241002618,
    0.038882916647174394,
    0.037279565136847541,
    0.035728546112582658,
    0.034228665212575872,
    0.032778739625436329,
    0.031377598393899871,
    0.03002408270743874,
    0.028717046183733733,
    0.027455355138982035,
    0.026237888847020645,
    0.02506353978725194,
    0.023931213881364662,
    0.022839830718850179,
    0.021788323771320492,
    0.020775640595640996,
    0.01980074302589762,
    0.018862607354224268,
    0.017960224500523186,
    0.017092600171116964,
    0.016258755006377482,
    0.015457724717383221,
    0.014688560211662637,
    0.013950327708087464,
    0.01324210884098587,
    0.012563000753551441,
    0.011912116180629876,
    0.011288583520971203,
    0.010691546899041069,
    0.010120166216490385,
    0.0095736171933882615,
    0.0090510913993286524,
    0.0085517962745266102,
    0.0080749551410254135,
    0.007619807204141054,
    0.0071856075442757539,
    0.0067716270992372223,
    0.0063771526372053133,
    0.0060014867204926121,
    0.0056439476602501608,
    0.0053038694622742086,
    0.0049806017640743423,
    0.0046735097633677581,
    0.0043819741381686877,
    0.0041053909586461716,
    0.0038431715909273594,
    0.0035947425930274446,
    0.0033595456030911016,
    0.0031370372201339329,
    0.0029266888774759536,
    0.0027279867090625113,
    0.0025404314088712894,
    0.0023635380836071443,
    0.002196836098889504,
    0.0020398689191398773,
    0.0018921939413797271,
    0.0017533823231514949,
    0.0016230188047779864,
    0.0015007015261775786,
    0.0013860418384548361,
    0.0012786641104880864,
    0.0011782055307373446,
    0.0010843159044976397,
    0.00099665744682434182,
    0.00091490457135845547,
    0.0008387436752810923,
    0.00076787292062740933,
    0.00070200201219123778,
    0.00064085197225240901,
    0.00058415491235941992,
    0.00053165380240056113,
    0.00048310223719696493,
    0.00043826420085121495,
    0.00039691382908518851,
    0.00035883516980068968,
    0.0003238219420961626,
    0.0002916772939723604,
    0.00026221355895928231,
    0.00023525201189597886,
    0.00021062262409396893,
    0.00018816381811400602,
    0.00016772222238478247,
    0.00014915242589086584,
    0.00013231673315572292,
    0.00011708491974410797,
    0.00010333398850636909,
    9.0947926785366328e-05,
    7.9817464804694501e-05,
    6.983983545476721e-05,
    6.0918535691045371e-05,
    5.2963089756287046e-05,
    4.5888814436156128e-05,
    3.9616586554857326e-05,
    3.4072612914666179e-05,
    2.9188202880296922e-05,
    2.4899543805999817e-05,
    2.1147479500105833e-05,
    1.7877291918440994e-05,
    1.5038486274619134e-05,
    1.2584579751690872e-05,
    1.0472893995981784e-05,
    8.6643515701956823e-06,
    7.1232765389922308e-06,
    5.817199356274391e-06,
    4.7166662193427456e-06,
    3.7950530508933246e-06,
    3.0283842655557202e-06,
    2.3951564732917668e-06,
    1.876167267504613e-06,
    1.4543492411464503e-06,
    1.1146093694633064e-06,
    8.436738932801332e-07,
    6.2993883191180983e-07,
    4.6332624988867166e-07,
    3.3514639671182794e-07,
    2.3796583380689135e-07,
    1.6548165772797982e-07,
    1.1240192348011164e-07,
    7.4332366580612006e-08,
    4.7669517172124736e-08,
    2.9500294134551125e-08,
    1.7508161724025829e-08,
    9.8859257972273447e-09,
    5.2552411622711736e-09,
    2.5928960365412975e-09,
    1.1639339904962804e-09,
    4.6166811817952852e-10,
    1.5463650332915445e-10,
    4.0542348100157951e-11,
    7.2164029758348085e-12,
    6.3358496678809455e-13,
    9.9001978756848123e-15,
    3.2000000000000002,
    3.1998601176659229,
    3.1994405032748774,
    3.1987412546492515,
    3.1977625347891427,
    3.1965045718165492,
    3.1949676588972697,
    3.1931521541405461,
    3.1910584804764919,
    3.1886871255113505,
    3.1860386413606583,
    3.1831136444603643,
    3.179912815356003,
    3.1764368984699942,
    3.1726867018471778,
    3.1686630968786846,
    3.1643670180042602,
    3.159799462393166,
    3.1549614896037905,
    3.1498542212221174,
    3.144478840479195,
    3.1388365918477699,
    3.1329287806182573,
    3.1267567724542178,
    3.1203219929275332,
    3.1136259270334694,
    3.1066701186858334,
    3.0994561701924357,
    3.0919857417110688,
    3.0842605506862384,
    3.076282371266879,
    3.06805303370529,
    3.0595744237375535,
    3.0508484819456823,
    3.0418772031017678,
    3.0326626354943995,
    3.0232068802376366,
    3.0135120905628163,
    3.0035804710934952,
    2.9934142771038235,
    2.9830158137606571,
    2.9723874353497255,
    2.9615315444861738,
    2.9504505913098042,
    2.9391470726653472,
    2.9276235312681114,
    2.9158825548553402,
    2.9039267753236411,
    2.8917588678528299,
    2.8793815500165576,
    2.8667975808800863,
    2.8540097600855789,
    2.8410209269252813,
    2.8278339594029807,
    2.8144517732841154,
    2.8008773211349345,
    2.7871135913510936,
    2.7731636071760839,
    2.7590304257098994,
    2.7447171369083394,
    2.730226862573359,
    2.7155627553348749,
    2.7007279976244392,
    2.6857258006412006,
    2.6705594033105671,
    2.655232071235996,
    2.6397470956443274,
    2.6241077923250948,
    2.6083175005642341,
    2.5923795820726188,
    2.5762974199098561,
    2.5600744174037717,
    2.5437139970660141,
    2.5272195995042157,
    2.5105946823311411,
    2.493842719071258,
    2.4769671980651609,
    2.4599716213722913,
    2.4428595036723775,
    2.4256343711660362,
    2.4082997604749621,
    2.3908592175421446,
    2.3733162965325394,
    2.3556745587346248,
    2.3379375714632733,
    2.320108906964367,
    2.3021921413215773,
    2.2841908533657427,
    2.266108623587253,
    2.2479490330518721,
    2.2297156623204097,
    2.2114120903726544,
    2.1930418935359874,
    2.1746086444190769,
    2.1561159108510632,
    2.1375672548266378,
    2.1189662314574065,
    2.1003163879299431,
    2.0816212624709127,
    2.0628843833196622,
    2.0441092677086488,
    2.0252994208520936,
    2.0064583349432308,
    1.9875894881605178,
    1.9686963436831757,
    1.949782348716413,
    1.9308509335266915,
    1.9119055104873766,
    1.8929494731351206,
    1.8739861952373116,
    1.855019029870919,
    1.836051308513065,
    1.8170863401436368,
    1.7981274103602536,
    1.7791777805058964,
    1.7602406868094997,
    1.7413193395397977,
    1.722416922172713,
    1.7035365905725683,
    1.6846814721873931,
    1.6658546652585899,
    1.6470592380452249,
    1.6282982280631855,
    1.6095746413394574,
    1.5908914516817538,
    1.5722515999637268,
    1.5536579934259822,
    1.5351135049931137,
    1.516620972606961,
    1.498183198576289,
    1.4798029489430826,
    1.4614829528656346,
    1.4432259020186053,
    1.4250344500102174,
    1.4069112118167453,
    1.3888587632344476,
    1.3708796403490884,
    1.3529763390231728,
    1.3351513144010314,
    1.3174069804318613,
    1.2997457094108373,
    1.2821698315383929,
    1.2646816344977574,
    1.2472833630508384,
    1.2299772186525191,
    1.2127653590834377,
    1.1956498981013082,
    1.178632905110826,
    1.1617164048522048,
    1.1449023771083722,
    1.128192756430848,
    1.1115894318843218,
    1.0950942468099332,
    1.0787089986072549,
    1.0624354385349668,
    1.0462752715302026,
    1.0302301560465417,
    1.0143017039106097,
    0.9984914801972461,
    0.98280100312318397,
    0.96723174395918365,
    0.95178512696055162,
    0.93646252931596774,
    0.92126528111453687,
    0.90619466533097248,
    0.8912519178288123,
    0.87643822738155808,
    0.86175473571162375,
    0.84720253754696928,
    0.83278268069528905,
    0.81849616613561682,
    0.80434394812720211,
    0.79032693433550527,
    0.77644598597515069,
    0.76270191796967235,
    0.74909549912787754,
    0.73562745233664761,
    0.72229845476998922,
    0.70910913811414178,
    0.69606008880854064,
    0.68315184830242881,
    0.67038491332690631,
    0.6577597361821943,
    0.6452767250398932,
    0.63293624426000028,
    0.6207386147224524,
    0.60868411417295054,
    0.59677297758281922,
    0.5850053975226468,
    0.57338152454945035,
    0.56190146760709991,
    0.55056529443973601,
    0.53937303201790665,
    0.52832466697714775,
    0.51742014606872355,
    0.5066593766222437,
    0.49604222701986522,
    0.48556852718178634,
    0.47523806906273447,
    0.46505060715914764,
    0.45500585902674401,
    0.44510350580817215,
    0.43534319277043143,
    0.42572452985174813,
    0.41624709221759076,
    0.40691042082550571,
    0.39771402299845149,
    0.38865737300630732,
    0.37973991265523083,
    0.37096105188453632,
    0.36232016937076461,
    0.35381661313861285,
    0.34544970117839224,
    0.33721872206967968,
    0.32912293561082845,
    0.32116157345400287,
    0.31333383974540002,
    0.30563891177032215,
    0.29807594060276243,
    0.29064405175916663,
    0.28334234585603424,
    0.2761698992710202,
    0.26912576480720263,
    0.26220897236017854,
    0.25541852958765315,
    0.24875342258118713,
    0.24221261653976928,
    0.23579505644488113,
    0.22949966773672312,
    0.22332535699127226,
    0.21727101259784407,
    0.21133550543683199,
    0.20551768955730107,
    0.19981640285411362,
    0.19423046774426758,
    0.18875869184213059,
    0.18339986863325555,
    0.17815277814646582,
    0.17301618762390206,
    0.16798885218872447,
    0.16306951551016907,
    0.15825691046565832,
    0.15354975979967164,
    0.1489467767790833,
    0.14444666584468052,
    0.14004812325857782,
    0.13574983774724725,
    0.13155049113988967,
    0.12744875900187527,
    0.1234433112629867,
    0.11953281284020262,
    0.11571592425476389,
    0.1119913022432701,
    0.10835760036255813,
    0.10481346958812036,
    0.10135755890582479,
    0.097988515896704556,
    0.094704987314590081,
    0.091505619656361989,
    0.088389059724609018,
    0.085353955182480276,
    0.082398955100527277,
    0.079522710495336935,
    0.076723874859762231,
    0.07400110468456389,
    0.071353059971281732,
    0.068778404736160925,
    0.066275807504964437,
    0.063843941798509149,
    0.061481486608769377,
    0.059187126865398024,
    0.056959553892521907,
    0.054797465855674236,
    0.052699568198733761,
    0.0506645740707466,
    0.048691204742513469,
    0.046778190012831407,
    0.044924268604285951,
    0.04312818854849633,
    0.041388707560722812,
    0.039704593403752209,
    0.038074624240984234,
    0.036497588978648014,
    0.034972287597085011,
    0.033497531471041127,
    0.032072143678917787,
    0.030694959300938216,
    0.029364825706192126,
    0.028080602828528669,
    0.026841163431274136,
    0.025645393360757718,
    0.024492191788635227,
    0.023380471443007308,
    0.022309158828335351,
    0.021277194434164849,
    0.020283532932672481,
    0.019327143365059765,
    0.01840700931682249,
    0.017522129081931646,
    0.016671515815967857,
    0.015854197678257687,
    0.015069217963066379,
    0.014315635219907833,
    0.013592523363038718,
    0.012898971770209671,
    0.012234085370752528,
    0.011596984723088441,
    0.010986806081747584,
    0.010402701453996851,
    0.0098438386461776797,
    0.0093094012998617016,
    0.0087985889179373611,
    0.0083106168807461234,
    0.0078447164523921444,
    0.0074001347773544985,
    0.006976134867536155,
    0.0065719955798889344,
    0.0061870115847585117,
    0.0058204933250983757,
    0.005471766966706303,
    0.0051401743396414746,
    0.0048250728709848109,
    0.0045258355091094323,
    0.0042418506396323576,
    0.0039725219932226436,
    0.0037172685454451271,
    0.0034755244088227723,
    0.0032467387173043252,
    0.0030303755033275624,
    0.0028259135676718633,
    0.0026328463422971419,
    0.0024506817463693605,
    0.0022789420356758771,
    0.002117163645636781,
    0.001964897028121144,
    0.0018217064822797129,
    0.001687169979608072,
    0.0015608789834566194,
    0.0014424382632059033,
    0.001331465703327907,
    0.0012275921075557673,
    0.0011304609983861676,
    0.0010397284121402527,
    0.00095506268981035924,
    0.00087614426392117911,
    0.00080266544163511545,
    0.00073433018433261097,
    0.00067085388389908132,
    0.00061196313595079726,
    0.00055739551023261584,
    0.00050689931842087213,
    0.00046023337956499678,
    0.00041716678340153728,
    0.0003774786517742137,
    0.00034095789839345335,
    0.0003074029871685043,
    0.00027662168934474004,
    0.00024843083967813107,
    0.0002226560918780724,
    0.00019913167354882632,
    0.00017770014085876181,
    0.00015821213316534946,
    0.00014052612782250516,
    0.00012450819539536599,
    0.00011003175550593137,
    9.6977333531210263e-05,
    8.5232318373585538e-05,
    7.4690721521036553e-05,
    6.5252937612657428e-05,
    5.6825506722567191e-05,
    4.9320878572835681e-05,
    4.265717888344388e-05,
    3.6757978064562751e-05,
    3.1552062453572098e-05,
    2.6973208296252202e-05,
    2.2959958668468274e-05,
    1.9455403531432755e-05,
    1.6406963110275761e-05,
    1.3766174782181355e-05,
    1.1488483656759224e-05,
    9.5330370276200516e-06,
    7.8624828703106735e-06,
    6.4427725578446653e-06,
    5.2429679610375021e-06,
    4.2350530967257535e-06,
    3.393750482719251e-06,
    2.6963423540066857e-06,
    2.1224968903112519e-06,
    1.6540996005765938e-06,
    1.2750900053572486e-06,
    9.7130375339488854e-07,
    7.3032030388490038e-07,
    5.4131630108015585e-07,
    3.9492476294326167e-07,
    2.8310020054819594e-07,
    1.9898977985014241e-07,
    1.3681063229168658e-07,
    9.1733415497510124e-08,
    5.9772220031552366e-08,
    3.768091285354582e-08,
    2.2856002719176348e-08,
    1.3246107323187368e-08,
    7.2680964908937335e-09,
    3.7299801841747706e-09,
    1.7606045064812569e-09,
    7.4621327114692246e-10,
    2.739270417908756e-10,
    8.2185866303004912e-11,
    1.8196210307234265e-11,
    2.4168556006922716e-12,
    1.1276738743808754e-13,
    1.5469228235701481e-16,
    1.5469228235701481e-16,
    1.1276738743808754e-13,
    2.4168556006922716e-12,
    1.8196210307234265e-11,
    8.2185866303004912e-11,
    2.739270417908756e-10,
    7.4621327114692246e-10,
    1.7606045064812569e-09,
    3.7299801841747706e-09,
    7.2680964908937335e-09,
    1.3246107323187368e-08,
    2.2856002719176348e-08,
    3.768091285354582e-08,
    5.9772220031552366e-08,
    9.1733415497510124e-08,
    1.3681063229168658e-07,
    1.9898977985014241e-07,
    2.8310020054819594e-07,
    3.9492476294326167e-07,
    5.4131630108015585e-07,
    7.3032030388490038e-07,
    9.7130375339488854e-07,
    1.2750900053572486e-06,
    1.6540996005765938e-06,
    2.1224968903112519e-06,
    2.6963423540066857e-06,
    3.393750482719251e-06,
    4.2350530967257535e-06,
    5.2429679610375021e-06,
    6.4427725578446653e-06,
    7.8624828703106735e-06,
    9.5330370276200516e-06,
    1.1488483656759224e-05,
    1.3766174782181355e-05,
    1.6406963110275761e-05,
    1.9455403531432755e-05,
    2.2959958668468274e-05,
    2.6973208296252202e-05,
    3.1552062453572098e-05,
    3.6757978064562751e-05,
    4.265717888344388e-05,
    4.9320878572835681e-05,
    5.6825506722567191e-05,
    6.5252937612657428e-05,
    7.4690721521036553e-05,
    8.5232318373585538e-05,
    9.6977333531210263e-05,
    0.00011003175550593137,
    0.00012450819539536599,
    0.00014052612782250516,
    0.00015821213316534946,
    0.00017770014085876181,
    0.00019913167354882632,
    0.0002226560918780724,
    0.00024843083967813107,
    0.00027662168934474004,
    0.0003074029871685043,
    0.00034095789839345335,
    0.0003774786517742137,
    0.00041716678340153728,
    0.00046023337956499678,
    0.00050689931842087213,
    0.00055739551023261584,
    0.00061196313595079726,
    0.00067085388389908132,
    0.00073433018433261097,
    0.00080266544163511545,
    0.00087614426392117911,
    0.00095506268981035924,
    0.0010397284121402527,
    0.0011304609983861676,
    0.0012275921075557673,
    0.001331465703327907,
    0.0014424382632059033,
    0.0015608789834566194,
    0.001687169979608072,
    0.0018217064822797129,
    0.001964897028121144,
    0.002117163645636781,
    0.0022789420356758771,
    0.0024506817463693605,
    0.0026328463422971419,
    0.0028259135676718633,
    0.0030303755033275624,
    0.0032467387173043252,
    0.0034755244088227723,
    0.0037172685454451271,
    0.0039725219932226436,
    0.0042418506396323576,
    0.0045258355091094323,
    0.0048250728709848109,
    0.0051401743396414746,
    0.005471766966706303,
    0.0058204933250983757,
    0.0061870115847585117,
    0.0065719955798889344,
    0.006976134867536155,
    0.0074001347773544985,
    0.0078447164523921444,
    0.0083106168807461234,
    0.0087985889179373611,
    0.0093094012998617016,
    0.0098438386461776797,
    0.010402701453996851,
    0.010986806081747584,
    0.011596984723088441,
    0.012234085370752528,
    0.012898971770209671,
    0.013592523363038718,
    0.014315635219907833,
    0.015069217963066379,
    0.015854197678257687,
    0.016671515815967857,
    0.017522129081931646,
    0.01840700931682249,
    0.019327143365059765,
    0.020283532932672481,
    0.021277194434164849,
    0.022309158828335351,
    0.023380471443007308,
    0.024492191788635227,
    0.025645393360757718,
    0.026841163431274136,
    0.028080602828528669,
    0.029364825706192126,
    0.030694959300938216,
    0.032072143678917787,
    0.033497531471041127,
    0.034972287597085011,
    0.036497588978648014,
    0.038074624240984234,
    0.039704593403752209,
    0.041388707560722812,
    0.04312818854849633,
    0.044924268604285951,
    0.046778190012831407,
    0.048691204742513469,
    0.0506645740707466,
    0.052699568198733761,
    0.054797465855674236,
    0.056959553892521907,
    0.059187126865398024,
    0.061481486608769377,
    0.063843941798509149,
    0.066275807504964437,
    0.068778404736160925,
    0.071353059971281732,
    0.07400110468456389,
    0.076723874859762231,
    0.079522710495336935,
    0.082398955100527277,
    0.085353955182480276,
    0.088389059724609018,
    0.091505619656361989,
    0.094704987314590081,
    0.097988515896704556,
    0.10135755890582479,
    0.10481346958812036,
    0.10835760036255813,
    0.1119913022432701,
    0.11571592425476389,
    0.11953281284020262,
    0.1234433112629867,
    0.12744875900187527,
    0.13155049113988967,
    0.13574983774724725,
    0.14004812325857782,
    0.14444666584468052,
    0.1489467767790833,
    0.15354975979967164,
    0.15825691046565832,
    0.16306951551016907,
    0.16798885218872447,
    0.17301618762390206,
    0.17815277814646582,
    0.18339986863325555,
    0.18875869184213059,
    0.19423046774426758,
    0.19981640285411362,
    0.20551768955730107,
    0.21133550543683199,
    0.21727101259784407,
    0.22332535699127226,
    0.22949966773672312,
    0.23579505644488113,
    0.24221261653976928,
    0.24875342258118713,
    0.25541852958765315,
    0.26220897236017854,
    0.26912576480720263,
    0.2761698992710202,
    0.28334234585603424,
    0.29064405175916663,
    0.29807594060276243,
    0.30563891177032215,
    0.31333383974540002,
    0.32116157345400287,
    0.32912293561082845,
    0.33721872206967968,
    0.34544970117839224,
    0.35381661313861285,
    0.36232016937076461,
    0.37096105188453632,
    0.37973991265523083,
    0.38865737300630732,
    0.39771402299845149,
    0.40691042082550571,
    0.41624709221759076,
    0.42572452985174813,
    0.43534319277043143,
    0.44510350580817215,
    0.45500585902674401,
    0.46505060715914764,
    0.47523806906273447,
    0.48556852718178634,
    0.49604222701986522,
    0.5066593766222437,
    0.51742014606872355,
    0.52832466697714775,
    0.53937303201790665,
    0.55056529443973601,
    0.56190146760709991,
    0.57338152454945035,
    0.5850053975226468,
    0.59677297758281922,
    0.60868411417295054,
    0.6207386147224524,
    0.63293624426000028,
    0.6452767250398932,
    0.6577597361821943,
    0.67038491332690631,
    0.68315184830242881,
    0.69606008880854064,
    0.70910913811414178,
    0.72229845476998922,
    0.73562745233664761,
    0.74909549912787754,
    0.76270191796967235,
    0.77644598597515069,
    0.79032693433550527,
    0.80434394812720211,
    0.81849616613561682,
    0.83278268069528905,
    0.84720253754696928,
    0.86175473571162375,
    0.87643822738155808,
    0.8912519178288123,
    0.90619466533097248,
    0.92126528111453687,
    0.93646252931596774,
    0.95178512696055162,
    0.96723174395918365,
    0.98280100312318397,
    0.9984914801972461,
    1.0143017039106097,
    1.0302301560465417,
    1.0462752715302026,
    1.0624354385349668,
    1.0787089986072549,
    1.0950942468099332,
    1.1115894318843218,
    1.128192756430848,
    1.1449023771083722,
    1.1617164048522048,
    1.178632905110826,
    1.1956498981013082,
    1.2127653590834377,
    1.2299772186525191,
    1.2472833630508384,
    1.2646816344977574,
    1.2821698315383929,
    1.2997457094108373,
    1.3174069804318613,
    1.3351513144010314,
    1.3529763390231728,
    1.3708796403490884,
    1.3888587632344476,
    1.4069112118167453,
    1.4250344500102174,
    1.4432259020186053,
    1.4614829528656346,
    1.4798029489430826,
    1.498183198576289,
    1.516620972606961,
    1.5351135049931137,
    1.5536579934259822,
    1.5722515999637268,
    1.5908914516817538,
    1.6095746413394574,
    1.6282982280631855,
    1.6470592380452249,
    1.6658546652585899,
    1.6846814721873931,
    1.7035365905725683,
    1.722416922172713,
    1.7413193395397977,
    1.7602406868094997,
    1.7791777805058964,
    1.7981274103602536,
    1.8170863401436368,
    1.836051308513065,
    1.855019029870919,
    1.8739861952373116,
    1.8929494731351206,
    1.9119055104873766,
    1.9308509335266915,
    1.949782348716413,
    1.9686963436831757,
    1.9875894881605178,
    2.0064583349432308,
    2.0252994208520936,
    2.0441092677086488,
    2.0628843833196622,
    2.0816212624709127,
    2.1003163879299431,
    2.1189662314574065,
    2.1375672548266378,
    2.1561159108510632,
    2.1746086444190769,
    2.1930418935359874,
    2.2114120903726544,
    2.2297156623204097,
    2.2479490330518721,
    2.266108623587253,
    2.2841908533657427,
    2.3021921413215773,
    2.320108906964367,
    2.3379375714632733,
    2.3556745587346248,
    2.3733162965325394,
    2.3908592175421446,
    2.4082997604749621,
    2.4256343711660362,
    2.4428595036723775,
    2.4599716213722913,
    2.4769671980651609,
    2.493842719071258,
    2.5105946823311411,
    2.5272195995042157,
    2.5437139970660141,
    2.5600744174037717,
    2.5762974199098561,
    2.5923795820726188,
    2.6083175005642341,
    2.6241077923250948,
    2.6397470956443274,
    2.655232071235996,
    2.6705594033105671,
    2.6857258006412006,
    2.7007279976244392,
    2.7155627553348749,
    2.730226862573359,
    2.7447171369083394,
    2.7590304257098994,
    2.7731636071760839,
    2.7871135913510936,
    2.8008773211349345,
    2.8144517732841154,
    2.8278339594029807,
    2.8410209269252813,
    2.8540097600855789,
    2.8667975808800863,
    2.8793815500165576,
    2.8917588678528299,
    2.9039267753236411,
    2.9158825548553402,
    2.9276235312681114,
    2.9391470726653472,
    2.9504505913098042,
    2.9615315444861738,
    2.9723874353497255,
    2.9830158137606571,
    2.9934142771038235,
    3.0035804710934952,
    3.0135120905628163,
    3.0232068802376366,
    3.0326626354943995,
    3.0418772031017678,
    3.0508484819456823,
    3.0595744237375535,
    3.06805303370529,
    3.076282371266879,
    3.0842605506862384,
    3.0919857417110688,
    3.0994561701924357,
    3.1066701186858334,
    3.1136259270334694,
    3.1203219929275332,
    3.1267567724542178,
    3.1329287806182573,
    3.1388365918477699,
    3.144478840479195,
    3.1498542212221174,
    3.1549614896037905,
    3.159799462393166,
    3.1643670180042602,
    3.1686630968786846,
    3.1726867018471778,
    3.1764368984699942,
    3.179912815356003,
    3.1831136444603643,
    3.1860386413606583,
    3.1886871255113505,
    3.1910584804764919,
    3.1931521541405461,
    3.1949676588972697,
    3.1965045718165492,
    3.1977625347891427,
    3.1987412546492515,
    3.1994405032748774,
    3.1998601176659229,
    0.40000000000000002,
    0.40923141821223619,
    0.41860323628034157,
    0.42811593852691343,
    0.43776997549584834,
    0.44756576338110271,
    0.45750368346403181,
    0.46758408155962972,
    0.47780726747198721,
    0.48817351445928492,
    0.49868305870863416,
    0.50933609882107544,
    0.52013279530704226,
    0.53107327009259342,
    0.54215760603671459,
    0.55338584645998479,
    0.56475799468490218,
    0.576274013588156,
    0.58793382516513149,
    0.59973731010692699,
    0.61168430739015889,
    0.62377461387982769,
    0.6360079839455095,
    0.6483841290911373,
    0.6609027175986264,
    0.67356337418559764,
    0.68636567967744344,
    0.69930917069397758,
    0.71239333935090499,
    0.72561763297633963,
    0.73898145384259484,
    0.7524841589134641,
    0.76612505960720179,
    0.77990342157541226,
    0.79381846449804261,
    0.80786936189467418,
    0.82205524095229676,
    0.83637518236974517,
    0.85082822021896953,
    0.86541334182330609,
    0.88012948765290411,
    0.89497555123746197,
    0.90995037909641452,
    0.92505277068670866,
    0.94028147836829645,
    0.95563520738746532,
    0.97111261587812125,
    0.98671231488112821,
    1.0024328683818051,
    1.0182727933656675,
    1.0342305598924988,
    1.0503045911888231,
    1.0664932637588489,
    1.0827949075139383,
    1.0992078059206547,
    1.1157301961674302,
    1.1323602693498835,
    1.1490961706748182,
    1.1659359996829137,
    1.1828778104901221,
    1.1999196120477651,
    1.2170593684213289,
    1.2342949990879353,
    1.2516243792524662,
    1.2690453401823083,
    1.2865556695606726,
    1.3041531118584404,
    1.3218353687244779,
    1.3396000993943458,
    1.3574449211173345,
    1.3753674096017352,
    1.3933650994782556,
    1.4114354847814783,
    1.4295760194492495,
    1.4477841178398851,
    1.466057155267058,
    1.4843924685522418,
    1.502787356594558,
    1.52123908095788,
    1.5397448664750337,
    1.5583019018689233,
    1.5769073403904084,
    1.595558300472748,
    1.6142518664024172,
    1.6329850890060977,
    1.6517549863536347,
    1.6705585444767403,
    1.6893927181032269,
    1.70825443140653,
    1.7271405787702918,
    1.7460480255677508,
    1.7649736089556887,
    1.783914138682674,
    1.8028663979113311,
    1.821827144054365,
    1.8407931096240533,
    1.8597610030949256,
    1.8787275097793266,
    1.897689292715566,
    1.9166429935683473,
    1.9355852335411576,
    1.9545126143003018,
    1.9734217189102505,
    1.9923091127799741,
    2.0111713446199189,
    2.0300049474092847,
    2.0488064393732563,
    2.067572324969829,
    2.086299095885872,
    2.1049832320420649,
    2.1236212026063339,
    2.1422094670154181,
    2.1607444760041838,
    2.1792226726423025,
    2.1976404933779095,
    2.2159943690878463,
    2.2342807261340938,
    2.2524959874259967,
    2.2706365734878795,
    2.288698903531639,
    2.3066793965339167,
    2.3245744723174271,
    2.3423805526360355,
    2.3600940622631628,
    2.3777114300830982,
    2.3952290901848001,
    2.4126434829577561,
    2.4299510561894819,
    2.4471482661642234,
    2.4642315787624427,
    2.4811974705606477,
    2.4980424299311372,
    2.5147629581412354,
    2.5313555704515665,
    2.5478167972129566,
    2.5641431849615084,
    2.5803312975114272,
    2.5963777170451632,
    2.6122790452004274,
    2.6280319041536586,
    2.6436329376995009,
    2.6590788123258653,
    2.674366218284137,
    2.6894918706541104,
    2.7044525104032107,
    2.7192449054395862,
    2.7338658516586398,
    2.7483121739825811,
    2.7625807273925767,
    2.7766683979530811,
    2.7905721038279347,
    2.8042887962878122,
    2.8178154607086188,
    2.8311491175604186,
    2.8442868233865015,
    2.8572256717721847,
    2.8699627943029502,
    2.8824953615115367,
    2.8948205838135834,
    2.906935712431459,
    2.9188380403058822,
    2.9305249029949714,
    2.9419936795603454,
    2.9532417934399153,
    2.9642667133070106,
    2.9750659539154767,
    2.9856370769304057,
    2.9959776917441525,
    3.0060854562772987,
    3.0159580777642354,
    3.0255933135230393,
    3.0349889717093226,
    3.0441429120537475,
    3.0530530465828947,
    3.0617173403231921,
    3.0701338119876072,
    3.0783005346448222,
    3.0862156363706119,
    3.0938773008811506,
    3.1012837681479914,
    3.1084333349944555,
    3.115324355673188,
    3.1219552424246348,
    3.1283244660162137,
    3.1344305562619499,
    3.1402721025223617,
    3.1458477541843872,
    3.1511562211211519,
    3.1561962741313847,
    3.1609667453582992,
    3.1654665286877663,
    3.1696945801256087,
    3.1736499181538611,
    3.1773316240658516,
    3.1807388422799545,
    3.1838707806318918,
    3.1867267106454569,
    3.1893059677815501,
    3.1916079516654157,
    3.1936321262919929,
    3.1953780202092914,
    3.196845226679712,
    3.1980334038192484,
    3.1989422747145131,
    3.199571627517531,
    3.1999213155182713,
    3.1999912571948785,
    3.1997814362415857,
    3.199291901574298,
    3.1985227673138423,
    3.1974742127468958,
    3.1961464822645977,
    3.1945398852788851,
    3.1926547961165759,
    3.1904916538912489,
    3.1880509623529751,
    3.1853332897159601,
    3.1823392684641703,
    3.1790695951350285,
    3.1755250300812645,
    3.1717063972110213,
    3.1676145837063303,
    3.1632505397200643,
    3.1586152780515073,
    3.153709873800663,
    3.1485354640014598,
    3.1430932472339967,
    3.1373844832159938,
    3.1314104923736226,
    3.125172655391891,
    3.1186724127447696,
    3.1119112642052627,
    3.1048907683356184,
    3.0976125419578997,
    3.0900782596051313,
    3.0822896529532531,
    3.0742485102341179,
    3.0659566756297751,
    3.0574160486482955,
    3.0486285834813951,
    3.0395962883441241,
    3.0303212247968996,
    3.0208055070501563,
    3.0110513012519129,
    3.0010608247585431,
    2.9908363453890527,
    2.9803801806631793,
    2.969694697023622,
    2.9587823090427254,
    2.9476454786139508,
    2.9362867141284581,
    2.9247085696371498,
    2.9129136439985133,
    2.9009045800126194,
    2.8886840635416307,
    2.8762548226171787,
    2.8636196265349838,
    2.8507812849370828,
    2.8377426468820461,
    2.8245065999035592,
    2.8110760690577625,
    2.7974540159597292,
    2.7836434378094839,
    2.7696473664079537,
    2.7554688671632568,
    2.7411110380877264,
    2.7265770087860886,
    2.7118699394351951,
    2.6969930197557308,
    2.6819494679763101,
    2.6667425297903828,
    2.6513754773063694,
    2.6358516079914489,
    2.6201742436094295,
    2.6043467291531242,
    2.588372431771659,
    2.5722547396931543,
    2.5559970611431941,
    2.5396028232595342,
    2.5230754710034646,
    2.5064184660682765,
    2.4896352857852526,
    2.4727294220276272,
    2.4557043801129446,
    2.4385636777042459,
    2.4213108437105313,
    2.4039494171869134,
    2.3864829462349082,
    2.3689149869032868,
    2.3512491020899215,
    2.3334888604450517,
    2.315637835276402,
    2.2976996034565715,
    2.2796777443331231,
    2.261575838641789,
    2.2433974674232151,
    2.2251462109436617,
    2.2068256476200667,
    2.1884393529498909,
    2.1699908984461493,
    2.1514838505780305,
    2.1329217697175085,
    2.1143082090923442,
    2.0956467137458694,
    2.0769408195039398,
    2.058194051949449,
    2.0394099254047799,
    2.0205919419225711,
    2.0017435902851721,
    1.9828683450131519,
    1.9639696653832275,
    1.9450509944559651,
    1.9261157581136088,
    1.9071673641083837,
    1.8882092011216132,
    1.8692446378339858,
    1.8502770220073035,
    1.8313096795780328,
    1.8123459137629756,
    1.7933890041773757,
    1.774442205965759,
    1.7555087489458125,
    1.7365918367655895,
    1.7176946460743285,
    1.6988203257071621,
    1.67997199588399,
    1.6611527474227774,
    1.6423656409675373,
    1.6236137062312483,
    1.6048999412539453,
    1.5862273116762238,
    1.5675987500283819,
    1.5490171550354206,
    1.530485390938114,
    1.5120062868303543,
    1.4935826360129689,
    1.4752171953641948,
    1.4569126847269946,
    1.4386717863133844,
    1.4204971441259375,
    1.4023913633966207,
    1.3843570100431124,
    1.3663966101427374,
    1.3485126494241568,
    1.3307075727769284,
    1.3129837837790561,
    1.2953436442426358,
    1.27778947377769,
    1.2603235493742859,
    1.2429481050030133,
    1.2256653312338979,
    1.2084773748738107,
    1.1913863386224295,
    1.1743942807468011,
    1.1575032147745385,
    1.140715109205686,
    1.1240318872432713,
    1.107455426542558,
    1.090987558979003,
    1.0746300704349125,
    1.0583847006047875,
    1.0422531428193336,
    1.0262370438881094,
    1.0103380039607734,
    0.99455757640688358,
    0.97889726771419794,
    0.96335853740541078,
    0.94794279797325587,
    0.93265141483389846,
    0.91748570629852833,
    0.90244694356306143,
    0.88753635071584713,
    0.87275510476327078,
    0.85810433567313538,
    0.84358512643569661,
    0.82919851314221837,
    0.81494548508091003,
    0.80082698485009651,
    0.7868439084884683,
    0.7729971056222491,
    0.75928737962911175,
    0.74571548781866892,
    0.73228214162935445,
    0.71898800684150621,
    0.7058337038064576,
    0.69281980769143181,
    0.6799468487400333,
    0.66721531254812183,
    0.65462564035484605,
    0.6421782293486139,
    0.6298734329877631,
    0.61771156133569816,
    0.60569288141024635,
    0.5938176175469867,
    0.58208595177629552,
    0.57049802421385121,
    0.55905393346433196,
    0.54775373703803909,
    0.53659745178017071,
    0.52558505431246905,
    0.51471648148695681,
    0.50399163085147791,
    0.49341036112675013,
    0.48297249269463538,
    0.4726778080973294,
    0.462526052547169,
    0.45251693444675078,
    0.44265012591905317,
    0.43292526334725012,
    0.42334194792390206,
    0.41389974620920605,
    0.40459819069798669,
    0.39543678039510394,
    0.38641498139895503,
    0.37753222749274279,
    0.36878792074318301,
    0.3601814321063202,
    0.35171210204012071,
    0.3433792411235102,
    0.33518213068152158,
    0.32712002341621799,
    0.3191921440430559,
    0.31139768993235112,
    0.303735831755511,
    0.29620571413569602,
    0.28880645630257279,
    0.28153715275082186,
    0.2743968739020628,
    0.26738466676985995,
    0.26049955562747351,
    0.2537405426780191,
    0.24710660872670318,
    0.24059671385479928,
    0.23420979809503445,
    0.22794478210805377,
    0.22180056785963501,
    0.21577603929832523,
    0.20987006303317418,
    0.20408148901124099,
    0.19840915119455291,
    0.19285186823619718,
    0.18740844415522984,
    0.18207766901008776,
    0.17685831957019316,
    0.17174915998544268,
    0.16674894245327648,
    0.1618564078830258,
    0.15707028655724134,
    0.15238929878970758,
    0.14781215557985269,
    0.14333755926326719,
    0.13896420415804808,
    0.13469077720669043,
    0.13051595861325083,
    0.12643842247551362,
    0.12245683741189349,
    0.11856986718281355,
    0.11477617130630278,
    0.11107440566756094,
    0.10746322312224472,
    0.10394127409323346,
    0.10050720716063813,
    0.097159669644822713,
    0.093897308182211994,
    0.090718769293665805,
    0.08762269994520501,
    0.08460774810088019,
    0.081672563267579851,
    0.07881579703158062,
    0.076036103586648099,
    0.073332140253502529,
    0.070702567990470172,
    0.068146051895146609,
    0.065661261696905129,
    0.063246872240089028,
    0.06090156395773344,
    0.058624023335668171,
    0.056412943366859897,
    0.054267023995858303,
    0.052184972553217274,
    0.050165504179768804,
    0.048207342240633978,
    0.046309218728861817,
    0.044469874658593603,
    0.042688060447656814,
    0.040962536289499647,
    0.039292072514383762,
    0.037675449939759478,
    0.036111460209754621,
    0.03459890612371487,
    0.033136601953740033,
    0.031723373751167745,
    0.030358059641962563,
    0.029039510110975286,
    0.02776658827504401,
    0.02653817014491517,
    0.025353144875969479,
    0.024210415007744315,
    0.023108896692250813,
    0.022047519911090482,
    0.02102522868138269,
    0.020040981250521013,
    0.019093750279782832,
    0.018182523016823039,
    0.017306301457089175,
    0.016464102494201528,
    0.01565495805934821,
    0.014877915249751253,
    0.014132036446266113,
    0.013416399420182985,
    0.012730097429304364,
    0.012072239303379331,
    0.011441949518980827,
    0.010838368263918092,
    0.01026065149128211,
    0.0097079709632275756,
    0.0091795142846004558,
    0.0086744849265256892,
    0.0081921022400749336,
    0.0077316014601395743,
    0.0072922336996393607,
    0.006873265934202134,
    0.0064739809774550967,
    0.0060936774470729104,
    0.0057316697217327109,
    0.0053872878891307379,
    0.005059877685219847,
    0.0047488004248315615,
    0.004453432923850636,
    0.0041731674131142878,
    0.0039074114442122845,
    0.0036555877873680367,
    0.0034171343215846244,
    0.0031915039172433764,
    0.002978164311346153,
    0.0027765979755959118,
    0.0025863019775133945,
    0.002406787834790924,
    0.0022375813630873148,
    0.0020782225174707462,
    0.001928265227719192,
    0.0017872772276905731,
    0.0016548398789772523,
    0.0015305479890617799,
    0.0014140096241929607,
    0.0013048459172033176,
    0.0012026908704908916,
    0.0011071911543900322,
    0.0010180059011574035,
    0.00093480649480084347,
    0.00085727635697999192,
    0.0007851107292087189,
    0.00071801645159035824,
    0.00065571173831757159,
    0.00059792595016933939,
    0.00054439936423809746,
    0.00049488294112040609,
    0.00044913808980476044,
    0.00040693643049022246,
    0.00036805955556947173,
    0.00033229878900964822,
    0.00029945494436397856,
    0.00026933808164665302,
    0.00024176726330274589,
    0.00021657030950414936,
    0.00019358355300152532,
    0.00017265159376116319,
    0.00015362705361437556,
    0.00013637033114566037,
    0.00012074935704431384,
    0.00010663935014249239,
    9.3922574360894714e-05,
    8.2488096781270552e-05,
    7.2231547062859186e-05,
    6.3054878417622166e-05,
    5.4866130356760921e-05,
    4.7579193418504208e-05,
    4.1113576084512265e-05,
    3.5394174089478253e-05,
    3.0351042325613014e-05,
    2.5919169540679171e-05,
    2.2038256025097347e-05,
    1.8652494480381931e-05,
    1.5710354257779528e-05,
    1.3164369152481474e-05,
    1.0970928935165209e-05,
    9.0900747988900036e-06,
    7.4852988955331387e-06,
    6.123348132005338e-06,
    4.9740323924318038e-06,
    4.010037348329983e-06,
    3.206742014560003e-06,
    2.542041204471082e-06,
    1.9961730332200117e-06,
    1.5515516136987556e-06,
    1.1926050848801471e-06,
    9.0561910767656198e-07,
    6.7858595860913924e-07,
    5.0105934670770267e-07,
    3.640150741069644e-07,
    2.5971765577599755e-07,
    1.8159300871842911e-07,
    1.2410731581351383e-07,
    8.265216423638399e-08,
    5.3436053102569969e-08,
    3.3382359630618603e-08,
    2.0033847710596553e-08,
    1.1463797308789999e-08,
    6.1938276333541296e-09,
    3.1184814354152385e-09,
    1.4366322285937581e-09,
    5.89770580533169e-10,
    2.0721996624200356e-10,
    5.8326978358496414e-11,
    1.1664967321361311e-11,
    1.2844383824833729e-12,
    3.7765948219436406e-14,
    2.4170735155792035e-18,
    1.7620337551928726e-15,
    2.8435385147717822e-13,
    4.2815252397032076e-12,
    2.7526362050794351e-11,
    1.1367607712498612e-10,
    3.5764160882722917e-10,
    9.3580314932752862e-10,
    2.1432869319099378e-09,
    4.4382728312692683e-09,
    8.4932965740654085e-09,
    1.5253504622479013e-08,
    2.6001814084663028e-08,
    4.2430924326311416e-08,
    6.6722121310552728e-08,
    1.0163081008248673e-07,
    1.505787052433617e-07,
    2.1775260373106111e-07,
    3.0820965874162987e-07,
    4.2798906819439413e-07,
    5.8423008576416097e-07,
    7.8529625718133874e-07,
    1.0409057792378815e-06,
    1.3622678737369787e-06,
    1.7622250634906015e-06,
    2.2554012324045623e-06,
    2.85835534669877e-06,
    3.5897407093939846e-06,
    4.4704696153586458e-06,
    5.52388326945326e-06,
    6.7759268256383943e-06,
    8.2553294003284026e-06,
    9.993788908779526e-06,
    1.2026161568900745e-05,
    1.4390655912571523e-05,
    1.7129031140344997e-05,
    2.0286799651311091e-05,
    2.3913433575893727e-05,
    2.8062575135462777e-05,
    3.2792250648856637e-05,
    3.8165088002238097e-05,
    4.4248537395146809e-05,
    5.1115095172168216e-05,
    5.8842530546313747e-05,
    6.7514115017002708e-05,
    7.7218854282454095e-05,
    8.8051722443339216e-05,
    0.00010011389829171556,
    0.0001135130034765593,
    0.00012836334233464267,
    0.00014478614317306134,
    0.00016290980078741168,
    0.00018287011999744548,
    0.00020481055997999644,
    0.00022888247917707611,
    0.00025524538055528029,
    0.00028406715699103203,
    0.00031552433655471245,
    0.00034980232746540258,
    0.00038709566248677004,
    0.00042760824253359667,
    0.00047155357925754665,
    0.00051915503638002669,
    0.00057064606953939048,
    0.00062627046441928607,
    0.00068628257292464379,
    0.00075094754717164498,
    0.00082054157105801006,
    0.00089535208918008874,
    0.0009756780328635308,
    0.0010618300430747607,
    0.0011541306899810768,
    0.0012529146889279422,
    0.0013585291126029283,
    0.0014713335991568199,
    0.0015917005560535877,
    0.0017200153594222757,
    0.0018566765486853483,
    0.0020020960162396785,
    0.0021566991919681542,
    0.0023209252223618114,
    0.0024952271440344827,
    0.0026800720514141845,
    0.0028759412583978282,
    0.0030833304537583603,
    0.0033027498500960914,
    0.0035347243261287641,
    0.003779793562117864,
    0.0040385121682317187,
    0.004311449805649183,
    0.0045991913002110061,
    0.0049023367484295043,
    0.0052215016156707209,
    0.0055573168263270464,
    0.0059104288458020939,
    0.0062814997541336525,
    0.0066712073110846313,
    0.0070802450125361811,
    0.0075093221380214924,
    0.0079591637892432905,
    0.0084305109194226201,
    0.0089241203533312018,
    0.009440764797864501,
    0.00998123284301752,
    0.0105463289531304,
    0.011136873448276017,
    0.011753702475667004,
    0.012397667970964973,
    0.013069637609380088,
    0.013770494746454722,
    0.014501138348430465,
    0.015262482912103508,
    0.016055458374079141,
    0.016881010009341994,
    0.017740098319064566,
    0.01863369890758259,
    0.019562802348471856,
    0.020528414039667257,
    0.021531554047571008,
    0.022573256940103287,
    0.023654571608654845,
    0.024776561078907494,
    0.025940302310494819,
    0.027146885985481976,
    0.028397416285649856,
    0.02969301065857553,
    0.031034799572507436,
    0.032423926260040439,
    0.033861546450602478,
    0.035348828091771228,
    0.036886951059445922,
    0.038477106856906119,
    0.040120498302796084,
    0.041818339208079945,
    0.043571854042019749,
    0.045382277587235259,
    0.047250854583910942,
    0.049178839363222461,
    0.051167495470061719,
    0.053218095275146191,
    0.055331919576604917,
    0.057510257191140411,
    0.059754404534872116,
    0.062065665193974003,
    0.064445349485225192,
    0.066894774006599295,
    0.069415261178024545,
    0.072008138772453295,
    0.074674739437386117,
    0.077416400207001676,
    0.080234462005050369,
    0.08313026913867573,
    0.086105168783333824,
    0.089160510458987208,
    0.092297645497755793,
    0.095517926503213349,
    0.098822706801524024,
    0.10221333988461925,
    0.10569117884562111,
    0.10925757580672418,
    0.11291388133975305,
    0.1166614438796186,
    0.12050160913090147,
    0.12443571946779618,
    0.12846511332765517,
    0.13259112459837646,
    0.13681508199988415,
    0.14113830845995573,
    0.14556212048465478,
    0.15008782752363259,
    0.1547167313305666,
    0.15945012531900798,
    0.16428929391391511,
    0.1692355118991537,
    0.17429004376124854,
    0.17945414302967563,
    0.18472905161398712,
    0.19011599913806559,
    0.19561620227180701,
    0.20123086406053572,
    0.20696117325245741,
    0.21280830362445982,
    0.21877341330657285,
    0.22485764410540363,
    0.23106212082686381,
    0.23738795059850926,
    0.24383622219181433,
    0.25040800534470564,
    0.25710435008468135,
    0.26392628605284441,
    0.27087482182917971,
    0.27795094425940692,
    0.28515561778374104,
    0.29248978376789525,
    0.29995435983666058,
    0.30755023921039842,
    0.31527829004478186,
    0.32313935477412337,
    0.33113424945862513,
    0.33926376313589002,
    0.34752865717703041,
    0.35592966464771159,
    0.36446748967446657,
    0.37314280681661877,
    0.38195626044414743,
    0.39090846412183078,
    0.40000000000000002,
    0.39090846412183078,
    0.38195626044414743,
    0.37314280681661877,
    0.36446748967446657,
    0.35592966464771159,
    0.34752865717703041,
    0.33926376313589002,
    0.33113424945862513,
    0.32313935477412337,
    0.31527829004478186,
    0.30755023921039842,
    0.29995435983666058,
    0.29248978376789525,
    0.28515561778374104,
    0.27795094425940692,
    0.27087482182917971,
    0.26392628605284441,
    0.25710435008468135,
    0.25040800534470564,
    0.24383622219181433,
    0.23738795059850926,
    0.23106212082686381,
    0.22485764410540363,
    0.21877341330657285,
    0.21280830362445982,
    0.20696117325245741,
    0.20123086406053572,
    0.19561620227180701,
    0.19011599913806559,
    0.18472905161398712,
    0.17945414302967563,
    0.17429004376124854,
    0.1692355118991537,
    0.16428929391391511,
    0.15945012531900798,
    0.1547167313305666,
    0.15008782752363259,
    0.14556212048465478,
    0.14113830845995573,
    0.13681508199988415,
    0.13259112459837646,
    0.12846511332765517,
    0.12443571946779618,
    0.12050160913090147,
    0.1166614438796186,
    0.11291388133975305,
    0.10925757580672418,
    0.10569117884562111,
    0.10221333988461925,
    0.098822706801524024,
    0.095517926503213349,
    0.092297645497755793,
    0.089160510458987208,
    0.086105168783333824,
    0.08313026913867573,
    0.080234462005050369,
    0.077416400207001676,
    0.074674739437386117,
    0.072008138772453295,
    0.069415261178024545,
    0.066894774006599295,
    0.064445349485225192,
    0.062065665193974003,
    0.059754404534872116,
    0.057510257191140411,
    0.055331919576604917,
    0.053218095275146191,
    0.051167495470061719,
    0.049178839363222461,
    0.047250854583910942,
    0.045382277587235259,
    0.043571854042019749,
    0.041818339208079945,
    0.040120498302796084,
    0.038477106856906119,
    0.036886951059445922,
    0.035348828091771228,
    0.033861546450602478,
    0.032423926260040439,
    0.031034799572507436,
    0.02969301065857553,
    0.028397416285649856,
    0.027146885985481976,
    0.025940302310494819,
    0.024776561078907494,
    0.023654571608654845,
    0.022573256940103287,
    0.021531554047571008,
    0.020528414039667257,
    0.019562802348471856,
    0.01863369890758259,
    0.017740098319064566,
    0.016881010009341994,
    0.016055458374079141,
    0.015262482912103508,
    0.014501138348430465,
    0.013770494746454722,
    0.013069637609380088,
    0.012397667970964973,
    0.011753702475667004,
    0.011136873448276017,
    0.0105463289531304,
    0.00998123284301752,
    0.009440764797864501,
    0.0089241203533312018,
    0.0084305109194226201,
    0.0079591637892432905,
    0.0075093221380214924,
    0.0070802450125361811,
    0.0066712073110846313,
    0.0062814997541336525,
    0.0059104288458020939,
    0.0055573168263270464,
    0.0052215016156707209,
    0.0049023367484295043,
    0.0045991913002110061,
    0.004311449805649183,
    0.0040385121682317187,
    0.003779793562117864,
    0.0035347243261287641,
    0.0033027498500960914,
    0.0030833304537583603,
    0.0028759412583978282,
    0.0026800720514141845,
    0.0024952271440344827,
    0.0023209252223618114,
    0.0021566991919681542,
    0.0020020960162396785,
    0.0018566765486853483,
    0.0017200153594222757,
    0.0015917005560535877,
    0.0014713335991568199,
    0.0013585291126029283,
    0.0012529146889279422,
    0.0011541306899810768,
    0.0010618300430747607,
    0.0009756780328635308,
    0.00089535208918008874,
    0.00082054157105801006,
    0.00075094754717164498,
    0.00068628257292464379,
    0.00062627046441928607,
    0.00057064606953939048,
    0.00051915503638002669,
    0.00047155357925754665,
    0.00042760824253359667,
    0.00038709566248677004,
    0.00034980232746540258,
    0.00031552433655471245,
    0.00028406715699103203,
    0.00025524538055528029,
    0.00022888247917707611,
    0.00020481055997999644,
    0.00018287011999744548,
    0.00016290980078741168,
    0.00014478614317306134,
    0.00012836334233464267,
    0.0001135130034765593,
    0.00010011389829171556,
    8.8051722443339216e-05,
    7.7218854282454095e-05,
    6.7514115017002708e-05,
    5.8842530546313747e-05,
    5.1115095172168216e-05,
    4.4248537395146809e-05,
    3.8165088002238097e-05,
    3.2792250648856637e-05,
    2.8062575135462777e-05,
    2.3913433575893727e-05,
    2.0286799651311091e-05,
    1.7129031140344997e-05,
    1.4390655912571523e-05,
    1.2026161568900745e-05,
    9.993788908779526e-06,
    8.2553294003284026e-06,
    6.7759268256383943e-06,
    5.52388326945326e-06,
    4.4704696153586458e-06,
    3.5897407093939846e-06,
    2.85835534669877e-06,
    2.2554012324045623e-06,
    1.7622250634906015e-06,
    1.3622678737369787e-06,
    1.0409057792378815e-06,
    7.8529625718133874e-07,
    5.8423008576416097e-07,
    4.2798906819439413e-07,
    3.0820965874162987e-07,
    2.1775260373106111e-07,
    1.505787052433617e-07,
    1.0163081008248673e-07,
    6.6722121310552728e-08,
    4.2430924326311416e-08,
    2.6001814084663028e-08,
    1.5253504622479013e-08,
    8.4932965740654085e-09,
    4.4382728312692683e-09,
    2.1432869319099378e-09,
    9.3580314932752862e-10,
    3.5764160882722917e-10,
    1.1367607712498612e-10,
    2.7526362050794351e-11,
    4.2815252397032076e-12,
    2.8435385147717822e-13,
    1.7620337551928726e-15,
    2.4170735155792035e-18,
    3.7765948219436406e-14,
    1.2844383824833729e-12,
    1.1664967321361311e-11,
    5.8326978358496414e-11,
    2.0721996624200356e-10,
    5.89770580533169e-10,
    1.4366322285937581e-09,
    3.1184814354152385e-09,
    6.1938276333541296e-09,
    1.1463797308789999e-08,
    2.0033847710596553e-08,
    3.3382359630618603e-08,
    5.3436053102569969e-08,
    8.265216423638399e-08,
    1.2410731581351383e-07,
    1.8159300871842911e-07,
    2.5971765577599755e-07,
    3.640150741069644e-07,
    5.0105934670770267e-07,
    6.7858595860913924e-07,
    9.0561910767656198e-07,
    1.1926050848801471e-06,
    1.5515516136987556e-06,
    1.9961730332200117e-06,
    2.542041204471082e-06,
    3.206742014560003e-06,
    4.010037348329983e-06,
    4.9740323924318038e-06,
    6.123348132005338e-06,
    7.4852988955331387e-06,
    9.0900747988900036e-06,
    1.0970928935165209e-05,
    1.3164369152481474e-05,
    1.5710354257779528e-05,
    1.8652494480381931e-05,
    2.2038256025097347e-05,
    2.5919169540679171e-05,
    3.0351042325613014e-05,
    3.5394174089478253e-05,
    4.1113576084512265e-05,
    4.7579193418504208e-05,
    5.4866130356760921e-05,
    6.3054878417622166e-05,
    7.2231547062859186e-05,
    8.2488096781270552e-05,
    9.3922574360894714e-05,
    0.00010663935014249239,
    0.00012074935704431384,
    0.00013637033114566037,
    0.00015362705361437556,
    0.00017265159376116319,
    0.00019358355300152532,
    0.00021657030950414936,
    0.00024176726330274589,
    0.00026933808164665302,
    0.00029945494436397856,
    0.00033229878900964822,
    0.00036805955556947173,
    0.00040693643049022246,
    0.00044913808980476044,
    0.00049488294112040609,
    0.00054439936423809746,
    0.00059792595016933939,
    0.00065571173831757159,
    0.00071801645159035824,
    0.0007851107292087189,
    0.00085727635697999192,
    0.00093480649480084347,
    0.0010180059011574035,
    0.0011071911543900322,
    0.0012026908704908916,
    0.0013048459172033176,
    0.0014140096241929607,
    0.0015305479890617799,
    0.0016548398789772523,
    0.0017872772276905731,
    0.001928265227719192,
    0.0020782225174707462,
    0.0022375813630873148,
    0.002406787834790924,
    0.0025863019775133945,
    0.0027765979755959118,
    0.002978164311346153,
    0.0031915039172433764,
    0.0034171343215846244,
    0.0036555877873680367,
    0.0039074114442122845,
    0.0041731674131142878,
    0.004453432923850636,
    0.0047488004248315615,
    0.005059877685219847,
    0.0053872878891307379,
    0.0057316697217327109,
    0.0060936774470729104,
    0.0064739809774550967,
    0.006873265934202134,
    0.0072922336996393607,
    0.0077316014601395743,
    0.0081921022400749336,
    0.0086744849265256892,
    0.0091795142846004558,
    0.0097079709632275756,
    0.01026065149128211,
    0.010838368263918092,
    0.011441949518980827,
    0.012072239303379331,
    0.012730097429304364,
    0.013416399420182985,
    0.014132036446266113,
    0.014877915249751253,
    0.01565495805934821,
    0.016464102494201528,
    0.017306301457089175,
    0.018182523016823039,
    0.019093750279782832,
    0.020040981250521013,
    0.02102522868138269,
    0.022047519911090482,
    0.023108896692250813,
    0.024210415007744315,
    0.025353144875969479,
    0.02653817014491517,
    0.02776658827504401,
    0.029039510110975286,
    0.030358059641962563,
    0.031723373751167745,
    0.033136601953740033,
    0.03459890612371487,
    0.036111460209754621,
    0.037675449939759478,
    0.039292072514383762,
    0.040962536289499647,
    0.042688060447656814,
    0.044469874658593603,
    0.046309218728861817,
    0.048207342240633978,
    0.050165504179768804,
    0.052184972553217274,
    0.054267023995858303,
    0.056412943366859897,
    0.058624023335668171,
    0.06090156395773344,
    0.063246872240089028,
    0.065661261696905129,
    0.068146051895146609,
    0.070702567990470172,
    0.073332140253502529,
    0.076036103586648099,
    0.07881579703158062,
    0.081672563267579851,
    0.08460774810088019,
    0.08762269994520501,
    0.090718769293665805,
    0.093897308182211994,
    0.097159669644822713,
    0.10050720716063813,
    0.10394127409323346,
    0.10746322312224472,
    0.11107440566756094,
    0.11477617130630278,
    0.11856986718281355,
    0.12245683741189349,
    0.12643842247551362,
    0.13051595861325083,
    0.13469077720669043,
    0.13896420415804808,
    0.14333755926326719,
    0.14781215557985269,
    0.15238929878970758,
    0.15707028655724134,
    0.1618564078830258,
    0.16674894245327648,
    0.17174915998544268,
    0.17685831957019316,
    0.18207766901008776,
    0.18740844415522984,
    0.19285186823619718,
    0.19840915119455291,
    0.20408148901124099,
    0.20987006303317418,
    0.21577603929832523,
    0.22180056785963501,
    0.22794478210805377,
    0.23420979809503445,
    0.24059671385479928,
    0.24710660872670318,
    0.2537405426780191,
    0.26049955562747351,
    0.26738466676985995,
    0.2743968739020628,
    0.28153715275082186,
    0.28880645630257279,
    0.29620571413569602,
    0.303735831755511,
    0.31139768993235112,
    0.3191921440430559,
    0.32712002341621799,
    0.33518213068152158,
    0.3433792411235102,
    0.35171210204012071,
    0.3601814321063202,
    0.36878792074318301,
    0.37753222749274279,
    0.38641498139895503,
    0.39543678039510394,
    0.40459819069798669,
    0.41389974620920605,
    0.42334194792390206,
    0.43292526334725012,
    0.44265012591905317,
    0.45251693444675078,
    0.462526052547169,
    0.4726778080973294,
    0.48297249269463538,
    0.49341036112675013,
    0.50399163085147791,
    0.51471648148695681,
    0.52558505431246905,
    0.53659745178017071,
    0.54775373703803909,
    0.55905393346433196,
    0.57049802421385121,
    0.58208595177629552,
    0.5938176175469867,
    0.60569288141024635,
    0.61771156133569816,
    0.6298734329877631,
    0.6421782293486139,
    0.65462564035484605,
    0.66721531254812183,
    0.6799468487400333,
    0.69281980769143181,
    0.7058337038064576,
    0.71898800684150621,
    0.73228214162935445,
    0.74571548781866892,
    0.75928737962911175,
    0.7729971056222491,
    0.7868439084884683,
    0.80082698485009651,
    0.81494548508091003,
    0.82919851314221837,
    0.84358512643569661,
    0.85810433567313538,
    0.87275510476327078,
    0.88753635071584713,
    0.90244694356306143,
    0.91748570629852833,
    0.93265141483389846,
    0.94794279797325587,
    0.96335853740541078,
    0.97889726771419794,
    0.99455757640688358,
    1.0103380039607734,
    1.0262370438881094,
    1.0422531428193336,
    1.0583847006047875,
    1.0746300704349125,
    1.090987558979003,
    1.107455426542558,
    1.1240318872432713,
    1.140715109205686,
    1.1575032147745385,
    1.1743942807468011,
    1.1913863386224295,
    1.2084773748738107,
    1.2256653312338979,
    1.2429481050030133,
    1.2603235493742859,
    1.27778947377769,
    1.2953436442426358,
    1.3129837837790561,
    1.3307075727769284,
    1.3485126494241568,
    1.3663966101427374,
    1.3843570100431124,
    1.4023913633966207,
    1.4204971441259375,
    1.4386717863133844,
    1.4569126847269946,
    1.4752171953641948,
    1.4935826360129689,
    1.5120062868303543,
    1.530485390938114,
    1.5490171550354206,
    1.5675987500283819,
    1.5862273116762238,
    1.6048999412539453,
    1.6236137062312483,
    1.6423656409675373,
    1.6611527474227774,
    1.67997199588399,
    1.6988203257071621,
    1.7176946460743285,
    1.7365918367655895,
    1.7555087489458125,
    1.774442205965759,
    1.7933890041773757,
    1.8123459137629756,
    1.8313096795780328,
    1.8502770220073035,
    1.8692446378339858,
    1.8882092011216132,
    1.9071673641083837,
    1.9261157581136088,
    1.9450509944559651,
    1.9639696653832275,
    1.9828683450131519,
    2.0017435902851721,
    2.0205919419225711,
    2.0394099254047799,
    2.058194051949449,
    2.0769408195039398,
    2.0956467137458694,
    2.1143082090923442,
    2.1329217697175085,
    2.1514838505780305,
    2.1699908984461493,
    2.1884393529498909,
    2.2068256476200667,
    2.2251462109436617,
    2.2433974674232151,
    2.261575838641789,
    2.2796777443331231,
    2.2976996034565715,
    2.315637835276402,
    2.3334888604450517,
    2.3512491020899215,
    2.3689149869032868,
    2.3864829462349082,
    2.4039494171869134,
    2.4213108437105313,
    2.4385636777042459,
    2.4557043801129446,
    2.4727294220276272,
    2.4896352857852526,
    2.5064184660682765,
    2.5230754710034646,
    2.5396028232595342,
    2.5559970611431941,
    2.5722547396931543,
    2.588372431771659,
    2.6043467291531242,
    2.6201742436094295,
    2.6358516079914489,
    2.6513754773063694,
    2.6667425297903828,
    2.6819494679763101,
    2.6969930197557308,
    2.7118699394351951,
    2.7265770087860886,
    2.7411110380877264,
    2.7554688671632568,
    2.7696473664079537,
    2.7836434378094839,
    2.7974540159597292,
    2.8110760690577625,
    2.8245065999035592,
    2.8377426468820461,
    2.8507812849370828,
    2.8636196265349838,
    2.8762548226171787,
    2.8886840635416307,
    2.9009045800126194,
    2.9129136439985133,
    2.9247085696371498,
    2.9362867141284581,
    2.9476454786139508,
    2.9587823090427254,
    2.969694697023622,
    2.9803801806631793,
    2.9908363453890527,
    3.0010608247585431,
    3.0110513012519129,
    3.0208055070501563,
    3.0303212247968996,
    3.0395962883441241,
    3.0486285834813951,
    3.0574160486482955,
    3.0659566756297751,
    3.0742485102341179,
    3.0822896529532531,
    3.0900782596051313,
    3.0976125419578997,
    3.1048907683356184,
    3.1119112642052627,
    3.1186724127447696,
    3.125172655391891,
    3.1314104923736226,
    3.1373844832159938,
    3.1430932472339967,
    3.1485354640014598,
    3.153709873800663,
    3.1586152780515073,
    3.1632505397200643,
    3.1676145837063303,
    3.1717063972110213,
    3.1755250300812645,
    3.1790695951350285,
    3.1823392684641703,
    3.1853332897159601,
    3.1880509623529751,
    3.1904916538912489,
    3.1926547961165759,
    3.1945398852788851,
    3.1961464822645977,
    3.1974742127468958,
    3.1985227673138423,
    3.199291901574298,
    3.1997814362415857,
    3.1999912571948785,
    3.1999213155182713,
    3.199571627517531,
    3.1989422747145131,
    3.1980334038192484,
    3.196845226679712,
    3.1953780202092914,
    3.1936321262919929,
    3.1916079516654157,
    3.1893059677815501,
    3.1867267106454569,
    3.1838707806318918,
    3.1807388422799545,
    3.1773316240658516,
    3.1736499181538611,
    3.1696945801256087,
    3.1654665286877663,
    3.1609667453582992,
    3.1561962741313847,
    3.1511562211211519,
    3.1458477541843872,
    3.1402721025223617,
    3.1344305562619499,
    3.1283244660162137,
    3.1219552424246348,
    3.115324355673188,
    3.1084333349944555,
    3.1012837681479914,
    3.0938773008811506,
    3.0862156363706119,
    3.0783005346448222,
    3.0701338119876072,
    3.0617173403231921,
    3.0530530465828947,
    3.0441429120537475,
    3.0349889717093226,
    3.0255933135230393,
    3.0159580777642354,
    3.0060854562772987,
    2.9959776917441525,
    2.9856370769304057,
    2.9750659539154767,
    2.9642667133070106,
    2.9532417934399153,
    2.9419936795603454,
    2.9305249029949714,
    2.9188380403058822,
    2.906935712431459,
    2.8948205838135834,
    2.8824953615115367,
    2.8699627943029502,
    2.8572256717721847,
    2.8442868233865015,
    2.8311491175604186,
    2.8178154607086188,
    2.8042887962878122,
    2.7905721038279347,
    2.7766683979530811,
    2.7625807273925767,
    2.7483121739825811,
    2.7338658516586398,
    2.7192449054395862,
    2.7044525104032107,
    2.6894918706541104,
    2.674366218284137,
    2.6590788123258653,
    2.6436329376995009,
    2.6280319041536586,
    2.6122790452004274,
    2.5963777170451632,
    2.5803312975114272,
    2.5641431849615084,
    2.5478167972129566,
    2.5313555704515665,
    2.5147629581412354,
    2.4980424299311372,
    2.4811974705606477,
    2.4642315787624427,
    2.4471482661642234,
    2.4299510561894819,
    2.4126434829577561,
    2.3952290901848001,
    2.3777114300830982,
    2.3600940622631628,
    2.3423805526360355,
    2.3245744723174271,
    2.3066793965339167,
    2.288698903531639,
    2.2706365734878795,
    2.2524959874259967,
    2.2342807261340938,
    2.2159943690878463,
    2.1976404933779095,
    2.1792226726423025,
    2.1607444760041838,
    2.1422094670154181,
    2.1236212026063339,
    2.1049832320420649,
    2.086299095885872,
    2.067572324969829,
    2.0488064393732563,
    2.0300049474092847,
    2.0111713446199189,
    1.9923091127799741,
    1.9734217189102505,
    1.9545126143003018,
    1.9355852335411576,
    1.9166429935683473,
    1.897689292715566,
    1.8787275097793266,
    1.8597610030949256,
    1.8407931096240533,
    1.821827144054365,
    1.8028663979113311,
    1.783914138682674,
    1.7649736089556887,
    1.7460480255677508,
    1.7271405787702918,
    1.70825443140653,
    1.6893927181032269,
    1.6705585444767403,
    1.6517549863536347,
    1.6329850890060977,
    1.6142518664024172,
    1.595558300472748,
    1.5769073403904084,
    1.5583019018689233,
    1.5397448664750337,
    1.52123908095788,
    1.502787356594558,
    1.4843924685522418,
    1.466057155267058,
    1.4477841178398851,
    1.4295760194492495,
    1.4114354847814783,
    1.3933650994782556,
    1.3753674096017352,
    1.3574449211173345,
    1.3396000993943458,
    1.3218353687244779,
    1.3041531118584404,
    1.2865556695606726,
    1.2690453401823083,
    1.2516243792524662,
    1.2342949990879353,
    1.2170593684213289,
    1.1999196120477651,
    1.1828778104901221,
    1.1659359996829137,
    1.1490961706748182,
    1.1323602693498835,
    1.1157301961674302,
    1.0992078059206547,
    1.0827949075139383,
    1.0664932637588489,
    1.0503045911888231,
    1.0342305598924988,
    1.0182727933656675,
    1.0024328683818051,
    0.98671231488112821,
    0.97111261587812125,
    0.95563520738746532,
    0.94028147836829645,
    0.92505277068670866,
    0.90995037909641452,
    0.89497555123746197,
    0.88012948765290411,
    0.86541334182330609,
    0.85082822021896953,
    0.83637518236974517,
    0.82205524095229676,
    0.80786936189467418,
    0.79381846449804261,
    0.77990342157541226,
    0.76612505960720179,
    0.7524841589134641,
    0.73898145384259484,
    0.72561763297633963,
    0.71239333935090499,
    0.69930917069397758,
    0.68636567967744344,
    0.67356337418559764,
    0.6609027175986264,
    0.6483841290911373,
    0.6360079839455095,
    0.62377461387982769,
    0.61168430739015889,
    0.59973731010692699,
    0.58793382516513149,
    0.576274013588156,
    0.56475799468490218,
    0.55338584645998479,
    0.54215760603671459,
    0.53107327009259342,
    0.52013279530704226,
    0.50933609882107544,
    0.49868305870863416,
    0.48817351445928492,
    0.47780726747198721,
    0.46758408155962972,
    0.45750368346403181,
    0.44756576338110271,
    0.43776997549584834,
    0.42811593852691343,
    0.41860323628034157,
    0.40923141821223619,
    0.010050506338833465,
    0.010618738310925543,
    0.011212522321601155,
    0.011832696493953285,
    0.012480114733971913,
    0.013155646680359803,
    0.013860177644019462,
    0.014594608537111285,
    0.015359855791588573,
    0.016156851267120957,
    0.016986542148323517,
    0.017849890831214976,
    0.01874787479883417,
    0.019681486485950278,
    0.02065173313280828,
    0.021659636627857418,
    0.02270623333941665,
    0.023792573936237488,
    0.024919723196930886,
    0.026088759808231374,
    0.027300776152078061,
    0.028556878081498639,
    0.029858184685289127,
    0.031205828041488637,
    0.032600952959655073,
    0.034044716711954422,
    0.03553828875308275,
    0.037082850429047076,
    0.038679594674837559,
    0.040329725701030592,
    0.04203445866936887,
    0.043795019357371311,
    0.045612643812032491,
    0.047488577992678013,
    0.049424077403048862,
    0.051420406712694627,
    0.053478839367762251,
    0.055600657191273446,
    0.057787149972990871,
    0.060039615048979582,
    0.062359356870977094,
    0.064747686565691853,
    0.067205921484156578,
    0.069735384741269307,
    0.072337404745661749,
    0.075013314720040566,
    0.077764452212154056,
    0.080592158596542618,
    0.083497778567237987,
    0.086482659621582164,
    0.089548151535343373,
    0.092695605829312216,
    0.095926375227567445,
    0.09924181310760638,
    0.10264327294254126,
    0.1061321077355683,
    0.10970966944692187,
    0.11337730841353212,
    0.11713637276160954,
    0.12098820781238562,
    0.12493415548124381,
    0.12897555367048055,
    0.13311373565594081,
    0.13735002946777794,
    0.1416857572655924,
    0.14612223470820865,
    0.15066077031835406,
    0.15530266484250854,
    0.16004921060619756,
    0.16490169086500617,
    0.16986137915159474,
    0.17492953861900257,
    0.18010742138052785,
    0.18539626784647764,
    0.19079730605808423,
    0.19631175101888793,
    0.20194080402389006,
    0.20768565198678232,
    0.21354746676556288,
    0.21952740448685101,
    0.22562660486921629,
    0.23184619054583994,
    0.23818726638682849,
    0.24465091882150297,
    0.25123821516098777,
    0.25795020292142617,
    0.2647879091481512,
    0.27175233974114144,
    0.27884447878209406,
    0.28606528786344759,
    0.2934157054196882,
    0.30089664606127542,
    0.30850899991152214,
    0.31625363194676609,
    0.32413138134016889,
    0.33214306080948103,
    0.34028945596910903,
    0.34857132468682239,
    0.35698939644543765,
    0.36554437170981613,
    0.37423692129951053,
    0.3830676857673973,
    0.3920372747846278,
    0.40114626653223179,
    0.41039520709970595,
    0.41978460989091726,
    0.42931495503765033,
    0.43898668882112613,
    0.44880022310181694,
    0.45875593475788068,
    0.46885416513253553,
    0.47909521949069284,
    0.48947936648516388,
    0.50000683763275355,
    0.51067782680055029,
    0.52149248970271966,
    0.53245094340810373,
    0.54355326585892683,
    0.55479949540090434,
    0.56618963032504499,
    0.57772362842143754,
    0.58940140654530349,
    0.60122284019559824,
    0.61318776310643364,
    0.62529596685159561,
    0.63754720046241964,
    0.64994117005928898,
    0.66247753849700874,
    0.67515592502430832,
    0.68797590495771721,
    0.70093700937005521,
    0.71403872479377029,
    0.72728049293935337,
    0.74066171042905338,
    0.7541817285461091,
    0.76783985299970747,
    0.78163534370587429,
    0.79556741458449487,
    0.80963523337265531,
    0.82383792145448953,
    0.83817455370771143,
    0.85264415836700147,
    0.86724571690441254,
    0.88197816392695338,
    0.89684038709149771,
    0.91183122703716335,
    0.92694947733529542,
    0.94219388445718233,
    0.95756314775962437,
    0.97305591948846792,
    0.9886708048002103,
    1.0044063618017711,
    1.0202611016085223,
    1.0362334884206541,
    1.0523219396179551,
    1.0685248258730666,
    1.0848404712832731,
    1.1012671535208753,
    1.1178031040021854,
    1.1344465080751813,
    1.1511955052258389,
    1.168048189303162,
    1.185002608762916,
    1.2020567669300635,
    1.2192086222798943,
    1.23645608873783,
    1.2537970359978781,
    1.2712292898597004,
    1.2887506325842519,
    1.3063588032679394,
    1.3240514982352372,
    1.3418263714496943,
    1.3596810349432524,
    1.3776130592637907,
    1.3956199739408028,
    1.4136992679691021,
    1.4318483903104451,
    1.4500647504129542,
    1.4683457187482074,
    1.4866886273658655,
    1.505090770465684,
    1.5235494049867651,
    1.542061751213881,
    1.5606249934007048,
    1.5792362804097688,
    1.5978927263689651,
    1.6165914113443947,
    1.6353293820293675,
    1.6541036524493369,
    1.6729112046825587,
    1.6917489895962468,
    1.7106139275979941,
    1.7295029094022181,
    1.7484127968113854,
    1.7673404235117602,
    1.7862825958834152,
    1.8052360938242362,
    1.8241976715876445,
    1.8431640586337545,
    1.8621319604936795,
    1.881098059646684,
    1.9000590164098865,
    1.9190114698402001,
    1.9379520386481961,
    1.9568773221235696,
    1.9757839010718796,
    1.9946683387622315,
    2.0135271818855602,
    2.0323569615231718,
    2.0511541941251923,
    2.0699153824985679,
    2.0886370168042592,
    2.1073155755632587,
    2.1259475266710646,
    2.1445293284202376,
    2.1630574305306531,
    2.1815282751870777,
    2.1999382980836688,
    2.2182839294750134,
    2.2365615952333089,
    2.2547677179112808,
    2.2728987178104445,
    2.2909510140542939,
    2.3089210256660122,
    2.3268051726502965,
    2.3445998770788705,
    2.3623015641792766,
    2.3799066634265249,
    2.3974116096371723,
    2.4148128440654113,
    2.4321068155007435,
    2.4492899813668028,
    2.4663588088209076,
    2.4833097758539027,
    2.5001393723898677,
    2.5168441013852503,
    2.5334204799269977,
    2.5498650403292489,
    2.5661743312281571,
    2.5823449186744014,
    2.5983733872229617,
    2.6142563410197153,
    2.6299904048844298,
    2.6455722253897105,
    2.6609984719354784,
    2.6762658378185464,
    2.6913710412968568,
    2.706310826647965,
    2.7210819652213325,
    2.7356812564840114,
    2.7501055290592911,
    2.7643516417578953,
    2.778416484601308,
    2.7922969798368107,
    2.8059900829438225,
    2.8194927836311341,
    2.8328021068246274,
    2.8459151136450793,
    2.8588289023756519,
    2.871540609418672,
    2.8840474102413105,
    2.8963465203097734,
    2.9084351960116224,
    2.9203107355658471,
    2.9319704799203139,
    2.9434118136362244,
    2.9546321657592181,
    2.9656290106767629,
    2.9763998689614786,
    2.9869423082000455,
    2.9972539438073551,
    3.0073324398255736,
    3.0171755097077755,
    3.0267809170858349,
    3.0361464765222514,
    3.0452700542455995,
    3.0541495688692941,
    3.0627829920933847,
    3.0711683493890694,
    3.079303720665663,
    3.0871872409197296,
    3.0948171008661132,
    3.1021915475506066,
    3.1093088849439998,
    3.1161674745172658,
    3.1227657357976342,
    3.1291021469053342,
    3.1351752450707724,
    3.1409836271319356,
    3.1465259500118066,
    3.1518009311756003,
    3.1568073490676225,
    3.1615440435275697,
    3.1660099161861033,
    3.1702039308395222,
    3.1741251138033855,
    3.1777725542449309,
    3.1811454044941581,
    3.1842428803334362,
    3.1870642612655247,
    3.1896088907598865,
    3.1918761764771983,
    3.1938655904719564,
    3.1955766693731,
    3.1970090145425702,
    3.198162292211745,
    3.1990362335956841,
    3.1996306349851471,
    3.1999453578163353,
    3.1999803287183353,
    3.1997355395382416,
    3.1992110473439492,
    3.1984069744046111,
    3.1973235081487754,
    3.1959609011002139,
    3.1943194707914722,
    3.1923995996551735,
    3.1902017348931309,
    3.18772638832331,
    3.1849741362047159,
    3.1819456190402762,
    3.1786415413577971,
    3.1750626714690933,
    3.1712098412073817,
    3.1670839456430615,
    3.1626859427779874,
    3.1580168532183714,
    3.1530777598264494,
    3.147869807351054,
    3.1423942020372535,
    3.1366522112152158,
    3.1306451628684733,
    3.124374445181763,
    3.1178415060686375,
    3.1110478526790395,
    3.1039950508870486,
    3.0966847247590095,
    3.0891185560022665,
    3.0812982833947369,
    3.073225702195554,
    3.0649026635370307,
    3.0563310737981935,
    3.0475128939601501,
    3.0384501389435585,
    3.0291448769284663,
    3.0195992286568165,
    3.0098153667178922,
    2.9997955148170128,
    2.98954194702777,
    2.979056987028128,
    2.9683430073206916,
    2.95740242843747,
    2.9462377181294688,
    2.9348513905414357,
    2.9232460053721114,
    2.9114241670203205,
    2.8993885237172656,
    2.8871417666453731,
    2.8746866290440565,
    2.8620258853027662,
    2.8491623500416958,
    2.8360988771805227,
    2.8228383589955648,
    2.8093837251657399,
    2.7957379418077148,
    2.7819040105006443,
    2.7678849673008927,
    2.753683881747139,
    2.7393038558562779,
    2.7247480231105152,
    2.7100195474360764,
    2.6951216221739407,
    2.6800574690430143,
    2.6648303370961659,
    2.6494435016695479,
    2.633900263325621,
    2.6182039467903127,
    2.6023578998847361,
    2.5863654924518986,
    2.5702301152788305,
    2.5539551790145625,
    2.5375441130843899,
    2.5210003646008516,
    2.504327397271862,
    2.4875286903064251,
    2.4706077373183688,
    2.453568045228534,
    2.4364131331658498,
    2.4191465313677289,
    2.401771780080217,
    2.3842924284583313,
    2.3667120334670098,
    2.3490341587831147,
    2.3312623736989053,
    2.3134002520274164,
    2.2954513710101616,
    2.2774193102275913,
    2.2593076505127141,
    2.2411199728683178,
    2.2228598573881877,
    2.2045308821827496,
    2.186136622309538,
    2.1676806487089051,
    2.1491665271453675,
    2.1305978171549955,
    2.11197807099924,
    2.0933108326255914,
    2.0745996366354569,
    2.0558480072596472,
    2.0370594573418446,
    2.0182374873304365,
    1.9993855842790829,
    1.980507220856381,
    1.9616058543649972,
    1.9426849257706147,
    1.9237478587410526,
    1.9047980586959026,
    1.8858389118670207,
    1.866873784370213,
    1.8479060212884404,
    1.8289389457668674,
    1.8099758581200722,
    1.7910200349517258,
    1.7720747282870484,
    1.7531431647183364,
    1.7342285445638548,
    1.7153340410403772,
    1.6964627994496506,
    1.6776179363790569,
    1.6588025389167329,
    1.6400196638814066,
    1.6212723370671975,
    1.6025635525036228,
    1.5838962717310423,
    1.5652734230917726,
    1.5466979010370834,
    1.5281725654502911,
    1.5097002409861537,
    1.4912837164267581,
    1.4729257440540908,
    1.4546290390394712,
    1.4363962788500177,
    1.4182301026723092,
    1.4001331108533992,
    1.3821078643593279,
    1.3641568842512706,
    1.346282651179453,
    1.3284876048949557,
    1.3107741437795197,
    1.2931446243934606,
    1.2756013610417831,
    1.2581466253585913,
    1.2407826459098668,
    1.2235116078146899,
    1.206335652384966,
    1.1892568767837097,
    1.1722773337019341,
    1.1553990310541804,
    1.138623931692718,
    1.1219539531404332,
    1.10539096734242,
    1.0889368004362754,
    1.0725932325410921,
    1.0563619975651384,
    1.0402447830321999,
    1.0242432299265554,
    1.0083589325565432,
    0.99259343843668013,
    0.97694824818826709,
    0.96142481545842651,
    0.94602454685749582,
    0.93074880191469955,
    0.91559889305201125,
    0.90057608557611146,
    0.8856815976883381,
    0.87091660051251785,
    0.85628221814056216,
    0.84177952769569864,
    0.82740955941320804,
    0.81317329673852345,
    0.79907167644254384,
    0.78510558875400815,
    0.77127587750876514,
    0.7575833403157719,
    0.74402872873964265,
    0.73061274849956592,
    0.71733605968439973,
    0.70419927698374896,
    0.69120296993482133,
    0.67834766318485351,
    0.66563383676889165,
    0.65306192640270566,
    0.64063232379060864,
    0.62834537694795023,
    0.61620139053804335,
    0.60420062622328075,
    0.59234330303019311,
    0.58062959772819056,
    0.56905964522173158,
    0.55763353895565104,
    0.54635133133337999,
    0.53521303414778154,
    0.5242186190243241,
    0.51336801787631003,
    0.50266112337187085,
    0.49209778941243809,
    0.48167783162239536,
    0.47140102784961158,
    0.46126711867655384,
    0.45127580794167371,
    0.44142676327075764,
    0.43171961661793012,
    0.4221539648159946,
    0.41272937013579353,
    0.40344536085426908,
    0.39430143183090083,
    0.38529704509219598,
    0.37643163042390565,
    0.36770458597063904,
    0.35911527884254429,
    0.35066304572872559,
    0.34234719351706261,
    0.33416699992009846,
    0.32612171410666163,
    0.31821055733888481,
    0.31043272361428526,
    0.30278738031256941,
    0.29527366884682382,
    0.28789070531875643,
    0.28063758117764981,
    0.27351336388268943,
    0.26651709756833153,
    0.259647803712373,
    0.25290448180638908,
    0.24628611002820441,
    0.23979164591606375,
    0.2334200270441712,
    0.22717017169926645,
    0.22104097955791008,
    0.21503133236415001,
    0.20914009460724414,
    0.20336611419911618,
    0.19770822315122333,
    0.19216523825051754,
    0.1867359617341843,
    0.18141918196284579,
    0.17621367409191749,
    0.1711182007408116,
    0.16613151265968221,
    0.16125234939341176,
    0.15647943994254115,
    0.15181150342084934,
    0.14724724970929245,
    0.14278538010601602,
    0.13842458797215773,
    0.13416355937316282,
    0.1300009737153377,
    0.12593550437737255,
    0.12196581933656764,
    0.11809058178950262,
    0.11430845076689364,
    0.11061808174238681,
    0.10701812723504232,
    0.10350723740526846,
    0.10008406064396963,
    0.096747244154678266,
    0.093495434528445401,
    0.090327278311270409,
    0.08724142256385628,
    0.08423651541348158,
    0.081311206597787206,
    0.078464148000280745,
    0.075693994177367885,
    0.072999402876726072,
    0.070379035546841676,
    0.06783155783753797,
    0.065355640091327674,
    0.062949957825429795,
    0.060613192204296971,
    0.058344030502505966,
    0.056141166557870062,
    0.054003301214639005,
    0.051929142756658352,
    0.049917407330366689,
    0.047966819357515962,
    0.046076111937506423,
    0.04424402723923479,
    0.04246931688236049,
    0.040750742307901829,
    0.039087075138080504,
    0.037477097525339703,
    0.035919602490467606,
    0.034413394249765097,
    0.032957288531203041,
    0.031550112879521294,
    0.030190706950228335,
    0.028877922792467262,
    0.027610625120720374,
    0.026387691575331527,
    0.025208012971831936,
    0.024070493539061884,
    0.022974051146087335,
    0.021917617517917168,
    0.020900138440033144,
    0.019920573951751449,
    0.018977898528440985,
    0.018071101252630093,
    0.017199185974039768,
    0.016361171458587811,
    0.01555609152641455,
    0.014782995178987108,
    0.014040946715345249,
    0.013329025837558024,
    0.012646327745466429,
    0.011991963220793189,
    0.011365058700706797,
    0.010764756340932602,
    0.01019021406850954,
    0.0096406056242967272,
    0.0091151205953396586,
    0.0086129644372112506,
    0.0081333584864482968,
    0.0076755399632091777,
    0.0072387619642838904,
    0.0068222934465924289,
    0.0064254192013126132,
    0.0060474398187832503,
    0.0056876716443332934,
    0.0053454467251922832,
    0.0050201127486418874,
    0.0047110329715727411,
    0.0044175861416151001,
    0.0041391664100159566,
    0.0038751832364393229,
    0.0036250612858703089,
    0.0033882403178073779,
    0.0031641750679308684,
    0.0029523351224393511,
    0.002752204785248825,
    0.0025632829382529862,
    0.0023850828948459527,
    0.0022171322469117954,
    0.0020589727054880925,
    0.0019101599353134196,
    0.0017702633834712591,
    0.0016388661023452441,
    0.0015155645671029209,
    0.0013999684879273605,
    0.0012917006172179367,
    0.0011903965519834293,
    0.0010957045316523156,
    0.0010072852315266552,
    0.00092481155210737825,
    0.00084796840452003709,
    0.00077645249227118363,
    0.00070997208956648846,
    0.00064824681642251943,
    0.00059100741080474846,
    0.00053799549802485984,
    0.00048896335763078439,
    0.00044367368802308279,
    0.00040189936903135858,
    0.0003634232226842766,
    0.00032803777240652133,
    0.00029554500087562886,
    0.00026575610677108364,
    0.00023849126064737753,
    0.00021357936016188953,
    0.00019085778488745742,
    0.00017017215093838181,
    0.00015137606563732605,
    0.00013433088244915544,
    0.0001189054564061973,
    0.00010497590024769898,
    9.2425341494417336e-05,
    8.1143680677290403e-05,
    7.1027350937022263e-05,
    6.1979079209155793e-05,
    5.3907649206818442e-05,
    4.6727666410802831e-05,
    4.0359325273990845e-05,
    3.4728178844346454e-05,
    2.9764911007792839e-05,
    2.5405111549253743e-05,
    2.1589054226979999e-05,
    1.8261478052002376e-05,
    1.5371371961152321e-05,
    1.2871763068576316e-05,
    1.0719508677038422e-05,
    8.8750922265627148e-06,
    7.3024233541140288e-06,
    5.9686422340550169e-06,
    4.8439283650520093e-06,
    3.9013139649343354e-06,
    3.1165021307442172e-06,
    2.4676899168497781e-06,
    1.9353964795348316e-06,
    1.5022964319287952e-06,
    1.1530585485011083e-06,
    8.7418995361983871e-07,
    6.5388592386670727e-07,
    4.8188542891350753e-07,
    3.4933253080092067e-07,
    2.4864375642307925e-07,
    1.7338155291306388e-07,
    1.1813393044897435e-07,
    7.840039176050733e-08,
    5.0484242315336365e-08,
    3.1391369806292411e-08,
    1.8735576147692692e-08,
    1.0650539725494381e-08,
    5.7084801346198377e-09,
    2.8455920812017893e-09,
    1.2943095310431482e-09,
    5.2245555171401646e-10,
    1.7932762787538146e-10,
    4.8762531101515555e-11,
    9.2191001647371264e-12,
    9.1153895556240427e-13,
    2.0070460109865819e-14,
    3.7766799476867912e-20,
    4.4431776340160974e-15,
    4.3016551135695328e-13,
    5.5901696770730152e-12,
    3.3510848222025947e-11,
    1.3284570765883041e-10,
    4.0689185961058063e-10,
    1.0446896742484216e-09,
    2.3591802024925592e-09,
    4.8323892002579967e-09,
    9.1676008339148947e-09,
    1.6348478414625042e-08,
    2.7705083803822596e-08,
    4.4986741456326458e-08,
    7.0441687424884305e-08,
    1.0690343804381717e-07,
    1.5788380744329988e-07,
    2.2767249752309759e-07,
    3.2144317853867609e-07,
    4.4536597302688385e-07,
    6.0672625042620812e-07,
    8.1404963443124693e-07,
    1.0772331198657922e-06,
    1.4076821906670362e-06,
    1.8184538254480979e-06,
    2.324405272050499e-06,
    2.9423484675155317e-06,
    3.6912099749967391e-06,
    4.5921963043080447e-06,
    5.6689644780564051e-06,
    6.9477977006471801e-06,
    8.4577859828776785e-06,
    1.0231011570352316e-05,
    1.230273901956444e-05,
    1.4711609761197786e-05,
    1.7499840986007548e-05,
    2.0713428684549728e-05,
    2.4402354668040408e-05,
    2.8620797393746427e-05,
    3.3427346414538111e-05,
    3.8885220268575506e-05,
    4.5062487621554561e-05,
    5.203229147051107e-05,
    5.9873076214869913e-05,
    6.8668817397237977e-05,
    7.8509253913372577e-05,
    8.9490122487815503e-05,
    0.00010171339420886787,
    0.00011528751291389518,
    0.00013032763521339584,
    0.00014695587193984318,
    0.00016530153080502168,
    0.00018550136004742349,
    0.00020769979284925461,
    0.00023204919230072137,
    0.00025871009668752875,
    0.00028785146487592404,
    0.00031965092156816406,
    0.00035429500219997118,
    0.00039197939725037613,
    0.00043290919573332195,
    0.00047729912763952773,
    0.00052537380509638126,
    0.00057736796201304709,
    0.00063352669197754322,
    0.00069410568417225627,
    0.00075937145707422743,
    0.00082960158970655595,
    0.00090508495020743423,
    0.00098612192148363762,
    0.001073024623715761,
    0.0011661171334831069,
    0.001265735699276894,
    0.001372228953171375,
    0.0014859581184235074,
    0.0016072972127730482,
    0.0017366332472162967,
    0.0018743664200282246,
    0.0020209103058093977,
    0.0021766920393358905,
    0.0023421524939923641,
    0.0025177464545705587,
    0.0027039427842177177,
    0.0029012245853218335,
    0.0031100893541231409,
    0.003331049128843968,
    0.0035646306311318461,
    0.0038113754006137597,
    0.0040718399223624866,
    0.0043465957470792162,
    0.0046362296037999948,
    0.00494134350493704,
    0.0052625548434695902,
    0.0056004964821027171,
    0.005955816834216407,
    0.0063291799364312224,
    0.006721265512621007,
    0.0071327690292073247,
    0.0075644017415747167,
    0.0080168907314503469,
    0.0084909789350962068,
    0.008987425162166789,
    0.0095070041050899206,
    1.9899494936611666,
    1.9710591975397103,
    1.9521476209348343,
    1.9332181962827502,
    1.9142743382452885,
    1.8953194427649196,
    1.8763568861308018,
    1.8573900240561849,
    1.838422190767498,
    1.8194566981054392,
    1.8004968346383814,
    1.7815458647884033,
    1.7626070279702433,
    1.743683537743475,
    1.7247785809781857,
    1.7058953170344469,
    1.6870368769558424,
    1.6682063626773278,
    1.6494068462476765,
    1.630641369066768,
    1.6119129411379594,
    1.5932245403357852,
    1.5745791116892063,
    1.5559795666806402,
    1.5374287825609791,
    1.5189296016808098,
    1.5004848308380301,
    1.4820972406420556,
    1.4637695648947995,
    1.4455044999886018,
    1.4273047043212719,
    1.409172797728409,
    1.3911113609331474,
    1.3731229350134682,
    1.3552100208872173,
    1.3373750788149501,
    1.319620527920722,
    1.3019487457309367,
    1.2843620677313485,
    1.2668627869423146,
    1.249453153512379,
    1.232135374330265,
    1.214911612655341,
    1.1977839877666194,
    1.1807545746303389,
    1.163825403586169,
    1.1469984600520726,
    1.1302756842478474,
    1.1136589709373657,
    1.0971501691895174,
    1.0807510821578545,
    1.064463466878931,
    1.0482890340893156,
    1.0322294480612555,
    1.0162863264569528,
    1.0004612402014135,
    0.98475571337381673,
    0.96917122311734594,
    0.9537091995674124,
    0.93837102579819898,
    0.92315803778743877,
    0.9080715243993368,
    0.89311272738553849,
    0.87828284140403401,
    0.86358301405588767,
    0.84901434593966774,
    0.83457789072344768,
    0.82027465523424281,
    0.8061055995647346,
    0.79207163719713569,
    0.77817363514403137,
    0.76441241410603622,
    0.75078874864609002,
    0.73730336738021363,
    0.72395695318454001,
    0.71075014341842491,
    0.69768353016343965,
    0.6847576604780381,
    0.67197303666768826,
    0.65933011657024676,
    0.646829313856355,
    0.63447099834462517,
    0.62225549633137978,
    0.61018309093470502,
    0.59825402245256898,
    0.5864684887347541,
    0.57482664556834473,
    0.56332860707650956,
    0.5519744461303091,
    0.5407641947732581,
    0.52969784465836567,
    0.51877534749737153,
    0.50799661552189379,
    0.49736152195619959,
    0.48686990150130277,
    0.47652155083009462,
    0.46631622909320475,
    0.45625365843528914,
    0.44633352452143732,
    0.43655547707338838,
    0.42691913041524254,
    0.4174240640283513,
    0.40806982311506845,
    0.39885591917103907,
    0.3897818305657047,
    0.38084700313069747,
    0.3720508507557963,
    0.36339275599211601,
    0.35487207066219756,
    0.34648811647666838,
    0.33824018565713759,
    0.33012754156499319,
    0.32214941933576396,
    0.31430502651871184,
    0.30659354372131598,
    0.29901412525831295,
    0.29156589980495501,
    0.28424797105414928,
    0.27705941837714104,
    0.26999929748740337,
    0.26306664110739825,
    0.25626045963787208,
    0.249579741829352,
    0.24302345545550799,
    0.23659054798804957,
    0.23027994727282458,
    0.22409056220679105,
    0.21802128341553378,
    0.21207098393099949,
    0.20623851986912628,
    0.20052273110704535,
    0.19492244195953515,
    0.18943646185441074,
    0.18406358600653369,
    0.17880259609013069,
    0.17365226090911168,
    0.16861133706508205,
    0.16367856962274591,
    0.15885269277240116,
    0.15413243048923106,
    0.14951649718909957,
    0.14500359838056245,
    0.14059243131281002,
    0.13628168561926052,
    0.13207004395652874,
    0.12795618263849753,
    0.12393877226522507,
    0.12001647834642495,
    0.11618796191926117,
    0.11245188016020426,
    0.10880688699070064,
    0.10525163367641147,
    0.10178476941978275,
    0.098404941945713822,
    0.095110798080096287,
    0.091900984321001342,
    0.088774147402298664,
    0.085728934849495661,
    0.082763995527591852,
    0.079877980180748598,
    0.077069541963580415,
    0.074337336963880368,
    0.071680024716597304,
    0.069096268708889699,
    0.066584736876086265,
    0.064144102088390331,
    0.061773042628170724,
    0.059470242657688736,
    0.057234392677116788,
    0.055064189972711015,
    0.052958339055006506,
    0.050915552086910384,
    0.048934549301574495,
    0.047014059409936179,
    0.045152819997822134,
    0.043349577912517041,
    0.04160308963870539,
    0.039912121663701602,
    0.038275450831890247,
    0.036691864688304945,
    0.035160161811281218,
    0.033679152134125388,
    0.032247657255748262,
    0.030864510740219179,
    0.029528558405202698,
    0.028238658599246919,
    0.026993682467899155,
    0.025792514208631375,
    0.024634051314564483,
    0.023517204806987199,
    0.02244089945667186,
    0.021404073993996079,
    0.02040568130788576,
    0.019444688633601451,
    0.018520077729396487,
    0.01763084504208182,
    0.016776001861538745,
    0.015954574464227107,
    0.0151656042457428,
    0.014408147842484552,
    0.013681277242496149,
    0.012984079885556306,
    0.012315658752594381,
    0.011675132444516031,
    0.011061635250528815,
    0.010474317206063425,
    0.0099123441403919724,
    0.0093748977140503276,
    0.0088611754461769918,
    0.0083703907318864422,
    0.0079017728498001621,
    0.0074545669598638162,
    0.0070280340915841238,
    0.0066214511228240309,
    0.0062341107492996668,
    0.0058653214449273794,
    0.0055144074131738404,
    0.0051807085295667802,
    0.0048635802755283789,
    0.0045623936636976919,
    0.0042765351549126935,
    0.0040054065670266413,
    0.003748424975737442,
    0.0035050226076125328,
    0.0032746467254955404,
    0.0030567595064845562,
    0.0028508379126753322,
    0.0026563735548660347,
    0.0024728725494233775,
    0.0022998553685130269,
    0.002136856683900068,
    0.0019834252045281241,
    0.0018391235080883414,
    0.001703527866791956,
    0.0015762280675625112,
    0.001456827226866001,
    0.001344941600399281,
    0.0012402003878590048,
    0.0011422455330151134,
    0.001050731519314532,
    0.00096532516124219739,
    0.00088570539166787433,
    0.00081156304540838545,
    0.00074260063923591772,
    0.00067853214856393614,
    0.00061908278104297072,
    0.0005639887472991147,
    0.00051299702904850162,
    0.00046586514482130375,
    0.00042236091352892361,
    0.00038226221610802407,
    0.00034535675547487169,
    0.00031144181502314546,
    0.00028032401589789225,
    0.00025181907327769139,
    0.00022575155189632516,
    0.00020195462103433782,
    0.00018026980920980731,
    0.00016054675879644971,
    0.00014264298079582759,
    0.00012642360998894284,
    0.00011176116069086036,
    9.8535283330235946e-05,
    8.6632522073707202e-05,
    7.5946073713055809e-05,
    6.6375548030860764e-05,
    5.7826729858039076e-05,
    5.0211343034214001e-05,
    4.3446816479262296e-05,
    3.7456052581673372e-05,
    3.2167198106506669e-05,
    2.7513417822760452e-05,
    2.3432671046867818e-05,
    1.9867491295815928e-05,
    1.6764769240044646e-05,
    1.4075539142822653e-05,
    1.1754768969225399e-05,
    9.761154344151924e-06,
    8.0569165350191635e-06,
    6.6076046308652799e-06,
    5.3819020855801785e-06,
    4.3514377888644257e-06,
    3.4906018242997683e-06,
    2.7763660695980148e-06,
    2.1881097896830123e-06,
    1.7074503687554709e-06,
    1.3180793228954007e-06,
    1.0056037300747248e-06,
    7.5739320968618454e-07,
    5.6243257884692561e-07,
    4.1118030780915306e-07,
    2.9543289180904611e-07,
    2.0819525161182019e-07,
    1.43557269868571e-07,
    9.6576565192507576e-08,
    6.3167600591600003e-08,
    3.9997217564790363e-08,
    2.4386681783027015e-08,
    1.4220320837801961e-08,
    7.8608290519492256e-09,
    4.0713088135724834e-09,
    1.9441123175161559e-09,
    8.3654198320026586e-10,
    3.1346216635332275e-10,
    9.6869098666591426e-11,
    2.2460277142474688e-11,
    3.2387874208147128e-12,
    1.8228615506138665e-13,
    5.9010301734039846e-16,
    2.7531946671350431e-17,
    6.6904361107310139e-14,
    1.7766267283204454e-12,
    1.4629206927328719e-11,
    6.9406064477957571e-11,
    2.38637222020269e-10,
    6.6416219032403316e-10,
    1.591762041529268e-09,
    3.4128343883299223e-09,
    6.7130772227209426e-09,
    1.2328141824679223e-08,
    2.1406209232259912e-08,
    3.5477439073566299e-08,
    5.6530233901383447e-08,
    8.7094256546424873e-08,
    1.3033013241858034e-07,
    1.9012576314070258e-07,
    2.7119917239973884e-07,
    3.7920779944878302e-07,
    5.2086415029425018e-07,
    7.0405771125818546e-07,
    9.3798302432000746e-07,
    1.2332738184180195e-06,
    1.6021430857320201e-06,
    2.0585289868775136e-06,
    2.6182464639225005e-06,
    3.299144435192743e-06,
    4.1212684409638122e-06,
    5.1070286043511921e-06,
    6.2813727670061508e-06,
    7.6719646546080615e-06,
    9.3093669226160702e-06,
    1.1227228928307496e-05,
    1.3462479070789775e-05,
    1.6055521536429841e-05,
    1.905043728300242e-05,
    2.2495189091819175e-05,
    2.6441830513166763e-05,
    3.0946718526556062e-05,
    3.6070729733569461e-05,
    4.1879479897490843e-05,
    4.8443546640415594e-05,
    5.5838695105168421e-05,
    6.4146106386106717e-05,
    7.3452608529759082e-05,
    8.3850909903244132e-05,
    9.5439834725536554e-05,
    0.00010832456055389618,
    0.00012261685751515516,
    0.00013843532906906826,
    0.00015590565408857477,
    0.00017516083003959833,
    0.00019634141704092525,
    0.00021959578258275416,
    0.00024508034668070005,
    0.00027295982724036787,
    0.0003034074854060836,
    0.00033660537066598669,
    0.00037274456548444708,
    0.00041202542923167462,
    0.00045465784117944036,
    0.00050086144233102274,
    0.00055086587585284055,
    0.00060491102587472247,
    0.00066324725442540615,
    0.00072613563626964975,
    0.0007938481914132765,
    0.00086666811504256447,
    0.00094489000466462935,
    0.0010288200842158417,
    0.0011187764249058578,
    0.0012150891625655289,
    0.0013181007112678049,
    0.0014281659729917233,
    0.0015456525431007291,
    0.0016709409114078473,
    0.0018044246586016775,
    0.0019465106478087602,
    0.0020976192110696,
    0.002258184330507514,
    0.0024286538139714968,
    0.0026094894649364708,
    0.0028011672464466025,
    0.0030041774388898314,
    0.0032190247913943559,
    0.0034462286666405691,
    0.0036863231788848141,
    0.0039398573249943619,
    0.0042073951082961586,
    0.0044895156550451999,
    0.0047868133233218077,
    0.0050998978041706458,
    0.0054293942147980081,
    0.0057759431836477276,
    0.0061402009271800066,
    0.0065228393181815288,
    0.0069245459454394228,
    0.0073460241646159464,
    0.0077879931401652067,
    0.0082511878781377744,
    0.0087363592497237061,
    0.009244274005389283,
    0.0097757147794676237,
    0.010331480085068373,
    0.010912384299176689,
    0.011519257637817014,
    0.012152946121162368,
    0.012814311528475284,
    0.013504231342772056,
    0.014223598685107419,
    0.014973322238382561,
    0.015754326160585048,
    0.016567549987375051,
    0.017413948523938218,
    0.018294491726031464,
    0.019210164570154023,
    0.020161966912782204,
    0.021150913338612489,
    0.022178032997763858,
    0.023244369431896485,
    0.024350980389210356,
    0.025498937628293743,
    0.026689326710797906,
    0.027923246782920918,
    0.029201810345690037,
    0.03052614301403864,
    0.0318973832646803,
    0.033316682172789283,
    0.03478520313750337,
    0.036304121596271645,
    0.037874624728076484,
    0.039497911145565875,
    0.041175190576138811,
    0.042907683532033232,
    0.044696620969472856,
    0.046543243936935833,
    0.048448803212615049,
    0.050414558931146533,
    0.052441780199689175,
    0.05453174470344576,
    0.056685738300721815,
    0.058905054607625698,
    0.061190994572519815,
    0.063544866040339446,
    0.06596798330690247,
    0.068461666663339499,
    0.071027241930780818,
    0.073666039985442508,
    0.076379396274260991,
    0.079168650321231493,
    0.082035145224611736,
    0.08498022714515939,
    0.088005244785576947,
    0.091111548861344441,
    0.094300491563126435,
    0.097573426010945305,
    0.10093170570031919,
    0.1043766839405684,
    0.10790971328550016,
    0.1115321449566869,
    0.11524532825955904,
    0.11905060999253865,
    0.12294933384944563,
    0.12694283981541346,
    0.13103246355655648,
    0.13521953580363627,
    0.13950538172997859,
    0.14389132032389873,
    0.14837866375589584,
    0.15296871674088361,
    0.15766277589572711,
    0.16246212909236135,
    0.16736805480677086,
    0.17238182146411329,
    0.17750468678027492,
    0.18273789710014873,
    0.1880826867329303,
    0.19354027728472958,
    0.1991118769888007,
    0.20479868003369461,
    0.21060186588964297,
    0.21652259863348414,
    0.22256202627244567,
    0.22872128006709952,
    0.23500147385380946,
    0.24140370336699205,
    0.2479290455615149,
    0.25457855793555795,
    0.26135327785426538,
    0.26825422187451775,
    0.27528238507115482,
    0.28243874036498168,
    0.28972423785289148,
    0.29713980414043967,
    0.3046863416772046,
    0.31236472809527133,
    0.32017581555117469,
    0.32812043007163921,
    0.33619937090345275,
    0.34441340986781188,
    0.35276329071947493,
    0.36124972851106085,
    0.36987340896282955,
    0.37863498783827954,
    0.38753509032589784,
    0.39657431042739566,
    0.40575321035276335,
    0.41507231992247468,
    0.42453213597717154,
    0.43413312179515584,
    0.44387570651801628,
    0.45376028458471229,
    0.46378721517443899,
    0.47395682165859099,
    0.48426939106214312,
    0.49472517353476164,
    0.50532438183195771,
    0.51606719080659091,
    0.52695373691102776,
    0.53798411771025767,
    0.54915839140626277,
    0.56047657637393755,
    0.57193865070884731,
    0.58354455178711284,
    0.59529417583770261,
    0.60718737752741025,
    0.61922396955879111,
    0.63140372228132624,
    0.64372636331607658,
    0.65619157719408705,
    0.66879900500879463,
    0.68154824408268633,
    0.69443884764845321,
    0.70747032454487457,
    0.72064213892766704,
    0.73395370999552156,
    0.74740441173154981,
    0.76099357266035372,
    0.77472047562092483,
    0.78858435755557565,
    0.80258440931509656,
    0.81671977548032815,
    0.83098955420032838,
    0.84539279704731107,
    0.85992850888852224,
    0.87459564777521526,
    0.88939312484887922,
    0.90431980426486525,
    0.9193745031335514,
    0.93455599147917712,
    0.94986299221647019,
    0.96529418114518539,
    0.98084818696266052,
    0.99652359129449342,
    1.0123189287434322,
    1.0282326869565626,
    1.0442633067108735,
    1.0604091820172656,
    1.076668660243068,
    1.0930400422531135,
    1.1095215825694185,
    1.1261114895495044,
    1.1428079255833881,
    1.159609007309262,
    1.1765128058478751,
    1.193517347055618,
    1.2106206117963081,
    1.227820536231659,
    1.2451150121304144,
    1.2625018871961138,
    1.2799789654134515,
    1.2975440074131837,
    1.3151947308555227,
    1.3329288108319584,
    1.3507438802854295,
    1.3686375304487679,
    1.3866073113013215,
    1.4046507320436574,
    1.4227652615902426,
    1.4409483290799818,
    1.4591973244044891,
    1.4775095987539655,
    1.4958824651805365,
    1.5143131991789038,
    1.5327990392841535,
    1.5513371876865543,
    1.569924810863176,
    1.5885590402261418,
    1.6072369727873277,
    1.6259556718393136,
    1.6447121676523759,
    1.6635034581873134,
    1.6823265098238815,
    1.7011782581046115,
    1.7200556084937735,
    1.7389554371512421,
    1.757874591721017,
    1.7768098921341344,
    1.7957581314257087,
    1.8147160765658317,
    1.8336804693040483,
    1.8526480270271242,
    1.8716154436298122,
    1.8905793903983192,
    1.9095365169061658,
    1.9284834519221283,
    1.9474168043299442,
    1.9663331640594566,
    1.9852291030288669,
    2.0041011760977603,
    2.0229459220305621,
    2.0417598644700767,
    2.0605395129207573,
    2.0792813637413481,
    2.0979819011465359,
    2.1166375982172436,
    2.1352449179191901,
    2.1538003141293451,
    2.1723002326698917,
    2.1907411123493108,
    2.2091193860102014,
    2.2274314815834404,
    2.2456738231482811,
    2.263842831997994,
    2.2819349277106422,
    2.299946529224584,
    2.3178740559182911,
    2.3357139286940702,
    2.3534625710652666,
    2.3711164102465392,
    2.3886718782467753,
    2.4061254129642302,
    2.4234734592834588,
    2.4407124701736214,
    2.4578389077877212,
    2.4748492445623582,
    2.4917399643175568,
    2.5085075633562415,
    2.5251485515629222,
    2.541659453501163,
    2.5580368095093915,
    2.5742771767946206,
    2.5903771305236463,
    2.6063332649112878,
    2.6221421943052361,
    2.6378005542670784,
    2.653305002649065,
    2.6686522206661882,
    2.6838389139631462,
    2.6988618136757547,
    2.7137176774863887,
    2.7284032906730236,
    2.7429154671514553,
    2.7572510505102734,
    2.7714069150381744,
    2.7853799667431942,
    2.7991671443634516,
    2.8127654203689816,
    2.8261718019542661,
    2.8393833320210446,
    2.8523970901510101,
    2.8652101935679948,
    2.8778197980892459,
    2.8902230990654112,
    2.9024173323088416,
    2.9143997750098336,
    2.926167746640437,
    2.9377186098454557,
    2.9490497713202704,
    2.9601586826751323,
    2.9710428412855627,
    2.9816997911285119,
    2.9921271236039333,
    3.0023224783414282,
    3.0122835439916398,
    3.0220080590020526,
    3.0314938123768917,
    3.0407386444207982,
    3.0497404474659753,
    3.0584971665825038,
    3.0670068002715336,
    3.0752674011410588,
    3.0832770765640021,
    3.0910339893183294,
    3.0985363582089342,
    3.1057824586710256,
    3.1127706233547752,
    3.1194992426909778,
    3.1259667654374881,
    3.1321716992062072,
    3.1381126109704014,
    3.1437881275521344,
    3.1491969360896208,
    3.1543377844842935,
    3.1592094818274079,
    3.1638108988059988,
    3.1681409680880241,
    3.1721986846865322,
    3.1759831063026991,
    3.1794933536475978,
    3.1827286107425574,
    3.1856881251979949,
    3.188371208470592,
    3.1907772360987221,
    3.1929056479160174,
    3.1947559482429955,
    3.1963277060566577,
    3.1976205551379953,
    3.1986341941973344,
    3.1993683869774738,
    3.1998229623345691,
    3.1999978142967294,
    3.1998929021002995,
    3.199508250203821,
    3.1988439482796567,
    3.1979001511832847,
    3.1966770789002794,
    3.1951750164709964,
    3.1933943138929965,
    3.1913353860012443,
    3.1889987123261392,
    3.1863848369294336,
    3.1834943682181054,
    3.1803279787362695,
    3.1768864049352046,
    3.1731704469216067,
    3.169180968184159,
    3.1649188952985416,
    3.1603852176110068,
    3.1555809869006413,
    3.15050731702047,
    3.1451653835175417,
    3.1395564232321571,
    3.1336817338764149,
    3.127542673592238,
    3.1211406604890768,
    3.114477172161477,
    3.1075537451867121,
    3.1003719746026954,
    3.0929335133663827,
    3.0852400717928976,
    3.0772934169756079,
    3.0690953721873995,
    3.060647816263391,
    3.0519526829653545,
    3.043011960328097,
    3.0338276899880818,
    3.0244019664945641,
    3.0147369366035304,
    3.0048347985547301,
    2.9946978013321055,
    2.9843282439079193,
    2.9737284744708989,
    2.9629008896387119,
    2.9518479336551025,
    2.940572097572018,
    2.9290759184170647,
    2.917361978346634,
    2.9054329037850524,
    2.8932913645501044,
    2.8809400729652928,
    2.8683817829591982,
    2.8556192891523118,
    2.8426554259317096,
    2.8294930665139528,
    2.8161351219966,
    2.8025845403987089,
    2.7888443056907342,
    2.7749174368142051,
    2.7608069866915903,
    2.7465160412267489,
    2.732047718296378,
    2.7174051667328629,
    2.7025915652989467,
    2.6876101216546311,
    2.6724640713167283,
    2.6571566766114882,
    2.6416912256207157,
    2.6260710311218092,
    2.610299429522148,
    2.5943797797882504,
    2.5783154623701381,
    2.5621098781213369,
    2.5457664472149428,
    2.5292886080561878,
    2.5126798161919441,
    2.4959435432175896,
    2.4790832756816812,
    2.4621025139888588,
    2.4450047713014262,
    2.4277935724400299,
    2.4104724527838788,
    2.3930449571709338,
    2.3755146387984984,
    2.3578850581246407,
    2.3401597817708777,
    2.3223423814265485,
    2.3044364327553031,
    2.2864455143041273,
    2.2683732064153332,
    2.2502230901419269,
    2.231998746166775,
    2.2137037537259854,
    2.1953416895369124,
    2.1769161267311925,
    2.1584306337932233,
    2.1398887735044823,
    2.1212941018940827,
    2.1026501671959714,
    2.0839605088131461,
    2.0652286562892894,
    2.0464581282882004,
    2.027652431581398,
    2.0088150600442742,
    1.9899494936611666,
    2.0088150600442742,
    2.027652431581398,
    2.0464581282882004,
    2.0652286562892894,
    2.0839605088131461,
    2.1026501671959714,
    2.1212941018940827,
    2.1398887735044823,
    2.1584306337932233,
    2.1769161267311925,
    2.1953416895369124,
    2.2137037537259854,
    2.231998746166775,
    2.2502230901419269,
    2.2683732064153332,
    2.2864455143041273,
    2.3044364327553031,
    2.3223423814265485,
    2.3401597817708777,
    2.3578850581246407,
    2.3755146387984984,
    2.3930449571709338,
    2.4104724527838788,
    2.4277935724400299,
    2.4450047713014262,
    2.4621025139888588,
    2.4790832756816812,
    2.4959435432175896,
    2.5126798161919441,
    2.5292886080561878,
    2.5457664472149428,
    2.5621098781213369,
    2.5783154623701381,
    2.5943797797882504,
    2.610299429522148,
    2.6260710311218092,
    2.6416912256207157,
    2.6571566766114882,
    2.6724640713167283,
    2.6876101216546311,
    2.7025915652989467,
    2.7174051667328629,
    2.732047718296378,
    2.7465160412267489,
    2.7608069866915903,
    2.7749174368142051,
    2.7888443056907342,
    2.8025845403987089,
    2.8161351219966,
    2.8294930665139528,
    2.8426554259317096,
    2.8556192891523118,
    2.8683817829591982,
    2.8809400729652928,
    2.8932913645501044,
    2.9054329037850524,
    2.917361978346634,
    2.9290759184170647,
    2.940572097572018,
    2.9518479336551025,
    2.9629008896387119,
    2.9737284744708989,
    2.9843282439079193,
    2.9946978013321055,
    3.0048347985547301,
    3.0147369366035304,
    3.0244019664945641,
    3.0338276899880818,
    3.043011960328097,
    3.0519526829653545,
    3.060647816263391,
    3.0690953721873995,
    3.0772934169756079,
    3.0852400717928976,
    3.0929335133663827,
    3.1003719746026954,
    3.1075537451867121,
    3.114477172161477,
    3.1211406604890768,
    3.127542673592238,
    3.1336817338764149,
    3.1395564232321571,
    3.1451653835175417,
    3.15050731702047,
    3.1555809869006413,
    3.1603852176110068,
    3.1649188952985416,
    3.169180968184159,
    3.1731704469216067,
    3.1768864049352046,
    3.1803279787362695,
    3.1834943682181054,
    3.1863848369294336,
    3.1889987123261392,
    3.1913353860012443,
    3.1933943138929965,
    3.1951750164709964,
    3.1966770789002794,
    3.1979001511832847,
    3.1988439482796567,
    3.199508250203821,
    3.1998929021002995,
    3.1999978142967294,
    3.1998229623345691,
    3.1993683869774738,
    3.1986341941973344,
    3.1976205551379953,
    3.1963277060566577,
    3.1947559482429955,
    3.1929056479160174,
    3.1907772360987221,
    3.188371208470592,
    3.1856881251979949,
    3.1827286107425574,
    3.1794933536475978,
    3.1759831063026991,
    3.1721986846865322,
    3.1681409680880241,
    3.1638108988059988,
    3.1592094818274079,
    3.1543377844842935,
    3.1491969360896208,
    3.1437881275521344,
    3.1381126109704014,
    3.1321716992062072,
    3.1259667654374881,
    3.1194992426909778,
    3.1127706233547752,
    3.1057824586710256,
    3.0985363582089342,
    3.0910339893183294,
    3.0832770765640021,
    3.0752674011410588,
    3.0670068002715336,
    3.0584971665825038,
    3.0497404474659753,
    3.0407386444207982,
    3.0314938123768917,
    3.0220080590020526,
    3.0122835439916398,
    3.0023224783414282,
    2.9921271236039333,
    2.9816997911285119,
    2.9710428412855627,
    2.9601586826751323,
    2.9490497713202704,
    2.9377186098454557,
    2.926167746640437,
    2.9143997750098336,
    2.9024173323088416,
    2.8902230990654112,
    2.8778197980892459,
    2.8652101935679948,
    2.8523970901510101,
    2.8393833320210446,
    2.8261718019542661,
    2.8127654203689816,
    2.7991671443634516,
    2.7853799667431942,
    2.7714069150381744,
    2.7572510505102734,
    2.7429154671514553,
    2.7284032906730236,
    2.7137176774863887,
    2.6988618136757547,
    2.6838389139631462,
    2.6686522206661882,
    2.653305002649065,
    2.6378005542670784,
    2.6221421943052361,
    2.6063332649112878,
    2.5903771305236463,
    2.5742771767946206,
    2.5580368095093915,
    2.541659453501163,
    2.5251485515629222,
    2.5085075633562415,
    2.4917399643175568,
    2.4748492445623582,
    2.4578389077877212,
    2.4407124701736214,
    2.4234734592834588,
    2.4061254129642302,
    2.3886718782467753,
    2.3711164102465392,
    2.3534625710652666,
    2.3357139286940702,
    2.3178740559182911,
    2.299946529224584,
    2.2819349277106422,
    2.263842831997994,
    2.2456738231482811,
    2.2274314815834404,
    2.2091193860102014,
    2.1907411123493108,
    2.1723002326698917,
    2.1538003141293451,
    2.1352449179191901,
    2.1166375982172436,
    2.0979819011465359,
    2.0792813637413481,
    2.0605395129207573,
    2.0417598644700767,
    2.0229459220305621,
    2.0041011760977603,
    1.9852291030288669,
    1.9663331640594566,
    1.9474168043299442,
    1.9284834519221283,
    1.9095365169061658,
    1.8905793903983192,
    1.8716154436298122,
    1.8526480270271242,
    1.8336804693040483,
    1.8147160765658317,
    1.7957581314257087,
    1.7768098921341344,
    1.757874591721017,
    1.7389554371512421,
    1.7200556084937735,
    1.7011782581046115,
    1.6823265098238815,
    1.6635034581873134,
    1.6447121676523759,
    1.6259556718393136,
    1.6072369727873277,
    1.5885590402261418,
    1.569924810863176,
    1.5513371876865543,
    1.5327990392841535,
    1.5143131991789038,
    1.4958824651805365,
    1.4775095987539655,
    1.4591973244044891,
    1.4409483290799818,
    1.4227652615902426,
    1.4046507320436574,
    1.3866073113013215,
    1.3686375304487679,
    1.3507438802854295,
    1.3329288108319584,
    1.3151947308555227,
    1.2975440074131837,
    1.2799789654134515,
    1.2625018871961138,
    1.2451150121304144,
    1.227820536231659,
    1.2106206117963081,
    1.193517347055618,
    1.1765128058478751,
    1.159609007309262,
    1.1428079255833881,
    1.1261114895495044,
    1.1095215825694185,
    1.0930400422531135,
    1.076668660243068,
    1.0604091820172656,
    1.0442633067108735,
    1.0282326869565626,
    1.0123189287434322,
    0.99652359129449342,
    0.98084818696266052,
    0.96529418114518539,
    0.94986299221647019,
    0.93455599147917712,
    0.9193745031335514,
    0.90431980426486525,
    0.88939312484887922,
    0.87459564777521526,
    0.85992850888852224,
    0.84539279704731107,
    0.83098955420032838,
    0.81671977548032815,
    0.80258440931509656,
    0.78858435755557565,
    0.77472047562092483,
    0.76099357266035372,
    0.74740441173154981,
    0.73395370999552156,
    0.72064213892766704,
    0.70747032454487457,
    0.69443884764845321,
    0.68154824408268633,
    0.66879900500879463,
    0.65619157719408705,
    0.64372636331607658,
    0.63140372228132624,
    0.61922396955879111,
    0.60718737752741025,
    0.59529417583770261,
    0.58354455178711284,
    0.57193865070884731,
    0.56047657637393755,
    0.54915839140626277,
    0.53798411771025767,
    0.52695373691102776,
    0.51606719080659091,
    0.50532438183195771,
    0.49472517353476164,
    0.48426939106214312,
    0.47395682165859099,
    0.46378721517443899,
    0.45376028458471229,
    0.44387570651801628,
    0.43413312179515584,
    0.42453213597717154,
    0.41507231992247468,
    0.40575321035276335,
    0.39657431042739566,
    0.38753509032589784,
    0.37863498783827954,
    0.36987340896282955,
    0.36124972851106085,
    0.35276329071947493,
    0.34441340986781188,
    0.33619937090345275,
    0.32812043007163921,
    0.32017581555117469,
    0.31236472809527133,
    0.3046863416772046,
    0.29713980414043967,
    0.28972423785289148,
    0.28243874036498168,
    0.27528238507115482,
    0.26825422187451775,
    0.26135327785426538,
    0.25457855793555795,
    0.2479290455615149,
    0.24140370336699205,
    0.23500147385380946,
    0.22872128006709952,
    0.22256202627244567,
    0.21652259863348414,
    0.21060186588964297,
    0.20479868003369461,
    0.1991118769888007,
    0.19354027728472958,
    0.1880826867329303,
    0.18273789710014873,
    0.17750468678027492,
    0.17238182146411329,
    0.16736805480677086,
    0.16246212909236135,
    0.15766277589572711,
    0.15296871674088361,
    0.14837866375589584,
    0.14389132032389873,
    0.13950538172997859,
    0.13521953580363627,
    0.13103246355655648,
    0.12694283981541346,
    0.12294933384944563,
    0.11905060999253865,
    0.11524532825955904,
    0.1115321449566869,
    0.10790971328550016,
    0.1043766839405684,
    0.10093170570031919,
    0.097573426010945305,
    0.094300491563126435,
    0.091111548861344441,
    0.088005244785576947,
    0.08498022714515939,
    0.082035145224611736,
    0.079168650321231493,
    0.076379396274260991,
    0.073666039985442508,
    0.071027241930780818,
    0.068461666663339499,
    0.06596798330690247,
    0.063544866040339446,
    0.061190994572519815,
    0.058905054607625698,
    0.056685738300721815,
    0.05453174470344576,
    0.052441780199689175,
    0.050414558931146533,
    0.048448803212615049,
    0.046543243936935833,
    0.044696620969472856,
    0.042907683532033232,
    0.041175190576138811,
    0.039497911145565875,
    0.037874624728076484,
    0.036304121596271645,
    0.03478520313750337,
    0.033316682172789283,
    0.0318973832646803,
    0.03052614301403864,
    0.029201810345690037,
    0.027923246782920918,
    0.026689326710797906,
    0.025498937628293743,
    0.024350980389210356,
    0.023244369431896485,
    0.022178032997763858,
    0.021150913338612489,
    0.020161966912782204,
    0.019210164570154023,
    0.018294491726031464,
    0.017413948523938218,
    0.016567549987375051,
    0.015754326160585048,
    0.014973322238382561,
    0.014223598685107419,
    0.013504231342772056,
    0.012814311528475284,
    0.012152946121162368,
    0.011519257637817014,
    0.010912384299176689,
    0.010331480085068373,
    0.0097757147794676237,
    0.009244274005389283,
    0.0087363592497237061,
    0.0082511878781377744,
    0.0077879931401652067,
    0.0073460241646159464,
    0.0069245459454394228,
    0.0065228393181815288,
    0.0061402009271800066,
    0.0057759431836477276,
    0.0054293942147980081,
    0.0050998978041706458,
    0.0047868133233218077,
    0.0044895156550451999,
    0.0042073951082961586,
    0.0039398573249943619,
    0.0036863231788848141,
    0.0034462286666405691,
    0.0032190247913943559,
    0.0030041774388898314,
    0.0028011672464466025,
    0.0026094894649364708,
    0.0024286538139714968,
    0.002258184330507514,
    0.0020976192110696,
    0.0019465106478087602,
    0.0018044246586016775,
    0.0016709409114078473,
    0.0015456525431007291,
    0.0014281659729917233,
    0.0013181007112678049,
    0.0012150891625655289,
    0.0011187764249058578,
    0.0010288200842158417,
    0.00094489000466462935,
    0.00086666811504256447,
    0.0007938481914132765,
    0.00072613563626964975,
    0.00066324725442540615,
    0.00060491102587472247,
    0.00055086587585284055,
    0.00050086144233102274,
    0.00045465784117944036,
    0.00041202542923167462,
    0.00037274456548444708,
    0.00033660537066598669,
    0.0003034074854060836,
    0.00027295982724036787,
    0.00024508034668070005,
    0.00021959578258275416,
    0.00019634141704092525,
    0.00017516083003959833,
    0.00015590565408857477,
    0.00013843532906906826,
    0.00012261685751515516,
    0.00010832456055389618,
    9.5439834725536554e-05,
    8.3850909903244132e-05,
    7.3452608529759082e-05,
    6.4146106386106717e-05,
    5.5838695105168421e-05,
    4.8443546640415594e-05,
    4.1879479897490843e-05,
    3.6070729733569461e-05,
    3.0946718526556062e-05,
    2.6441830513166763e-05,
    2.2495189091819175e-05,
    1.905043728300242e-05,
    1.6055521536429841e-05,
    1.3462479070789775e-05,
    1.1227228928307496e-05,
    9.3093669226160702e-06,
    7.6719646546080615e-06,
    6.2813727670061508e-06,
    5.1070286043511921e-06,
    4.1212684409638122e-06,
    3.299144435192743e-06,
    2.6182464639225005e-06,
    2.0585289868775136e-06,
    1.6021430857320201e-06,
    1.2332738184180195e-06,
    9.3798302432000746e-07,
    7.0405771125818546e-07,
    5.2086415029425018e-07,
    3.7920779944878302e-07,
    2.7119917239973884e-07,
    1.9012576314070258e-07,
    1.3033013241858034e-07,
    8.7094256546424873e-08,
    5.6530233901383447e-08,
    3.5477439073566299e-08,
    2.1406209232259912e-08,
    1.2328141824679223e-08,
    6.7130772227209426e-09,
    3.4128343883299223e-09,
    1.591762041529268e-09,
    6.6416219032403316e-10,
    2.38637222020269e-10,
    6.9406064477957571e-11,
    1.4629206927328719e-11,
    1.7766267283204454e-12,
    6.6904361107310139e-14,
    2.7531946671350431e-17,
    5.9010301734039846e-16,
    1.8228615506138665e-13,
    3.2387874208147128e-12,
    2.2460277142474688e-11,
    9.6869098666591426e-11,
    3.1346216635332275e-10,
    8.3654198320026586e-10,
    1.9441123175161559e-09,
    4.0713088135724834e-09,
    7.8608290519492256e-09,
    1.4220320837801961e-08,
    2.4386681783027015e-08,
    3.9997217564790363e-08,
    6.3167600591600003e-08,
    9.6576565192507576e-08,
    1.43557269868571e-07,
    2.0819525161182019e-07,
    2.9543289180904611e-07,
    4.1118030780915306e-07,
    5.6243257884692561e-07,
    7.5739320968618454e-07,
    1.0056037300747248e-06,
    1.3180793228954007e-06,
    1.7074503687554709e-06,
    2.1881097896830123e-06,
    2.7763660695980148e-06,
    3.4906018242997683e-06,
    4.3514377888644257e-06,
    5.3819020855801785e-06,
    6.6076046308652799e-06,
    8.0569165350191635e-06,
    9.761154344151924e-06,
    1.1754768969225399e-05,
    1.4075539142822653e-05,
    1.6764769240044646e-05,
    1.9867491295815928e-05,
    2.3432671046867818e-05,
    2.7513417822760452e-05,
    3.2167198106506669e-05,
    3.7456052581673372e-05,
    4.3446816479262296e-05,
    5.0211343034214001e-05,
    5.7826729858039076e-05,
    6.6375548030860764e-05,
    7.5946073713055809e-05,
    8.6632522073707202e-05,
    9.8535283330235946e-05,
    0.00011176116069086036,
    0.00012642360998894284,
    0.00014264298079582759,
    0.00016054675879644971,
    0.00018026980920980731,
    0.00020195462103433782,
    0.00022575155189632516,
    0.00025181907327769139,
    0.00028032401589789225,
    0.00031144181502314546,
    0.00034535675547487169,
    0.00038226221610802407,
    0.00042236091352892361,
    0.00046586514482130375,
    0.00051299702904850162,
    0.0005639887472991147,
    0.00061908278104297072,
    0.00067853214856393614,
    0.00074260063923591772,
    0.00081156304540838545,
    0.00088570539166787433,
    0.00096532516124219739,
    0.001050731519314532,
    0.0011422455330151134,
    0.0012402003878590048,
    0.001344941600399281,
    0.001456827226866001,
    0.0015762280675625112,
    0.001703527866791956,
    0.0018391235080883414,
    0.0019834252045281241,
    0.002136856683900068,
    0.0022998553685130269,
    0.0024728725494233775,
    0.0026563735548660347,
    0.0028508379126753322,
    0.0030567595064845562,
    0.0032746467254955404,
    0.0035050226076125328,
    0.003748424975737442,
    0.0040054065670266413,
    0.0042765351549126935,
    0.0045623936636976919,
    0.0048635802755283789,
    0.0051807085295667802,
    0.0055144074131738404,
    0.0058653214449273794,
    0.0062341107492996668,
    0.0066214511228240309,
    0.0070280340915841238,
    0.0074545669598638162,
    0.0079017728498001621,
    0.0083703907318864422,
    0.0088611754461769918,
    0.0093748977140503276,
    0.0099123441403919724,
    0.010474317206063425,
    0.011061635250528815,
    0.011675132444516031,
    0.012315658752594381,
    0.012984079885556306,
    0.013681277242496149,
    0.014408147842484552,
    0.0151656042457428,
    0.015954574464227107,
    0.016776001861538745,
    0.01763084504208182,
    0.018520077729396487,
    0.019444688633601451,
    0.02040568130788576,
    0.021404073993996079,
    0.02244089945667186,
    0.023517204806987199,
    0.024634051314564483,
    0.025792514208631375,
    0.026993682467899155,
    0.028238658599246919,
    0.029528558405202698,
    0.030864510740219179,
    0.032247657255748262,
    0.033679152134125388,
    0.035160161811281218,
    0.036691864688304945,
    0.038275450831890247,
    0.039912121663701602,
    0.04160308963870539,
    0.043349577912517041,
    0.045152819997822134,
    0.047014059409936179,
    0.048934549301574495,
    0.050915552086910384,
    0.052958339055006506,
    0.055064189972711015,
    0.057234392677116788,
    0.059470242657688736,
    0.061773042628170724,
    0.064144102088390331,
    0.066584736876086265,
    0.069096268708889699,
    0.071680024716597304,
    0.074337336963880368,
    0.077069541963580415,
    0.079877980180748598,
    0.082763995527591852,
    0.085728934849495661,
    0.088774147402298664,
    0.091900984321001342,
    0.095110798080096287,
    0.098404941945713822,
    0.10178476941978275,
    0.10525163367641147,
    0.10880688699070064,
    0.11245188016020426,
    0.11618796191926117,
    0.12001647834642495,
    0.12393877226522507,
    0.12795618263849753,
    0.13207004395652874,
    0.13628168561926052,
    0.14059243131281002,
    0.14500359838056245,
    0.14951649718909957,
    0.15413243048923106,
    0.15885269277240116,
    0.16367856962274591,
    0.16861133706508205,
    0.17365226090911168,
    0.17880259609013069,
    0.18406358600653369,
    0.18943646185441074,
    0.19492244195953515,
    0.20052273110704535,
    0.20623851986912628,
    0.21207098393099949,
    0.21802128341553378,
    0.22409056220679105,
    0.23027994727282458,
    0.23659054798804957,
    0.24302345545550799,
    0.249579741829352,
    0.25626045963787208,
    0.26306664110739825,
    0.26999929748740337,
    0.27705941837714104,
    0.28424797105414928,
    0.29156589980495501,
    0.29901412525831295,
    0.30659354372131598,
    0.31430502651871184,
    0.32214941933576396,
    0.33012754156499319,
    0.33824018565713759,
    0.34648811647666838,
    0.35487207066219756,
    0.36339275599211601,
    0.3720508507557963,
    0.38084700313069747,
    0.3897818305657047,
    0.39885591917103907,
    0.40806982311506845,
    0.4174240640283513,
    0.42691913041524254,
    0.43655547707338838,
    0.44633352452143732,
    0.45625365843528914,
    0.46631622909320475,
    0.47652155083009462,
    0.48686990150130277,
    0.49736152195619959,
    0.50799661552189379,
    0.51877534749737153,
    0.52969784465836567,
    0.5407641947732581,
    0.5519744461303091,
    0.56332860707650956,
    0.57482664556834473,
    0.5864684887347541,
    0.59825402245256898,
    0.61018309093470502,
    0.62225549633137978,
    0.63447099834462517,
    0.646829313856355,
    0.65933011657024676,
    0.67197303666768826,
    0.6847576604780381,
    0.69768353016343965,
    0.71075014341842491,
    0.72395695318454001,
    0.73730336738021363,
    0.75078874864609002,
    0.76441241410603622,
    0.77817363514403137,
    0.79207163719713569,
    0.8061055995647346,
    0.82027465523424281,
    0.83457789072344768,
    0.84901434593966774,
    0.86358301405588767,
    0.87828284140403401,
    0.89311272738553849,
    0.9080715243993368,
    0.92315803778743877,
    0.93837102579819898,
    0.9537091995674124,
    0.96917122311734594,
    0.98475571337381673,
    1.0004612402014135,
    1.0162863264569528,
    1.0322294480612555,
    1.0482890340893156,
    1.064463466878931,
    1.0807510821578545,
    1.0971501691895174,
    1.1136589709373657,
    1.1302756842478474,
    1.1469984600520726,
    1.163825403586169,
    1.1807545746303389,
    1.1977839877666194,
    1.214911612655341,
    1.232135374330265,
    1.249453153512379,
    1.2668627869423146,
    1.2843620677313485,
    1.3019487457309367,
    1.319620527920722,
    1.3373750788149501,
    1.3552100208872173,
    1.3731229350134682,
    1.3911113609331474,
    1.409172797728409,
    1.4273047043212719,
    1.4455044999886018,
    1.4637695648947995,
    1.4820972406420556,
    1.5004848308380301,
    1.5189296016808098,
    1.5374287825609791,
    1.5559795666806402,
    1.5745791116892063,
    1.5932245403357852,
    1.6119129411379594,
    1.630641369066768,
    1.6494068462476765,
    1.6682063626773278,
    1.6870368769558424,
    1.7058953170344469,
    1.7247785809781857,
    1.743683537743475,
    1.7626070279702433,
    1.7815458647884033,
    1.8004968346383814,
    1.8194566981054392,
    1.838422190767498,
    1.8573900240561849,
    1.8763568861308018,
    1.8953194427649196,
    1.9142743382452885,
    1.9332181962827502,
    1.9521476209348343,
    1.9710591975397103,
    0.010050506338833465,
    0.0095070041050899206,
    0.008987425162166789,
    0.0084909789350962068,
    0.0080168907314503469,
    0.0075644017415747167,
    0.0071327690292073247,
    0.006721265512621007,
    0.0063291799364312224,
    0.005955816834216407,
    0.0056004964821027171,
    0.0052625548434695902,
    0.00494134350493704,
    0.0046362296037999948,
    0.0043465957470792162,
    0.0040718399223624866,
    0.0038113754006137597,
    0.0035646306311318461,
    0.003331049128843968,
    0.0031100893541231409,
    0.0029012245853218335,
    0.0027039427842177177,
    0.0025177464545705587,
    0.0023421524939923641,
    0.0021766920393358905,
    0.0020209103058093977,
    0.0018743664200282246,
    0.0017366332472162967,
    0.0016072972127730482,
    0.0014859581184235074,
    0.001372228953171375,
    0.001265735699276894,
    0.0011661171334831069,
    0.001073024623715761,
    0.00098612192148363762,
    0.00090508495020743423,
    0.00082960158970655595,
    0.00075937145707422743,
    0.00069410568417225627,
    0.00063352669197754322,
    0.00057736796201304709,
    0.00052537380509638126,
    0.00047729912763952773,
    0.00043290919573332195,
    0.00039197939725037613,
    0.00035429500219997118,
    0.00031965092156816406,
    0.00028785146487592404,
    0.00025871009668752875,
    0.00023204919230072137,
    0.00020769979284925461,
    0.00018550136004742349,
    0.00016530153080502168,
    0.00014695587193984318,
    0.00013032763521339584,
    0.00011528751291389518,
    0.00010171339420886787,
    8.9490122487815503e-05,
    7.8509253913372577e-05,
    6.8668817397237977e-05,
    5.9873076214869913e-05,
    5.203229147051107e-05,
    4.5062487621554561e-05,
    3.8885220268575506e-05,
    3.3427346414538111e-05,
    2.8620797393746427e-05,
    2.4402354668040408e-05,
    2.0713428684549728e-05,
    1.7499840986007548e-05,
    1.4711609761197786e-05,
    1.230273901956444e-05,
    1.0231011570352316e-05,
    8.4577859828776785e-06,
    6.9477977006471801e-06,
    5.6689644780564051e-06,
    4.5921963043080447e-06,
    3.6912099749967391e-06,
    2.9423484675155317e-06,
    2.324405272050499e-06,
    1.8184538254480979e-06,
    1.4076821906670362e-06,
    1.0772331198657922e-06,
    8.1404963443124693e-07,
    6.0672625042620812e-07,
    4.4536597302688385e-07,
    3.2144317853867609e-07,
    2.2767249752309759e-07,
    1.5788380744329988e-07,
    1.0690343804381717e-07,
    7.0441687424884305e-08,
    4.4986741456326458e-08,
    2.7705083803822596e-08,
    1.6348478414625042e-08,
    9.1676008339148947e-09,
    4.8323892002579967e-09,
    2.3591802024925592e-09,
    1.0446896742484216e-09,
    4.0689185961058063e-10,
    1.3284570765883041e-10,
    3.3510848222025947e-11,
    5.5901696770730152e-12,
    4.3016551135695328e-13,
    4.4431776340160974e-15,
    3.7766799476867912e-20,
    2.0070460109865819e-14,
    9.1153895556240427e-13,
    9.2191001647371264e-12,
    4.8762531101515555e-11,
    1.7932762787538146e-10,
    5.2245555171401646e-10,
    1.2943095310431482e-09,
    2.8455920812017893e-09,
    5.7084801346198377e-09,
    1.0650539725494381e-08,
    1.8735576147692692e-08,
    3.1391369806292411e-08,
    5.0484242315336365e-08,
    7.840039176050733e-08,
    1.1813393044897435e-07,
    1.7338155291306388e-07,
    2.4864375642307925e-07,
    3.4933253080092067e-07,
    4.8188542891350753e-07,
    6.5388592386670727e-07,
    8.7418995361983871e-07,
    1.1530585485011083e-06,
    1.5022964319287952e-06,
    1.9353964795348316e-06,
    2.4676899168497781e-06,
    3.1165021307442172e-06,
    3.9013139649343354e-06,
    4.8439283650520093e-06,
    5.9686422340550169e-06,
    7.3024233541140288e-06,
    8.8750922265627148e-06,
    1.0719508677038422e-05,
    1.2871763068576316e-05,
    1.5371371961152321e-05,
    1.8261478052002376e-05,
    2.1589054226979999e-05,
    2.5405111549253743e-05,
    2.9764911007792839e-05,
    3.4728178844346454e-05,
    4.0359325273990845e-05,
    4.6727666410802831e-05,
    5.3907649206818442e-05,
    6.1979079209155793e-05,
    7.1027350937022263e-05,
    8.1143680677290403e-05,
    9.2425341494417336e-05,
    0.00010497590024769898,
    0.0001189054564061973,
    0.00013433088244915544,
    0.00015137606563732605,
    0.00017017215093838181,
    0.00019085778488745742,
    0.00021357936016188953,
    0.00023849126064737753,
    0.00026575610677108364,
    0.00029554500087562886,
    0.00032803777240652133,
    0.0003634232226842766,
    0.00040189936903135858,
    0.00044367368802308279,
    0.00048896335763078439,
    0.00053799549802485984,
    0.00059100741080474846,
    0.00064824681642251943,
    0.00070997208956648846,
    0.00077645249227118363,
    0.00084796840452003709,
    0.00092481155210737825,
    0.0010072852315266552,
    0.0010957045316523156,
    0.0011903965519834293,
    0.0012917006172179367,
    0.0013999684879273605,
    0.0015155645671029209,
    0.0016388661023452441,
    0.0017702633834712591,
    0.0019101599353134196,
    0.0020589727054880925,
    0.0022171322469117954,
    0.0023850828948459527,
    0.0025632829382529862,
    0.002752204785248825,
    0.0029523351224393511,
    0.0031641750679308684,
    0.0033882403178073779,
    0.0036250612858703089,
    0.0038751832364393229,
    0.0041391664100159566,
    0.0044175861416151001,
    0.0047110329715727411,
    0.0050201127486418874,
    0.0053454467251922832,
    0.0056876716443332934,
    0.0060474398187832503,
    0.0064254192013126132,
    0.0068222934465924289,
    0.0072387619642838904,
    0.0076755399632091777,
    0.0081333584864482968,
    0.0086129644372112506,
    0.0091151205953396586,
    0.0096406056242967272,
    0.01019021406850954,
    0.010764756340932602,
    0.011365058700706797,
    0.011991963220793189,
    0.012646327745466429,
    0.013329025837558024,
    0.014040946715345249,
    0.014782995178987108,
    0.01555609152641455,
    0.016361171458587811,
    0.017199185974039768,
    0.018071101252630093,
    0.018977898528440985,
    0.019920573951751449,
    0.020900138440033144,
    0.021917617517917168,
    0.022974051146087335,
    0.024070493539061884,
    0.025208012971831936,
    0.026387691575331527,
    0.027610625120720374,
    0.028877922792467262,
    0.030190706950228335,
    0.031550112879521294,
    0.032957288531203041,
    0.034413394249765097,
    0.035919602490467606,
    0.037477097525339703,
    0.039087075138080504,
    0.040750742307901829,
    0.04246931688236049,
    0.04424402723923479,
    0.046076111937506423,
    0.047966819357515962,
    0.049917407330366689,
    0.051929142756658352,
    0.054003301214639005,
    0.056141166557870062,
    0.058344030502505966,
    0.060613192204296971,
    0.062949957825429795,
    0.065355640091327674,
    0.06783155783753797,
    0.070379035546841676,
    0.072999402876726072,
    0.075693994177367885,
    0.078464148000280745,
    0.081311206597787206,
    0.08423651541348158,
    0.08724142256385628,
    0.090327278311270409,
    0.093495434528445401,
    0.096747244154678266,
    0.10008406064396963,
    0.10350723740526846,
    0.10701812723504232,
    0.11061808174238681,
    0.11430845076689364,
    0.11809058178950262,
    0.12196581933656764,
    0.12593550437737255,
    0.1300009737153377,
    0.13416355937316282,
    0.13842458797215773,
    0.14278538010601602,
    0.14724724970929245,
    0.15181150342084934,
    0.15647943994254115,
    0.16125234939341176,
    0.16613151265968221,
    0.1711182007408116,
    0.17621367409191749,
    0.18141918196284579,
    0.1867359617341843,
    0.19216523825051754,
    0.19770822315122333,
    0.20336611419911618,
    0.20914009460724414,
    0.21503133236415001,
    0.22104097955791008,
    0.22717017169926645,
    0.2334200270441712,
    0.23979164591606375,
    0.24628611002820441,
    0.25290448180638908,
    0.259647803712373,
    0.26651709756833153,
    0.27351336388268943,
    0.28063758117764981,
    0.28789070531875643,
    0.29527366884682382,
    0.30278738031256941,
    0.31043272361428526,
    0.31821055733888481,
    0.32612171410666163,
    0.33416699992009846,
    0.34234719351706261,
    0.35066304572872559,
    0.35911527884254429,
    0.36770458597063904,
    0.37643163042390565,
    0.38529704509219598,
    0.39430143183090083,
    0.40344536085426908,
    0.41272937013579353,
    0.4221539648159946,
    0.43171961661793012,
    0.44142676327075764,
    0.45127580794167371,
    0.46126711867655384,
    0.47140102784961158,
    0.48167783162239536,
    0.49209778941243809,
    0.50266112337187085,
    0.51336801787631003,
    0.5242186190243241,
    0.53521303414778154,
    0.54635133133337999,
    0.55763353895565104,
    0.56905964522173158,
    0.58062959772819056,
    0.59234330303019311,
    0.60420062622328075,
    0.61620139053804335,
    0.62834537694795023,
    0.64063232379060864,
    0.65306192640270566,
    0.66563383676889165,
    0.67834766318485351,
    0.69120296993482133,
    0.70419927698374896,
    0.71733605968439973,
    0.73061274849956592,
    0.74402872873964265,
    0.7575833403157719,
    0.77127587750876514,
    0.78510558875400815,
    0.79907167644254384,
    0.81317329673852345,
    0.82740955941320804,
    0.84177952769569864,
    0.85628221814056216,
    0.87091660051251785,
    0.8856815976883381,
    0.90057608557611146,
    0.91559889305201125,
    0.93074880191469955,
    0.94602454685749582,
    0.96142481545842651,
    0.97694824818826709,
    0.99259343843668013,
    1.0083589325565432,
    1.0242432299265554,
    1.0402447830321999,
    1.0563619975651384,
    1.0725932325410921,
    1.0889368004362754,
    1.10539096734242,
    1.1219539531404332,
    1.138623931692718,
    1.1553990310541804,
    1.1722773337019341,
    1.1892568767837097,
    1.206335652384966,
    1.2235116078146899,
    1.2407826459098668,
    1.2581466253585913,
    1.2756013610417831,
    1.2931446243934606,
    1.3107741437795197,
    1.3284876048949557,
    1.346282651179453,
    1.3641568842512706,
    1.3821078643593279,
    1.4001331108533992,
    1.4182301026723092,
    1.4363962788500177,
    1.4546290390394712,
    1.4729257440540908,
    1.4912837164267581,
    1.5097002409861537,
    1.5281725654502911,
    1.5466979010370834,
    1.5652734230917726,
    1.5838962717310423,
    1.6025635525036228,
    1.6212723370671975,
    1.6400196638814066,
    1.6588025389167329,
    1.6776179363790569,
    1.6964627994496506,
    1.7153340410403772,
    1.7342285445638548,
    1.7531431647183364,
    1.7720747282870484,
    1.7910200349517258,
    1.8099758581200722,
    1.8289389457668674,
    1.8479060212884404,
    1.866873784370213,
    1.8858389118670207,
    1.9047980586959026,
    1.9237478587410526,
    1.9426849257706147,
    1.9616058543649972,
    1.980507220856381,
    1.9993855842790829,
    2.0182374873304365,
    2.0370594573418446,
    2.0558480072596472,
    2.0745996366354569,
    2.0933108326255914,
    2.11197807099924,
    2.1305978171549955,
    2.1491665271453675,
    2.1676806487089051,
    2.186136622309538,
    2.2045308821827496,
    2.2228598573881877,
    2.2411199728683178,
    2.2593076505127141,
    2.2774193102275913,
    2.2954513710101616,
    2.3134002520274164,
    2.3312623736989053,
    2.3490341587831147,
    2.3667120334670098,
    2.3842924284583313,
    2.401771780080217,
    2.4191465313677289,
    2.4364131331658498,
    2.453568045228534,
    2.4706077373183688,
    2.4875286903064251,
    2.504327397271862,
    2.5210003646008516,
    2.5375441130843899,
    2.5539551790145625,
    2.5702301152788305,
    2.5863654924518986,
    2.6023578998847361,
    2.6182039467903127,
    2.633900263325621,
    2.6494435016695479,
    2.6648303370961659,
    2.6800574690430143,
    2.6951216221739407,
    2.7100195474360764,
    2.7247480231105152,
    2.7393038558562779,
    2.753683881747139,
    2.7678849673008927,
    2.7819040105006443,
    2.7957379418077148,
    2.8093837251657399,
    2.8228383589955648,
    2.8360988771805227,
    2.8491623500416958,
    2.8620258853027662,
    2.8746866290440565,
    2.8871417666453731,
    2.8993885237172656,
    2.9114241670203205,
    2.9232460053721114,
    2.9348513905414357,
    2.9462377181294688,
    2.95740242843747,
    2.9683430073206916,
    2.979056987028128,
    2.98954194702777,
    2.9997955148170128,
    3.0098153667178922,
    3.0195992286568165,
    3.0291448769284663,
    3.0384501389435585,
    3.0475128939601501,
    3.0563310737981935,
    3.0649026635370307,
    3.073225702195554,
    3.0812982833947369,
    3.0891185560022665,
    3.0966847247590095,
    3.1039950508870486,
    3.1110478526790395,
    3.1178415060686375,
    3.124374445181763,
    3.1306451628684733,
    3.1366522112152158,
    3.1423942020372535,
    3.147869807351054,
    3.1530777598264494,
    3.1580168532183714,
    3.1626859427779874,
    3.1670839456430615,
    3.1712098412073817,
    3.1750626714690933,
    3.1786415413577971,
    3.1819456190402762,
    3.1849741362047159,
    3.18772638832331,
    3.1902017348931309,
    3.1923995996551735,
    3.1943194707914722,
    3.1959609011002139,
    3.1973235081487754,
    3.1984069744046111,
    3.1992110473439492,
    3.1997355395382416,
    3.1999803287183353,
    3.1999453578163353,
    3.1996306349851471,
    3.1990362335956841,
    3.198162292211745,
    3.1970090145425702,
    3.1955766693731,
    3.1938655904719564,
    3.1918761764771983,
    3.1896088907598865,
    3.1870642612655247,
    3.1842428803334362,
    3.1811454044941581,
    3.1777725542449309,
    3.1741251138033855,
    3.1702039308395222,
    3.1660099161861033,
    3.1615440435275697,
    3.1568073490676225,
    3.1518009311756003,
    3.1465259500118066,
    3.1409836271319356,
    3.1351752450707724,
    3.1291021469053342,
    3.1227657357976342,
    3.1161674745172658,
    3.1093088849439998,
    3.1021915475506066,
    3.0948171008661132,
    3.0871872409197296,
    3.079303720665663,
    3.0711683493890694,
    3.0627829920933847,
    3.0541495688692941,
    3.0452700542455995,
    3.0361464765222514,
    3.0267809170858349,
    3.0171755097077755,
    3.0073324398255736,
    2.9972539438073551,
    2.9869423082000455,
    2.9763998689614786,
    2.9656290106767629,
    2.9546321657592181,
    2.9434118136362244,
    2.9319704799203139,
    2.9203107355658471,
    2.9084351960116224,
    2.8963465203097734,
    2.8840474102413105,
    2.871540609418672,
    2.8588289023756519,
    2.8459151136450793,
    2.8328021068246274,
    2.8194927836311341,
    2.8059900829438225,
    2.7922969798368107,
    2.778416484601308,
    2.7643516417578953,
    2.7501055290592911,
    2.7356812564840114,
    2.7210819652213325,
    2.706310826647965,
    2.6913710412968568,
    2.6762658378185464,
    2.6609984719354784,
    2.6455722253897105,
    2.6299904048844298,
    2.6142563410197153,
    2.5983733872229617,
    2.5823449186744014,
    2.5661743312281571,
    2.5498650403292489,
    2.5334204799269977,
    2.5168441013852503,
    2.5001393723898677,
    2.4833097758539027,
    2.4663588088209076,
    2.4492899813668028,
    2.4321068155007435,
    2.4148128440654113,
    2.3974116096371723,
    2.3799066634265249,
    2.3623015641792766,
    2.3445998770788705,
    2.3268051726502965,
    2.3089210256660122,
    2.2909510140542939,
    2.2728987178104445,
    2.2547677179112808,
    2.2365615952333089,
    2.2182839294750134,
    2.1999382980836688,
    2.1815282751870777,
    2.1630574305306531,
    2.1445293284202376,
    2.1259475266710646,
    2.1073155755632587,
    2.0886370168042592,
    2.0699153824985679,
    2.0511541941251923,
    2.0323569615231718,
    2.0135271818855602,
    1.9946683387622315,
    1.9757839010718796,
    1.9568773221235696,
    1.9379520386481961,
    1.9190114698402001,
    1.9000590164098865,
    1.881098059646684,
    1.8621319604936795,
    1.8431640586337545,
    1.8241976715876445,
    1.8052360938242362,
    1.7862825958834152,
    1.7673404235117602,
    1.7484127968113854,
    1.7295029094022181,
    1.7106139275979941,
    1.6917489895962468,
    1.6729112046825587,
    1.6541036524493369,
    1.6353293820293675,
    1.6165914113443947,
    1.5978927263689651,
    1.5792362804097688,
    1.5606249934007048,
    1.542061751213881,
    1.5235494049867651,
    1.505090770465684,
    1.4866886273658655,
    1.4683457187482074,
    1.4500647504129542,
    1.4318483903104451,
    1.4136992679691021,
    1.3956199739408028,
    1.3776130592637907,
    1.3596810349432524,
    1.3418263714496943,
    1.3240514982352372,
    1.3063588032679394,
    1.2887506325842519,
    1.2712292898597004,
    1.2537970359978781,
    1.23645608873783,
    1.2192086222798943,
    1.2020567669300635,
    1.185002608762916,
    1.168048189303162,
    1.1511955052258389,
    1.1344465080751813,
    1.1178031040021854,
    1.1012671535208753,
    1.0848404712832731,
    1.0685248258730666,
    1.0523219396179551,
    1.0362334884206541,
    1.0202611016085223,
    1.0044063618017711,
    0.9886708048002103,
    0.97305591948846792,
    0.95756314775962437,
    0.94219388445718233,
    0.92694947733529542,
    0.91183122703716335,
    0.89684038709149771,
    0.88197816392695338,
    0.86724571690441254,
    0.85264415836700147,
    0.83817455370771143,
    0.82383792145448953,
    0.80963523337265531,
    0.79556741458449487,
    0.78163534370587429,
    0.76783985299970747,
    0.7541817285461091,
    0.74066171042905338,
    0.72728049293935337,
    0.71403872479377029,
    0.70093700937005521,
    0.68797590495771721,
    0.67515592502430832,
    0.66247753849700874,
    0.64994117005928898,
    0.63754720046241964,
    0.62529596685159561,
    0.61318776310643364,
    0.60122284019559824,
    0.58940140654530349,
    0.57772362842143754,
    0.56618963032504499,
    0.55479949540090434,
    0.54355326585892683,
    0.53245094340810373,
    0.52149248970271966,
    0.51067782680055029,
    0.50000683763275355,
    0.48947936648516388,
    0.47909521949069284,
    0.46885416513253553,
    0.45875593475788068,
    0.44880022310181694,
    0.43898668882112613,
    0.42931495503765033,
    0.41978460989091726,
    0.41039520709970595,
    0.40114626653223179,
    0.3920372747846278,
    0.3830676857673973,
    0.37423692129951053,
    0.36554437170981613,
    0.35698939644543765,
    0.34857132468682239,
    0.34028945596910903,
    0.33214306080948103,
    0.32413138134016889,
    0.31625363194676609,
    0.30850899991152214,
    0.30089664606127542,
    0.2934157054196882,
    0.28606528786344759,
    0.27884447878209406,
    0.27175233974114144,
    0.2647879091481512,
    0.25795020292142617,
    0.25123821516098777,
    0.24465091882150297,
    0.23818726638682849,
    0.23184619054583994,
    0.22562660486921629,
    0.21952740448685101,
    0.21354746676556288,
    0.20768565198678232,
    0.20194080402389006,
    0.19631175101888793,
    0.19079730605808423,
    0.18539626784647764,
    0.18010742138052785,
    0.17492953861900257,
    0.16986137915159474,
    0.16490169086500617,
    0.16004921060619756,
    0.15530266484250854,
    0.15066077031835406,
    0.14612223470820865,
    0.1416857572655924,
    0.13735002946777794,
    0.13311373565594081,
    0.12897555367048055,
    0.12493415548124381,
    0.12098820781238562,
    0.11713637276160954,
    0.11337730841353212,
    0.10970966944692187,
    0.1061321077355683,
    0.10264327294254126,
    0.09924181310760638,
    0.095926375227567445,
    0.092695605829312216,
    0.089548151535343373,
    0.086482659621582164,
    0.083497778567237987,
    0.080592158596542618,
    0.077764452212154056,
    0.075013314720040566,
    0.072337404745661749,
    0.069735384741269307,
    0.067205921484156578,
    0.064747686565691853,
    0.062359356870977094,
    0.060039615048979582,
    0.057787149972990871,
    0.055600657191273446,
    0.053478839367762251,
    0.051420406712694627,
    0.049424077403048862,
    0.047488577992678013,
    0.045612643812032491,
    0.043795019357371311,
    0.04203445866936887,
    0.040329725701030592,
    0.038679594674837559,
    0.037082850429047076,
    0.03553828875308275,
    0.034044716711954422,
    0.032600952959655073,
    0.031205828041488637,
    0.029858184685289127,
    0.028556878081498639,
    0.027300776152078061,
    0.026088759808231374,
    0.024919723196930886,
    0.023792573936237488,
    0.02270623333941665,
    0.021659636627857418,
    0.02065173313280828,
    0.019681486485950278,
    0.01874787479883417,
    0.017849890831214976,
    0.016986542148323517,
    0.016156851267120957,
    0.015359855791588573,
    0.014594608537111285,
    0.013860177644019462,
    0.013155646680359803,
    0.012480114733971913,
    0.011832696493953285,
    0.011212522321601155,
    0.010618738310925543
};

// Precomputed offsets to access 'coords' and 'weights' arrays
static const int offsets[][3] = {
    { 0, 0, 0},
    { 823, 0, 823},
    { 0, 823, 0},
    { 823, 823, 823},
    { 823, 0, 0},
    { 0, 0, 823},
    { 823, 823, 0},
    { 0, 823, 823},
    { 1646, 0, 1646},
    { 2469, 0, 1646},
    { 1646, 823, 1646},
    { 2469, 823, 1646},
    { 1646, 0, 2469},
    { 2469, 0, 2469},
    { 1646, 823, 2469},
    { 2469, 823, 2469},
    { 0, 1646, 0},
    { 823, 1646, 0},
    { 0, 2469, 0},
    { 823, 2469, 0},
    { 0, 1646, 823},
    { 823, 1646, 823},
    { 0, 2469, 823},
    { 823, 2469, 823},
    { 1646, 1646, 1646},
    { 2469, 1646, 1646},
    { 1646, 2469, 1646},
    { 2469, 2469, 1646},
    { 1646, 1646, 2469},
    { 2469, 1646, 2469},
    { 1646, 2469, 2469},
    { 2469, 2469, 2469},
    { 1646, 0, 0},
    { 2469, 0, 0},
    { 1646, 823, 0},
    { 2469, 823, 0},
    { 1646, 0, 823},
    { 2469, 0, 823},
    { 1646, 823, 823},
    { 2469, 823, 823},
    { 0, 0, 1646},
    { 823, 0, 1646},
    { 0, 823, 1646},
    { 823, 823, 1646},
    { 0, 0, 2469},
    { 823, 0, 2469},
    { 0, 823, 2469},
    { 823, 823, 2469},
    { 1646, 1646, 0},
    { 2469, 1646, 0},
    { 1646, 2469, 0},
    { 2469, 2469, 0},
    { 1646, 1646, 823},
    { 2469, 1646, 823},
    { 1646, 2469, 823},
    { 2469, 2469, 823},
    { 0, 1646, 1646},
    { 823, 1646, 1646},
    { 0, 2469, 1646},
    { 823, 2469, 1646},
    { 0, 1646, 2469},
    { 823, 1646, 2469},
    { 0, 2469, 2469},
    { 823, 2469, 2469},
    { 3292, 0, 3292},
    { 4115, 0, 3292},
    { 3292, 823, 3292},
    { 4115, 823, 3292},
    { 3292, 0, 4115},
    { 4115, 0, 4115},
    { 3292, 823, 4115},
    { 4115, 823, 4115},
    { 4938, 0, 3292},
    { 5761, 0, 3292},
    { 4938, 823, 3292},
    { 5761, 823, 3292},
    { 4938, 0, 4115},
    { 5761, 0, 4115},
    { 4938, 823, 4115},
    { 5761, 823, 4115},
    { 3292, 1646, 3292},
    { 4115, 1646, 3292},
    { 3292, 2469, 3292},
    { 4115, 2469, 3292},
    { 3292, 1646, 4115},
    { 4115, 1646, 4115},
    { 3292, 2469, 4115},
    { 4115, 2469, 4115},
    { 4938, 1646, 3292},
    { 5761, 1646, 3292},
    { 4938, 2469, 3292},
    { 5761, 2469, 3292},
    { 4938, 1646, 4115},
    { 5761, 1646, 4115},
    { 4938, 2469, 4115},
    { 5761, 2469, 4115},
    { 3292, 0, 4938},
    { 4115, 0, 4938},
    { 3292, 823, 4938},
    { 4115, 823, 4938},
    { 3292, 0, 5761},
    { 4115, 0, 5761},
    { 3292, 823, 5761},
    { 4115, 823, 5761},
    { 4938, 0, 4938},
    { 5761, 0, 4938},
    { 4938, 823, 4938},
    { 5761, 823, 4938},
    { 4938, 0, 5761},
    { 5761, 0, 5761},
    { 4938, 823, 5761},
    { 5761, 823, 5761},
    { 3292, 1646, 4938},
    { 4115, 1646, 4938},
    { 3292, 2469, 4938},
    { 4115, 2469, 4938},
    { 3292, 1646, 5761},
    { 4115, 1646, 5761},
    { 3292, 2469, 5761},
    { 4115, 2469, 5761},
    { 4938, 1646, 4938},
    { 5761, 1646, 4938},
    { 4938, 2469, 4938},
    { 5761, 2469, 4938},
    { 4938, 1646, 5761},
    { 5761, 1646, 5761},
    { 4938, 2469, 5761},
    { 5761, 2469, 5761},
    { 0, 3292, 0},
    { 823, 3292, 0},
    { 0, 4115, 0},
    { 823, 4115, 0},
    { 0, 3292, 823},
    { 823, 3292, 823},
    { 0, 4115, 823},
    { 823, 4115, 823},
    { 1646, 3292, 0},
    { 2469, 3292, 0},
    { 1646, 4115, 0},
    { 2469, 4115, 0},
    { 1646, 3292, 823},
    { 2469, 3292, 823},
    { 1646, 4115, 823},
    { 2469, 4115, 823},
    { 0, 4938, 0},
    { 823, 4938, 0},
    { 0, 5761, 0},
    { 823, 5761, 0},
    { 0, 4938, 823},
    { 823, 4938, 823},
    { 0, 5761, 823},
    { 823, 5761, 823},
    { 1646, 4938, 0},
    { 2469, 4938, 0},
    { 1646, 5761, 0},
    { 2469, 5761, 0},
    { 1646, 4938, 823},
    { 2469, 4938, 823},
    { 1646, 5761, 823},
    { 2469, 5761, 823},
    { 0, 3292, 1646},
    { 823, 3292, 1646},
    { 0, 4115, 1646},
    { 823, 4115, 1646},
    { 0, 3292, 2469},
    { 823, 3292, 2469},
    { 0, 4115, 2469},
    { 823, 4115, 2469},
    { 1646, 3292, 1646},
    { 2469, 3292, 1646},
    { 1646, 4115, 1646},
    { 2469, 4115, 1646},
    { 1646, 3292, 2469},
    { 2469, 3292, 2469},
    { 1646, 4115, 2469},
    { 2469, 4115, 2469},
    { 0, 4938, 1646},
    { 823, 4938, 1646},
    { 0, 5761, 1646},
    { 823, 5761, 1646},
    { 0, 4938, 2469},
    { 823, 4938, 2469},
    { 0, 5761, 2469},
    { 823, 5761, 2469},
    { 1646, 4938, 1646},
    { 2469, 4938, 1646},
    { 1646, 5761, 1646},
    { 2469, 5761, 1646},
    { 1646, 4938, 2469},
    { 2469, 4938, 2469},
    { 1646, 5761, 2469},
    { 2469, 5761, 2469},
    { 3292, 3292, 3292},
    { 4115, 3292, 3292},
    { 3292, 4115, 3292},
    { 4115, 4115, 3292},
    { 3292, 3292, 4115},
    { 4115, 3292, 4115},
    { 3292, 4115, 4115},
    { 4115, 4115, 4115},
    { 4938, 3292, 3292},
    { 5761, 3292, 3292},
    { 4938, 4115, 3292},
    { 5761, 4115, 3292},
    { 4938, 3292, 4115},
    { 5761, 3292, 4115},
    { 4938, 4115, 4115},
    { 5761, 4115, 4115},
    { 3292, 4938, 3292},
    { 4115, 4938, 3292},
    { 3292, 5761, 3292},
    { 4115, 5761, 3292},
    { 3292, 4938, 4115},
    { 4115, 4938, 4115},
    { 3292, 5761, 4115},
    { 4115, 5761, 4115},
    { 4938, 4938, 3292},
    { 5761, 4938, 3292},
    { 4938, 5761, 3292},
    { 5761, 5761, 3292},
    { 4938, 4938, 4115},
    { 5761, 4938, 4115},
    { 4938, 5761, 4115},
    { 5761, 5761, 4115},
    { 3292, 3292, 4938},
    { 4115, 3292, 4938},
    { 3292, 4115, 4938},
    { 4115, 4115, 4938},
    { 3292, 3292, 5761},
    { 4115, 3292, 5761},
    { 3292, 4115, 5761},
    { 4115, 4115, 5761},
    { 4938, 3292, 4938},
    { 5761, 3292, 4938},
    { 4938, 4115, 4938},
    { 5761, 4115, 4938},
    { 4938, 3292, 5761},
    { 5761, 3292, 5761},
    { 4938, 4115, 5761},
    { 5761, 4115, 5761},
    { 3292, 4938, 4938},
    { 4115, 4938, 4938},
    { 3292, 5761, 4938},
    { 4115, 5761, 4938},
    { 3292, 4938, 5761},
    { 4115, 4938, 5761},
    { 3292, 5761, 5761},
    { 4115, 5761, 5761},
    { 4938, 4938, 4938},
    { 5761, 4938, 4938},
    { 4938, 5761, 4938},
    { 5761, 5761, 4938},
    { 4938, 4938, 5761},
    { 5761, 4938, 5761},
    { 4938, 5761, 5761},
    { 5761, 5761, 5761},
    { 3292, 0, 0},
    { 4115, 0, 0},
    { 3292, 823, 0},
    { 4115, 823, 0},
    { 3292, 0, 823},
    { 4115, 0, 823},
    { 3292, 823, 823},
    { 4115, 823, 823},
    { 4938, 0, 0},
    { 5761, 0, 0},
    { 4938, 823, 0},
    { 5761, 823, 0},
    { 4938, 0, 823},
    { 5761, 0, 823},
    { 4938, 823, 823},
    { 5761, 823, 823},
    { 3292, 1646, 0},
    { 4115, 1646, 0},
    { 3292, 2469, 0},
    { 4115, 2469, 0},
    { 3292, 1646, 823},
    { 4115, 1646, 823},
    { 3292, 2469, 823},
    { 4115, 2469, 823},
    { 4938, 1646, 0},
    { 5761, 1646, 0},
    { 4938, 2469, 0},
    { 5761, 2469, 0},
    { 4938, 1646, 823},
    { 5761, 1646, 823},
    { 4938, 2469, 823},
    { 5761, 2469, 823},
    { 3292, 0, 1646},
    { 4115, 0, 1646},
    { 3292, 823, 1646},
    { 4115, 823, 1646},
    { 3292, 0, 2469},
    { 4115, 0, 2469},
    { 3292, 823, 2469},
    { 4115, 823, 2469},
    { 4938, 0, 1646},
    { 5761, 0, 1646},
    { 4938, 823, 1646},
    { 5761, 823, 1646},
    { 4938, 0, 2469},
    { 5761, 0, 2469},
    { 4938, 823, 2469},
    { 5761, 823, 2469},
    { 3292, 1646, 1646},
    { 4115, 1646, 1646},
    { 3292, 2469, 1646},
    { 4115, 2469, 1646},
    { 3292, 1646, 2469},
    { 4115, 1646, 2469},
    { 3292, 2469, 2469},
    { 4115, 2469, 2469},
    { 4938, 1646, 1646},
    { 5761, 1646, 1646},
    { 4938, 2469, 1646},
    { 5761, 2469, 1646},
    { 4938, 1646, 2469},
    { 5761, 1646, 2469},
    { 4938, 2469, 2469},
    { 5761, 2469, 2469},
    { 0, 0, 3292},
    { 823, 0, 3292},
    { 0, 823, 3292},
    { 823, 823, 3292},
    { 0, 0, 4115},
    { 823, 0, 4115},
    { 0, 823, 4115},
    { 823, 823, 4115},
    { 1646, 0, 3292},
    { 2469, 0, 3292},
    { 1646, 823, 3292},
    { 2469, 823, 3292},
    { 1646, 0, 4115},
    { 2469, 0, 4115},
    { 1646, 823, 4115},
    { 2469, 823, 4115},
    { 0, 1646, 3292},
    { 823, 1646, 3292},
    { 0, 2469, 3292},
    { 823, 2469, 3292},
    { 0, 1646, 4115},
    { 823, 1646, 4115},
    { 0, 2469, 4115},
    { 823, 2469, 4115},
    { 1646, 1646, 3292},
    { 2469, 1646, 3292},
    { 1646, 2469, 3292},
    { 2469, 2469, 3292},
    { 1646, 1646, 4115},
    { 2469, 1646, 4115},
    { 1646, 2469, 4115},
    { 2469, 2469, 4115},
    { 0, 0, 4938},
    { 823, 0, 4938},
    { 0, 823, 4938},
    { 823, 823, 4938},
    { 0, 0, 5761},
    { 823, 0, 5761},
    { 0, 823, 5761},
    { 823, 823, 5761},
    { 1646, 0, 4938},
    { 2469, 0, 4938},
    { 1646, 823, 4938},
    { 2469, 823, 4938},
    { 1646, 0, 5761},
    { 2469, 0, 5761},
    { 1646, 823, 5761},
    { 2469, 823, 5761},
    { 0, 1646, 4938},
    { 823, 1646, 4938},
    { 0, 2469, 4938},
    { 823, 2469, 4938},
    { 0, 1646, 5761},
    { 823, 1646, 5761},
    { 0, 2469, 5761},
    { 823, 2469, 5761},
    { 1646, 1646, 4938},
    { 2469, 1646, 4938},
    { 1646, 2469, 4938},
    { 2469, 2469, 4938},
    { 1646, 1646, 5761},
    { 2469, 1646, 5761},
    { 1646, 2469, 5761},
    { 2469, 2469, 5761},
    { 3292, 3292, 0},
    { 4115, 3292, 0},
    { 3292, 4115, 0},
    { 4115, 4115, 0},
    { 3292, 3292, 823},
    { 4115, 3292, 823},
    { 3292, 4115, 823},
    { 4115, 4115, 823},
    { 4938, 3292, 0},
    { 5761, 3292, 0},
    { 4938, 4115, 0},
    { 5761, 4115, 0},
    { 4938, 3292, 823},
    { 5761, 3292, 823},
    { 4938, 4115, 823},
    { 5761, 4115, 823},
    { 3292, 4938, 0},
    { 4115, 4938, 0},
    { 3292, 5761, 0},
    { 4115, 5761, 0},
    { 3292, 4938, 823},
    { 4115, 4938, 823},
    { 3292, 5761, 823},
    { 4115, 5761, 823},
    { 4938, 4938, 0},
    { 5761, 4938, 0},
    { 4938, 5761, 0},
    { 5761, 5761, 0},
    { 4938, 4938, 823},
    { 5761, 4938, 823},
    { 4938, 5761, 823},
    { 5761, 5761, 823},
    { 3292, 3292, 1646},
    { 4115, 3292, 1646},
    { 3292, 4115, 1646},
    { 4115, 4115, 1646},
    { 3292, 3292, 2469},
    { 4115, 3292, 2469},
    { 3292, 4115, 2469},
    { 4115, 4115, 2469},
    { 4938, 3292, 1646},
    { 5761, 3292, 1646},
    { 4938, 4115, 1646},
    { 5761, 4115, 1646},
    { 4938, 3292, 2469},
    { 5761, 3292, 2469},
    { 4938, 4115, 2469},
    { 5761, 4115, 2469},
    { 3292, 4938, 1646},
    { 4115, 4938, 1646},
    { 3292, 5761, 1646},
    { 4115, 5761, 1646},
    { 3292, 4938, 2469},
    { 4115, 4938, 2469},
    { 3292, 5761, 2469},
    { 4115, 5761, 2469},
    { 4938, 4938, 1646},
    { 5761, 4938, 1646},
    { 4938, 5761, 1646},
    { 5761, 5761, 1646},
    { 4938, 4938, 2469},
    { 5761, 4938, 2469},
    { 4938, 5761, 2469},
    { 5761, 5761, 2469},
    { 0, 3292, 3292},
    { 823, 3292, 3292},
    { 0, 4115, 3292},
    { 823, 4115, 3292},
    { 0, 3292, 4115},
    { 823, 3292, 4115},
    { 0, 4115, 4115},
    { 823, 4115, 4115},
    { 1646, 3292, 3292},
    { 2469, 3292, 3292},
    { 1646, 4115, 3292},
    { 2469, 4115, 3292},
    { 1646, 3292, 4115},
    { 2469, 3292, 4115},
    { 1646, 4115, 4115},
    { 2469, 4115, 4115},
    { 0, 4938, 3292},
    { 823, 4938, 3292},
    { 0, 5761, 3292},
    { 823, 5761, 3292},
    { 0, 4938, 4115},
    { 823, 4938, 4115},
    { 0, 5761, 4115},
    { 823, 5761, 4115},
    { 1646, 4938, 3292},
    { 2469, 4938, 3292},
    { 1646, 5761, 3292},
    { 2469, 5761, 3292},
    { 1646, 4938, 4115},
    { 2469, 4938, 4115},
    { 1646, 5761, 4115},
    { 2469, 5761, 4115},
    { 0, 3292, 4938},
    { 823, 3292, 4938},
    { 0, 4115, 4938},
    { 823, 4115, 4938},
    { 0, 3292, 5761},
    { 823, 3292, 5761},
    { 0, 4115, 5761},
    { 823, 4115, 5761},
    { 1646, 3292, 4938},
    { 2469, 3292, 4938},
    { 1646, 4115, 4938},
    { 2469, 4115, 4938},
    { 1646, 3292, 5761},
    { 2469, 3292, 5761},
    { 1646, 4115, 5761},
    { 2469, 4115, 5761},
    { 0, 4938, 4938},
    { 823, 4938, 4938},
    { 0, 5761, 4938},
    { 823, 5761, 4938},
    { 0, 4938, 5761},
    { 823, 4938, 5761},
    { 0, 5761, 5761},
    { 823, 5761, 5761},
    { 1646, 4938, 4938},
    { 2469, 4938, 4938},
    { 1646, 5761, 4938},
    { 2469, 5761, 4938},
    { 1646, 4938, 5761},
    { 2469, 4938, 5761},
    { 1646, 5761, 5761},
    { 2469, 5761, 5761}
};

extern void get_coords_by_index(double * const pa, int index)
// Retrieves the coordinates of the cubature point with
// the given index and computes its associated weight 
{
    const int N = 823; // Number of points in the seed lattice
    int iN = index / N;
    int ix = index % N + offsets[iN][0];
    int iy = (index * 230) % N + offsets[iN][1];
    int iz = (index * 311) % N + offsets[iN][2];

    // Coordinates
    pa[0] = coords[ix];
    pa[1] = coords[iy];
    pa[2] = coords[iz];

    // Weight
    pa[3] = weights[ix] * weights[iy] * weights[iz];

}
