% plotfig3.m
% This Matlab script file plots histograms of four ratios that
% are by LAPACK's test suite  to verify the accuracy of the routines
% xGELSZ, xGELSY, and xGELSD.  It produces figures similar to those
% in Figure 3 in the paper by Foster and Kommu.  
%
% Before using plotfig3 run the Fortran program cfig3.f, dfig3.f,
% sfig3.f or zfig3.f which will produce the output files 
% fig3_12.out, fig3_14.out, fig3_16.out and fig3_17.out.  Following
% this run plotfig3.m from inside Matlab. 
% 
% In addition to the plot the program prints out a table of summarizing
% any failures of the least squares routines xGELSZ, xGELSY and xGELSD
% where a failure is defined as any ratio bigger than 30.

% This code is part of a package for solving rank deficient least
% squares problems, written by:
% ==================================================================
% L. Foster                   and   R. Kommu
% Department of Mathematics         Department of Physics
% San Jose State University         San Jose State University
% San Jose, CA 95192                San Jose, CA 95192
% foster@math.sjsu.edu              rkommu@email.sjsu.edu
% ==================================================================
% 03/05/2004
%
 
% It may be of use to change the value of ndiv.  The parameter ndiv
% specifies the number of subdivision used in the histogram.
% Values of ndiv in the range between 20 and 60 seem reasonable
% depending on the number of samples generated by fig3.f.

ndiv = 60;
%ndiv = 50;
%ndiv = 30;

MAXRATIO = 30;
% Following the LAPACK guidelines a method is considered to fail if it 
% has any ratio larger than  MAXRATIO. LAPACK uses a MAXRATIO of 30.
% The ratios bigger than MAXRATIO will be counted separately. 


clf

load fig3_16.out
load fig3_12.out
load fig3_17.out
load fig3_14.out

istopplot = 1; %stop plotting in histogram after last nonzero value
% istopplot = 0 % plots the zero values

%**********************************************
% create histograms for ratios produced by LAPACK's xQRT16
subplot(221)
accs = fig3_16;
valsd = accs(:,1);
valsy = accs(:,2);
valsz = accs(:,3);
ntotsam = length(valsy);

count_fail_ratio16_y = sum( valsy > MAXRATIO );
count_fail_ratio16_z = sum( valsz > MAXRATIO );
count_fail_ratio16_d = sum( valsd > MAXRATIO );

% histogram for xGELSY
maxv = max(max(valsy),max(valsz));
maxv = maxv*(1+10000*eps); 
maxv = min( maxv, MAXRATIO );
if ( maxv == 0 ) 
   xgeny = 0;
   ngeny = length(valsy);
else
   xdivy = maxv/ndiv;
   edgesy = (0:ndiv) * xdivy;
   edgesy(ndiv+1) =  maxv;
   xgeny = ( edgesy(1:ndiv) + edgesy(2:ndiv+1) ) / 2 ; %midpoint
   ngeny = histc(valsy,edgesy) ; %hist information
   sngeny = length(ngeny);   %combine last two counts
   ngeny(sngeny-1) = ngeny(sngeny-1)+ngeny(sngeny) + count_fail_ratio16_y; 
   ngeny = ngeny(1:sngeny-1);
   if (istopplot == 1) 
      posngen = find(ngeny>0);%stop plotting after last nonzero value
      last = max(posngen);   
      ngeny = ngeny(1:last);   
      xgeny = xgeny(1:last);
   end 
end

% histogram for xGELSZ
if ( maxv == 0 ) 
   xgenz = 0;
   ngenz = length(valsz);
else
   edgesz = edgesy;
   xgenz = ( edgesz(1:ndiv) + edgesz(2:ndiv+1) ) / 2 ; %midpoint
   ngenz = histc(valsz,edgesz);
   sngenz = length(ngenz);   %combine last two counts
   ngenz(sngenz-1) = ngenz(sngenz-1)+ngenz(sngenz) + count_fail_ratio16_z;
   ngenz = ngenz(1:sngenz-1);
   if (istopplot == 1) 
      posngen = find(ngenz>0);  %stop plotting after last nonzero value
      last = max(posngen);
      ngenz = ngenz(1:last);
      xgenz = xgenz(1:last);
   end   
end

% histogram for xGELSD
maxvald = max(valsd);
maxvald = min( maxvald, MAXRATIO );
if ( maxvald == 0 ) 
   xgend = 0;
   ngend = length(valsd);
else
   ndivd = ceil( maxvald / xdivy );
   edgesd = (0:ndivd) * xdivy;
   xgend = ( edgesd(1:ndivd) + edgesd(2:ndivd+1) ) / 2 ; %midpoint
   ngend = histc(valsd, edgesd);
   sngend = length(ngend); %combine last two counts
   ngend(sngend-1) = ngend(sngend-1) + ngend(sngend) + count_fail_ratio16_d;
   ngend = ngend(1:sngend-1);
   if (istopplot == 1) 
      posngen = find(ngend>0);  %stop plotting after last nonzero value
      last = max(posngen);
      ngend = ngend(1:last);
      xgend = xgend(1:last);   %comment out to keep +'s accross the graph
   end
end

plot(xgenz,ngenz,'x-',xgeny,ngeny,'o-',xgend,ngend,'+-')
legend('DGELSZ','DGELSY','DGELSD')
title(['Histogram'])
ylabel('count')
xlabel(' ||b-Ax|| / [ max(m,n) ||A|| ||x|| \epsilon ]')
grid


%**************************************************
% create histograms for ratios produced by LAPACK's xQRT12

subplot(222)
accs = fig3_12;
valsy = accs(:,2);
valsz = accs(:,3);
valsd = accs(:,1);
ntotsam = length(valsy);

count_fail_ratio12_y = sum( valsy > MAXRATIO );
count_fail_ratio12_z = sum( valsz > MAXRATIO );
count_fail_ratio12_d = sum( valsd > MAXRATIO );

% histogram for xGELSY
maxv = max(max(valsy),max(valsz));
maxv = maxv*(1+10000*eps);
maxv = min( maxv, MAXRATIO );
if ( maxv == 0 ) 
   xgeny = 0;
   ngeny = length(valsy);
else
   xdivy = maxv/ndiv;
   edgesy = (0:ndiv) * xdivy;
   edgesy(ndiv+1) = maxv;
   xgeny = ( edgesy(1:ndiv) + edgesy(2:ndiv+1) ) / 2 ; %midpoint
   ngeny = histc(valsy,edgesy);  %hist information
   sngeny = length(ngeny);   %combine last two counts
   ngeny(sngeny-1) = ngeny(sngeny-1) + ngeny(sngeny) + count_fail_ratio12_y; 
   ngeny = ngeny(1:sngeny-1);
   if (istopplot == 1) 
      posngen = find(ngeny>0);%stop plotting after last nonzero value
      last = max(posngen);   
      ngeny = ngeny(1:last);
      xgeny = xgeny(1:last);
   end   
end

% histogram for xGELSZ
if ( maxv == 0 ) 
   xgenz = 0;
   ngenz = length(valsz);
else
   edgesz = edgesy;
   xgenz = ( edgesz(1:ndiv) + edgesz(2:ndiv+1) ) / 2 ; %midpoint
   ngenz = histc(valsz,edgesz);
   sngenz = length(ngenz);   %combine last two counts
   ngenz(sngenz-1) = ngenz(sngenz-1) + ngenz(sngenz) + count_fail_ratio12_z;
   ngenz = ngenz(1:sngenz-1);
   if (istopplot == 1) 
      posngen = find(ngenz>0);  %stop plotting after last nonzero value
      last = max(posngen);
      ngenz = ngenz(1:last);
      xgenz = xgenz(1:last);
   end
end

% histogram for xGELSD
maxvald = max(valsd);
maxvald = min( maxvald, MAXRATIO );

if ( maxvald == 0 ) 
   xgend = 0;
   ngend = length(valsd);
else
   ndivd = ceil( maxvald / xdivy );
   edgesd = (0:ndivd) * xdivy;
   xgend = ( edgesd(1:ndivd) + edgesd(2:ndivd+1) ) / 2 ; %midpoint
   ngend = histc(valsd,edgesd);
   sngend = length(ngend); %combine last two counts
   ngend(sngend-1) = ngend(sngend-1) + ngend(sngend) + count_fail_ratio12_d;
   ngend = ngend(1:sngend-1);
   if (istopplot == 1) 
      posngen = find(ngend>0);  %stop plotting after last nonzero value
      last = max(posngen);
      ngend = ngend(1:last);
      xgend = xgend(1:last);
   end
end

plot(xgenz,ngenz,'x-',xgeny,ngeny,'o-',xgend,ngend,'+-')
legend('DGELSZ','DGELSY','DGELSD')
grid
title(['Histogram'])
ylabel('count')
xlabel('|| s_{1:k} - svals(T_{11}) || / [ ||svals(T_{11})|| max(m,n) \epsilon ]')

%*****************************************************
% create histograms for ratios produced by LAPACK's xQRT17


subplot(223)
accs = fig3_17;
valsy = accs(:,2);
valsz = accs(:,3);
valsd = accs(:,1);
ntotsam = length(valsy);

count_fail_ratio17_y = sum( valsy > MAXRATIO );
count_fail_ratio17_z = sum( valsz > MAXRATIO );
count_fail_ratio17_d = sum( valsd > MAXRATIO );

% histogram for xGELSY
maxv = max(max(valsy),max(valsz));
maxv = maxv * (1 + 10000 * eps);
maxv = min( maxv, MAXRATIO );
if ( maxv == 0 ) 
   xgeny = 0;
   ngeny = length(valsy);
else
   xdivy = maxv / ndiv;
   edgesy = (0:ndiv) * xdivy;
   edgesy(ndiv+1) = maxv;
   xgeny = ( edgesy(1:ndiv) + edgesy(2:ndiv+1) ) / 2 ; %midpoint
   ngeny = histc(valsy,edgesy);  %hist information
   sngeny = length(ngeny);   %combine last two counts
   ngeny(sngeny-1) = ngeny(sngeny-1) + ngeny(sngeny) + count_fail_ratio17_y;
   ngeny = ngeny(1:sngeny-1);
   if (istopplot == 1)  
      posngen = find(ngeny>0);%stop plotting after last nonzero value
      last = max(posngen);   
      ngeny = ngeny(1:last);
      xgeny = xgeny(1:last);
   end
end

% histogram for xGELSZ
if ( maxv == 0 ) 
   xgenz = 0;
   ngenz = length(valsz);
else
   edgesz = edgesy;
   xgenz = ( edgesz(1:ndiv) + edgesz(2:ndiv+1) ) / 2 ; %midpoint
   ngenz = histc(valsz,edgesz);
   sngenz = length(ngenz);   %combine last two counts
   ngenz(sngenz-1) = ngenz(sngenz-1) + ngenz(sngenz) + count_fail_ratio17_z;
   ngenz = ngenz(1:sngenz-1) ;
   if (istopplot == 1) 
      posngen = find(ngenz>0);  %stop plotting after last nonzero value
      last = max(posngen);
      ngenz = ngenz(1:last);
      xgenz = xgenz(1:last);
   end   
end

% histogram for xGELSD
maxvald = max(valsd);
maxvald = min( maxvald, MAXRATIO );

if ( maxvald == 0 ) 
   xgend = 0;
   ngend = length(valsd);
else
   ndivd = ceil( maxvald / xdivy );
   edgesd = (0:ndivd) * xdivy;
   xgend = ( edgesd(1:ndivd) + edgesd(2:ndivd+1) ) / 2 ; %midpoint
   ngend = histc(valsd,edgesd);
   sngend = length(ngend); %combine last two counts
   ngend(sngend-1) = ngend(sngend-1) + ngend(sngend) + count_fail_ratio17_d;
   ngend = ngend(1:sngend-1);
   if (istopplot == 1) 
      posngen = find(ngend>0);  %stop plotting after last nonzero value
      last = max(posngen);
      ngend = ngend(1:last);
      xgend = xgend(1:last);
   end
end
plot(xgenz,ngenz,'x-',xgeny,ngeny,'o-',xgend,ngend,'+-')
legend('DGELSZ','DGELSY','DGELSD')
grid
title(['Histogram'])
ylabel('count')
xlabel('|| r^T  A || / [ ||A|| ||b|| max(m,n) \epsilon ]                    ') 


%********************************************
% create histograms for ratios produced by LAPACK's xQRT14

subplot(224)
accs = fig3_14;
valsy = accs(:,2);
valsz = accs(:,3);
valsd = accs(:,1);
ntotsam = length(valsy);

count_fail_ratio14_y = sum( valsy > MAXRATIO );
count_fail_ratio14_z = sum( valsz > MAXRATIO );
count_fail_ratio14_d = sum( valsd > MAXRATIO );

% histogram for xGELSY
maxv = max(max(valsy),max(valsz));
maxv = maxv * (1 + 10000 * eps); 
maxv = min( maxv, MAXRATIO );
if ( maxv == 0 ) 
   xgeny = 0;
   ngeny = length(valsy);
else
   xdivy = maxv / ndiv;
   edgesy = (0:ndiv) * xdivy;
   edgesy(ndiv+1) = maxv;
   xgeny = ( edgesy(1:ndiv) + edgesy(2:ndiv+1) ) / 2 ; %midpoint
   ngeny = histc(valsy,edgesy);  %hist information
   sngeny = length(ngeny);   %combine last two counts
   ngeny(sngeny-1) = ngeny(sngeny-1) + ngeny(sngeny) + count_fail_ratio14_y;
   ngeny = ngeny(1:sngeny-1);
   if (istopplot == 1) 
     posngen = find(ngeny>0);%stop plotting after last nonzero value
     last = max(posngen);   
     ngeny = ngeny(1:last);
     xgeny = xgeny(1:last);
   end
end

% histogram for xGELSZ
if ( maxv == 0)
   xgenz = 0;
   ngenz = length(valsz);
else   
   edgesz = edgesy;
   xgenz = ( edgesz(1:ndiv) + edgesz(2:ndiv+1) ) / 2 ; %midpoint
   ngenz = histc(valsz,edgesz);
   sngenz = length(ngenz);   %combine last two counts
   ngenz(sngenz-1) = ngenz(sngenz-1)+ngenz(sngenz)+ count_fail_ratio14_z;
   ngenz = ngenz(1:sngenz-1) ;
   if (istopplot == 1) 
      posngen = find(ngenz>0);  %stop plotting after last nonzero value
      last = max(posngen);
      ngenz = ngenz(1:last);
      xgenz = xgenz(1:last);
   end
end

% histogram for xGELSD
maxvald = max(valsd);
maxvald = min( maxvald, MAXRATIO );

if ( maxvald == 0 ) 
   xgend = 0;
   ngend = length(valsd);
else
   ndivd = ceil(maxvald/xdivy);
   edgesd = (0:ndivd)*xdivy;
   xgend = ( edgesd(1:ndivd) + edgesd(2:ndivd+1) ) / 2 ; %midpoint
   ngend = histc(valsd,edgesd);
   sngend = length(ngend); %combine last two counts
   ngend(sngend-1) = ngend(sngend-1)+ngend(sngend) + count_fail_ratio14_d;
   ngend = ngend(1:sngend-1) ;
   if (istopplot == 1) 
      posngen = find(ngend>0);  %stop plotting after last nonzero value
      last = max(posngen);
      ngend = ngend(1:last);
      xgend = xgend(1:last);
   end
end

plot(xgenz,ngenz,'x-',xgeny,ngeny,'o-',xgend,ngend,'+-')
legend('DGELSZ','DGELSY','DGELSD')
grid
title(['Histogram'])
ylabel('count')
xlabel(' ratio testing that x is in the row space of A')

shg

disp(' ')
disp(['In ',int2str(ntotsam),' trials the number of failures ', ...
   '( any ratio > ',int2str(MAXRATIO),'). '])
disp('Ratio calculated by:             xQRT16    xQRT12    xQRT17    xQRT14 ')
disp('Using least squares routine ')
disp(['                     xGELSD: ',  ...
      sprintf('%10d',count_fail_ratio16_d),  ...
      sprintf('%10d',count_fail_ratio12_d), ...
      sprintf('%10d',count_fail_ratio17_d), ...
      sprintf('%10d',count_fail_ratio14_d)])
disp(['                     xGELSY: ',  ...
      sprintf('%10d',count_fail_ratio16_y),  ...
      sprintf('%10d',count_fail_ratio12_y), ...
      sprintf('%10d',count_fail_ratio17_y), ...
      sprintf('%10d',count_fail_ratio14_y)])
disp(['                     xGELSZ: ',  ...
      sprintf('%10d',count_fail_ratio16_z),  ...
      sprintf('%10d',count_fail_ratio12_z), ...
      sprintf('%10d',count_fail_ratio17_z), ...
      sprintf('%10d',count_fail_ratio14_z)])



