function [T,Omega,V] = vsv_qrit(p,num_ref,T,Omega,V)
%  vsv_qrit --> Refinement of VSV decomposition via block QR-iterations.
%
%  <Synopsis>
%    [L]         = vsv_qrit(p,num_ref,L)
%    [L,V]       = vsv_qrit(p,num_ref,L,[],V)
%    [L,Omega]   = vsv_qrit(p,num_ref,L,Omega)
%    [L,Omega,V] = vsv_qrit(p,num_ref,L,Omega,V)
%    [R,Omega]   = vsv_qrit(p,num_ref,R,Omega)
%    [R,Omega,V] = vsv_qrit(p,num_ref,R,Omega,V)
%
%  <Description>
%    Given a VSV decomposition with numerical rank p, of one of the forms
%        A = V*L'*L*V'         (semidefinite A, lower triangular L)
%        A = V*L'*Omega*L*V'   (indefinite A, lower triangular L)
%        A = V*R'*Omega*R*V'   (indefinite A, upper triangular R)
%    the function refines the rank-revealing decomposition via num_ref
%    steps of block QR iterations applied to the triangular matrix.
%
%  <Input Parameters>
%    1. p       --> numerical rank of A;
%    2. num_ref --> number of refinement iterations;
%    3. T       --> triangur matrix (L or R, depending on VSV decomposition);
%    4. Omega   --> signature matrix (indef. case) or empty (semidef. case);
%    5. V       --> orthogonal matrix;
%
%  <Output Parameters>
%    1. T          --> refined triangular matrix
%    2. Omega or V --> refined Omega (indef. case) or refined V (semidef. case)
%    3. V          --> refined V (indef. case)
%
%  <Algorithm>
%    Refinement is identical to block QR iteration, in which the
%    off-diagonal block of the triangular matrix is "flipped"
%    to the diagonally opposite position and then back again.
%
%  <See Also>
%    ulv_ref --> Refine one column of L in the ULV decomposition.
%    urv_ref --> Refine one column of R in the URV decomposition.

%  <References>
%  [1] R. Mathias and G.W. Stewart, "A Block QR Algorithm and the Singular
%      Value Decomposition", Lin. Alg. Appl., 182 (1993), pp. 91--100.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%    Peter S.K. Hansen, IMM, Technical University of Denmark
%
%    Last revised: March 11, 2004
%-----------------------------------------------------------------------

% Check the number of input and output arguments.
if (nargin < 3), error('Too few input arguments'), end
semidef = 0;
vflag = 0;
if (nargin == 3 | isempty(Omega)), semidef = 1; end
if (nargin == 5) & ~isempty(V), vflag = 1; end

% Initialize.
[n,n] = size(T);

if (p ~= abs(round(p))) | (p > n)
  error('Requires p to be an integer between 0 and n.')
end
if (num_ref ~= abs(round(num_ref)))
  error('Requires positive integer value for num_ref.')
end

if (semidef)
  % The semidefinite case; T is lower triangular.  Use UTV Tools.
  if (vflag)
    [T,Omega] = ulv_qrit(p,num_ref,T,V);  % Orthogonal matrix is 2. output arg.
  else
    T = ulv_qrit(p,num_ref,T);
  end
  
elseif norm(triu(T,1))==0
  % The indefinite case; T is lower triangular.
  
  % Refinement loop.
  for i = 1:num_ref
    % Flip the off-diagonal block of T using quadratic rotations on the left.
    for r = p+1:n
      for i = p:-1:1
        % Apply rotation to T on the left.
        [c,s,T(i,i),sgn] = gen_qrot(T(i,i),T(r,i),Omega(i,i),Omega(r,r));
        T(r,i) = 0;  % Eliminate T(r,i).
        [T(i,[1:i-1,p+1:r]),T(r,[1:i-1,p+1:r]),Omega(i,i),Omega(r,r)] = ...
          app_qrot(T(i,[1:i-1,p+1:r]),T(r,[1:i-1,p+1:r]),c,s,Omega(i,i),Omega(r,r),sgn);
      end
    end

    % Restore lower triangular form with Givens rotations on the right.
    for r = n:-1:p+1
      for i = 1:p
        % Apply Givens rotation to T on the right.
        [c,s,T(i,i)] = gen_giv(T(i,i),T(i,r));
        T(i,r) = 0;  % Eliminate T(i,r).
        [T(i+1:p,i),T(i+1:p,r)] = app_giv(T(i+1:p,i),T(i+1:p,r),c,s);
        [T(r:n,i),T(r:n,r)]     = app_giv(T(r:n,i),T(r:n,r),c,s);

        % Apply rotation to V on the right.
        if (vflag)
          [V(1:n,i),V(1:n,r)] = app_giv(V(1:n,i),V(1:n,r),c,s);
        end
      end
    end
  end
  
else
  % The indefinite case; T is upper triangular.

  % Refinement loop.
  for i = 1:num_ref
    % Flip the off-diagonal block of T using Givens rotations on the right.
    for r = p+1:n
      for i = p:-1:1
        % Apply rotation to T on the right.
        [c,s,T(i,i)] = gen_giv(T(i,i),T(i,r));
        T(i,r) = 0;  % Eliminate T(i,r).
        [T(1:i-1,i),T(1:i-1,r)] = app_giv(T(1:i-1,i),T(1:i-1,r),c,s);
        [T(p+1:r,i),T(p+1:r,r)] = app_giv(T(p+1:r,i),T(p+1:r,r),c,s);

        % Apply rotation to V on the right.
        if (vflag)
          [V(1:n,i),V(1:n,r)] = app_giv(V(1:n,i),V(1:n,r),c,s);
        end
      end
    end

    % Restore upper triangular form with Givens rotations on the left.
    for r = n:-1:p+1
      for i = 1:p
        % Apply quadratic rotation to T on the left.
        [c,s,T(i,i),sgn] = gen_qrot(T(i,i),T(r,i),Omega(i,i),Omega(r,r));
        T(r,i) = 0;  % Eliminate T(r,i).
        [T(i,[i+1:p,r:n]),T(r,[i+1:p,r:n]),Omega(i,i),Omega(r,r)] = ...
          app_qrot(T(i,[i+1:p,r:n]),T(r,[i+1:p,r:n]),c,s,Omega(i,i),Omega(r,r),sgn);
      end
    end
  end

end

%-----------------------------------------------------------------------
% End of function vsv_qrit
%-----------------------------------------------------------------------