function [p,L,Omega,V] = lvsvid(A,tol_rank,max_iter,est_type,fixed_rank) 
%  lvsvid --> Low-rank revealing decompostion of a symmectric indefinite matrix.
%
%  <Synopsis>
%    [p,L,Omega,V] = lvsvid(A)
%    [p,L,Omega,V] = lvsvid(A,tol_rank)
%    [p,L,Omega,V] = lvsvid(A,tol_rank,max_iter)
%    [p,L,Omega,V] = lvsvid(A,tol_rank,max_iter,est_type)
%    [p,L,Omega,V] = lvsvid(A,tol_rank,max_iter,est_type,fixed_rank)
%
%  <Description>
%    Computes a rank-revealing VSV decompostion A = V*(R'*Omega*R)*V' of a
%    symmetric indefinite n-by-n matrix.  Only the upper triangular part needs
%    to be specified.  Optimized for matrices whose rank p is small compared to n.
%
%  <Input parameters>
%    1. A          --> symmetric indefinite matrix;   
%    2. tol_rank   --> rank decision tolerance;
%    3. max_iter   --> max. number of power/Lanczos iterations per deflation step,
%                      used in the singular vector estimator;
%    4. est_type   --> if true, then estimate singular vectors by means of
%                      the Lanczos procedure, else use the power method;
%    5. fixed_rank --> deflate to the fixed rank given by fixed_rank instead of
%                      using the rank decision tolerance;
%
%   Defaults:   tol_rank  = n*norm(A,1)*eps;
%                max_iter = 5;
%                est_type = 0 (power method);
%
%  <Output parameters> 
%    1. p     --> numerical rank of A;
%    2. L     --> lower triangular matrix in A = V*(L'*Omega*L)*V';
%    3. Omega --> signature matrix in A = V*(L'*Omega*L)*V';
%    4. V     --> orthogonal matrix in A = V*(L'*Omega*L)*V';
%
% <Algorithm>
%    The symmetric indefinite matrix A is preprocessed by a pivoted LDL'
%    factorization.  An interim stage (where D is made diagonal) is followed
%    by a rank-revealing ULV-like decomposition, using power or Lanczos
%    iterations for singular vector estimation.
%
%  <See Also>
%    hvsvid   --> High-rank revealing VSV alg. for symmetric indefinite matrices
%    hvsvid_R --> High-rank revealing VSV alg., sym. indef. matrices, R version
%    hvsvsd   --> High-rank revealing VSV alg. for symmetric semidefinite matrices
%    lvsvsd   --> Low-rank revealing VSV alg. for symmetric semidefinite matrices

%  <References> 
%  [1] P.C. Hansen & P.Y. Yalamov, "Computing Symmetric Rank-Revealing
%      Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%      Appl., 23 (2001), pp. 443--458.
%
%  <Revision>
%    P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: March 13, 2004
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 1)
  error('Not enough input arguments.')
end

[m,n] = size(A);
if (m*n == 0)
  error('Empty input matrix A not allowed.')
elseif (m ~= n)
  error('Matrix A must be square.')
end

% Check the optional input arguments, and set defaults.
if (nargin==1)
  tol_rank = n*norm(A,1)*eps;
  max_iter = 5;
  est_type = 0;
  fixed_rank = n;
elseif (nargin==2)
  if isempty(tol_rank), tol_rank = n*norm(A,1)*eps; end
  max_iter = 5;
  est_type = 0;
  fixed_rank = n;
elseif (nargin==3)
  if isempty(tol_rank), tol_rank = n*norm(A,1)*eps; end
  if isempty(max_iter), max_iter = 5; end
  est_type = 0;
  fixed_rank = n;
elseif (nargin==4)
  if isempty(tol_rank), tol_rank = n*norm(A,1)*eps; end
  if isempty(max_iter), max_iter = 5; end
  if isempty(est_type), est_type = 0; end
  fixed_rank = n;
else
  tol_rank = realmin;
  if isempty(max_iter), max_iter = 5; end
  if isempty(est_type), est_type = 0; end
end

% Initial LDLT factorization A = Pi'*L*D*L'*Pi.
A = triu(A) + triu(A,1)';     % Make sure A is symmetric.
[L,D,Pi] = ldlt_symm(A,'r');  % Use rook pivoting.

% Interim process, L*D*L' = W*R'*Omega*R*W'.
[W,R,Omega] = vsvid_ip(L,D);

% Permutations.
L = R(n:-1:1,n:-1:1);
Omega = Omega(n:-1:1,n:-1:1);
if nargout==4, V = eye(n); else V = []; end

% Rank-revealing procedure.

% Estimate the principal singular value and the corresponding right
% singular vector of L'*Omega*L by using the Lanczos or power method.
if (est_type)
  [smax,vmax] = TOTlanczos(L,Omega,max_iter);
else
  [smax,vmax] = TOTpowiter(L,Omega,max_iter);
end

p = 0;                                         % Init. loop to rank zero.
while ((smax >= tol_rank) & (p < fixed_rank))
  % New rank estimate after the problem is deflated.
  p = p + 1;

  % Apply deflation procedure to p'th column of L in the VSV decomposition.
  [L,Omega,V] = lvsvid_cdef(L,Omega,V,p,vmax);
  
  % A block QR iteration reduces the norm of the off-diagonal block.
  [L,Omega,V] = vsv_qrit(p,1,L,Omega,V);

  % Estimate the principal singular value and the corresponding right
  % singular vector of trailing submatrix by using the Lanczos or power method.
  if (p < n)
    if (est_type)
      [smax,vmax] = TOTlanczos(L(p+1:n,p+1:n),Omega(p+1:n,p+1:n),max_iter);
    else
      [smax,vmax] = TOTpowiter(L(p+1:n,p+1:n),Omega(p+1:n,p+1:n),max_iter);
    end
  else
    smax = 0;                           % No (n+1)th singular value
  end
end

% Return optional output.
if (nargout >= 3) 
  V = Pi'*W(:,n:-1:1)*V;
end

%-----------------------------------------------------------------------
% End of function lvsvid
%-----------------------------------------------------------------------