function [W,C,Omega] = vsvid_ip(L,D)
%  vsvid_ip --> Interim process for indefinite VSV algorithm.
%
%  <Synopsis>
%   [W,C,Omega] = vsvid_ip(L,D)
%
%  <Description> 
%    The interim process transforms the block diagonal matrix D into a
%    signature matrix Omega.
%
%  <Input parameters>
%    1-2. L,D --> Factors in LDL' factorization;   
%
%  <Output parameters> 
%    1-3. W,C,Omega --> Matrices in L*D*L' = W*C'*Omega*C*W'.
%
%  <Algorithm>
%    The factorization L*D*L' is replaced with W*C'*Omega*C*W', where W is
%    orthogonal, C is upper triangular, and Omega is a signature matrix.
%    This is accomplished via small eigenvalue decompositions of the 1-by-1
%    and 2-by-2 blocks of D.

%  <References> 
%  [1] P.C. Hansen, & P.Y. Yalamov, "Computing Symmetric Rank-Revealing
%      Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%      Appl., 23 (2001),  pp. 443--458.
%
%  <Revision>
%    J. Bratland, J. Frimodt & P.C. Hansen, IMM, Technical University of Denmark.
%
%    Last revised: March 1, 2004
%-----------------------------------------------------------------------

% Initialize
n = size(L,1);

% Compute eigenvalue decompostion of blocks of D. 
Wbar = eye(n); Lambda = ones(n,1); Lambda(n) = D(n,n);
blocks = []; % Save the location of the 2x2 blocks
i = 1; j = 1;

while(i<n)
  if (D(i,i+1)==0) 
    Lambda(i) = D(i,i);
    i = i+1;
  else
    [w,Lambda_eig] = eig(D(i:i+1,i:i+1)); % Matlab's eigenvalue decomposition
    Wbar(i:i+1,i:i+1) = w; 
    Lambda(i:i+1) = diag(Lambda_eig);
    blocks(j) = i+1; j = j+1;
    i = i+2;
  end
end

% Write lambda as lambda = sqrt(abs(lambda))*sign*sqrt(abs(lambda)).
Omega = zeros(n,1);
Omega(1:n) = sign(Lambda(1:n));
Lambda(1:n) = sqrt(abs(Lambda(1:n)));

% Compute an orthogonal W such that C' = W'*L*Wbar*lambda_new is
% lower triangular.
W = eye(n);
C = L*Wbar*diag(Lambda);

% Givens rotations to make C lower triangular
for i=1:length(blocks)
  k = blocks(i); % Recall where block elements created fill-ins
  [c,s] = gen_giv(C(k,k),C(k-1,k));
  [C(k,:),C(k-1,:)] = app_giv(C(k,:),C(k-1,:),c,s);
  [W(:,k),W(:,k-1)] = app_giv(W(:,k),W(:,k-1),c,s);
end

% Rearrange matrices
C = tril(C)'; % C is upper triangular now.
Omega(find(Omega==0)) = 1;  % Fix zero entries.
Omega = sparse(diag(Omega));

%-----------------------------------------------------------------------
% End of function vsvid_ip
%-----------------------------------------------------------------------