function [p,LA,L,V,UA,UB,vec] = ulliv_up_B(p,LA,L,V,UA,UB,b,tol_rank,...
                                           tol_ref,max_ref,fixed_rank)
%  ulliv_up_B --> Update the B-part of the rank-revealing ULLIV decomposition.
%
%  <Synopsis>
%    [p,LA,L,V,UA,UB,vec] = ulliv_up_B(p,LA,L,V,UA,UB,b)
%    [p,LA,L,V,UA,UB,vec] = ulliv_up_B(p,LA,L,V,UA,UB,b,tol_rank)
%    [p,LA,L,V,UA,UB,vec] = ulliv_up_B(p,LA,L,V,UA,UB,b,tol_rank,...
%                                                    tol_ref,max_ref)
%    [p,LA,L,V,UA,UB,vec] = ulliv_up_B(p,LA,L,V,UA,UB,b,tol_rank,...
%                                                    tol_ref,max_ref,fixed_rank)
%
%  <Description>
%    Given a rank-revealing ULLIV decomposition of the mA-by-n matrix with
%    mA >= n, and the full-rank mB-by-n matrix B = UB*L*V' with mB < n,
%    the function computes the updated decomposition
%
%             A = UA*LA*[ L 0 ]*V'  and  [ B ] = UB*[ L 0 ]*V'
%                       [ 0 I ]          [ b ]
%
%    where b is the new row added to B.  Note that the updated matrix [B;b]
%    must have full row rank, that the row dimension of UB will increase by one,
%    and that the matrices UA and UB can be left out by inserting an empty
%    matrix [] while V is required.
%
%  <Input Parameters>
%    1.   p            --> numerical rank of A*pinv(B)_A revealed in LA;
%    2-6. LA,L,V,UA,UB --> the ULLIV factors of A and B;
%    7.   b            --> the new row added to B;
%    8.   tol_rank     --> rank decision tolerance;
%    9.   tol_ref      --> upper bound on the 2-norm of the off-diagonal block
%                          LA(p+1:mB,1:p) relative to the Frobenius-norm of LA;
%    10.  max_ref      --> max. number of refinement steps per singular value
%                          to achieve the upper bound tol_ref;
%    11.  fixed_rank   --> if true, deflate to the fixed rank given by p
%                          instead of using the rank decision tolerance;
%
%    Defaults: tol_rank = sqrt(n)*norm(LA,1)*eps;
%              tol_ref  = 1e-04;
%              max_ref  = 0;
%
%  <Output Parameters>
%    1.   p            --> numerical rank of updated quotient;
%    2-6. LA,L,V,UA,UB --> the updated ULLV factors;
%    7.   vec          --> a 5-by-1 vector with:
%         vec(1) = upper bound of norm(LA(p+1:mB,1:p)),
%         vec(2) = estimate of pth singular value,
%         vec(3) = estimate of (p+1)th singular value,
%         vec(4) = a posteriori upper bound of num. nullspace angle,
%         vec(5) = a posteriori upper bound of num. range angle.
%
%  <See Also>
%    ulliv_up_A --> Update the A-part of the rank-revealing ULLIV decomp.

%  <References>
%  [1] F.T.Luk and S. Qiao, "A New Matrix Decomposition for Signal
%      Processing", Automatica, 30 (1994), pp. 39--43.
%  [2] F.T.Luk and S. Qiao, "An adaptive algoithm for interference
%      cancelling in array processing; in F.T. Luk (Ed.), "Advanced
%      Signal Processing Algorithms, Architectures, and Implementations
%      VI," SPIE Proceedings, Vol. 2846 (1996), pp. 151-161.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%
%    Last revised: February 12, 2004
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 7)
  error('Not enough input arguments.')
end

[mLA,n] = size(LA);
[mL,nL] = size(L);
[mV,nV] = size(V);
if (mLA*n == 0) | (mL*nL == 0) | (mV*nV == 0)
  error('Empty input matrices LA, L and V not allowed.')
elseif (mLA ~= n) | (mL ~= nL) | (mV ~= nV)
  error('Square matrices LA, L and V required.')
elseif (nV ~= n)
  error('LA and V must have the same dimensions.')
elseif (nL >= n)
  error('L is too large.')
end

[mA,nA] = size(UA);
if (mA*nA == 0)
  uAflag = 0;
elseif (nA ~= n)
  error('Wrong no. of columns in UA.')
elseif (mA < n)
  error('The A-part of the system is underdetermined.');
else
  uAflag = 1;
end

[mB,nB] = size(UB);
if (mB*nB == 0)
  uBflag = 0;
elseif (nB ~= nL)
  error('Wrong no. of columns in UB.')
else
  uBflag = 1;
  UB(mB+1,nB+1) = 1;
end

if (length(b) ~= n)
  error('Not a valid update vector.')
end

if (p ~= abs(round(p))) | (p > nL)
  error('Rank p must be an integer between 0 and size(L,1).')
end

% Check the optional input arguments, and set defaults.
if (nargin == 7)
  tol_rank   = sqrt(n)*norm(LA,1)*eps;
  tol_ref    = 1e-04;
  max_ref    = 0;
  fixed_rank = 0;
elseif (nargin == 8)
  if isempty(tol_rank), tol_rank = sqrt(n)*norm(LA,1)*eps; end
  tol_ref    = 1e-04;
  max_ref    = 0;
  fixed_rank = 0;
elseif (nargin == 9)
  if isempty(tol_rank), tol_rank = sqrt(n)*norm(LA,1)*eps; end
  if isempty(tol_ref), tol_ref  = 1e-04; end
  max_ref    = 0;
  fixed_rank = 0;
elseif (nargin == 10)
  if isempty(tol_rank), tol_rank = sqrt(n)*norm(LA,1)*eps; end
  if isempty(tol_ref), tol_ref  = 1e-04; end
  if isempty(max_ref), max_ref  = 0; end
  fixed_rank = 0;
elseif (nargin == 11)
  if isempty(tol_ref), tol_ref  = 1e-04; end
  if isempty(max_ref), max_ref  = 0; end
  if (fixed_rank)
    tol_rank = realmax;
  else
    if isempty(tol_rank), tol_rank = sqrt(n)*norm(LA,1)*eps; end
    fixed_rank = 0;
  end
end

if (tol_rank ~= abs(tol_rank)) | (tol_ref ~= abs(tol_ref))
  error('Requires positive values for tol_rank and tol_ref.')
end
if (max_ref ~= abs(round(max_ref)))
  error('Requires positive integer value for max_ref.')
end

% Check the number of output arguments.
if (nargout ~= 7)
  vecflag = 0;
else
  vecflag = 1;
end

LL = eye(n); LL(1:nL,1:nL) = L;
A = UA*LA*LL*V';
B = UB*LL(1:nL,:)*V';
Bup = [B;b(:)'];

% Update the decomposition.

% The row vector d is appended to the bottom of LL = [L 0;0 I].
d = b(:)'*V;
  
% Eliminate elements d(nL+2:n) using right rotations
for i=n:-1:nL+2
  % Use d(i-1) to annihilate d(i) by a right rotation.
  [c,s,d(i-1)] = gen_giv(d(i-1),d(i)); d(i) = 0;
  [V(:,i-1),V(:,i)] = app_giv(V(:,i-1),V(:,i),c,s);
  % No change in UB; apply rotation from the right to LA.
  [LA(i-1:n,i-1),LA(i-1:n,i)] = app_giv(LA(i-1:n,i-1),LA(i-1:n,i),c,s); % sign?
  % Restore triang. form of LA using left rotation.
  [c,s,LA(i,i)] = gen_giv(LA(i,i),LA(i-1,i));
  [LA(i,1:i-1),LA(i-1,1:i-1)] = app_giv(LA(i,1:i-1),LA(i-1,1:i-1),c,s); LA(i-1,i)=0;
  % Propagate into UA from the right.
  if (uAflag)
    [UA(:,i-1),UA(:,i)] = app_giv(UA(:,i-1),UA(:,i),c,-s);
  end
end

% Check if updated B-matrix is obviously rank deficient.
if abs(d(nL+1)) < n*eps*norm(L(1:nL,1:nL),1)
    warning('Updated B-matrix is (almost) rank deficient.')
end

% Swap rows nL and n+1 of L, and the same columns of LA (at this stage the size
% of L increases by one).
L(nL+1,1:nL+1) = d(1:nL+1);
LA(:,n+1) = LA(:,nL+1);  % Also swap columns of expanded LA matrix.
LA(:,nL+1) = zeros(n,1);

% Perform a cyclic down-shift of rows p+1:nL+1 of LA, and annihilate
% the horizontal spike in row p+1.
LA([p+1:nL+1],:) = LA([nL+1,p+1:nL],:);
if (uAflag)
    UA(:,[p+1:nL+1]) = UA(:,[nL+1,p+1:nL]);
end
for i=nL:-1:p+2
  [c,s,LA(p+1,i-1)] = gen_giv(LA(p+1,i-1),LA(p+1,i)); LA(p+1,i) = 0;
  [LA(i:n,i-1),LA(i:n,i)] = app_giv(LA(i:n,i-1),LA(i:n,i),c,s);
  % Propagate into UB from the right.
  if (uBflag)
    [UB(:,i-1),UB(:,i)] = app_giv(UB(:,i-1),UB(:,i),c,s);
  end
  % Propagate into L from the left.
  [L(i-1,1:i),L(i,1:i)] = app_giv(L(i-1,1:i),L(i,1:i),c,s);
  % Annihilate fill in L(i-1,i) by right rotation.
  [c,s,L(i-1,i-1)] = gen_giv(L(i-1,i-1),L(i-1,i)); L(i-1,i) = 0;
  [L(i:nL+1,i-1),L(i:nL+1,i)] = app_giv(L(i:nL+1,i-1),L(i:nL+1,i),c,s);
  % Propagate into V.
  [V(:,i-1),V(:,i)] = app_giv(V(:,i-1),V(:,i),c,s);
end

% Eliminate all elements of d(2:nL+1) using interleaved right/left rotations.
for (i = nL+1:-1:2)
  % Swap columns of L (and V) and eliminate fill by left rotation.
  L(:,[i-1,i]) = L(:,[i,i-1]);
  V(:,[i-1,i]) = V(:,[i,i-1]);
  [c,s,L(i,i)] = gen_giv(L(i,i),L(i-1,i)); L(i-1,i) = 0;
  [L(i,1:i-1),L(i-1,1:i-1)] = app_giv(L(i,1:i-1),L(i-1,1:i-1),c,s);
  % Propagate into UB from the right.
  if (uBflag)
    [UB(:,i-1),UB(:,i)] = app_giv(UB(:,i-1),UB(:,i),c,-s);
  end
  % Propagate into LA from the right.
  [LA(i-1:n,i-1),LA(i-1:n,i)] = app_giv(LA(i-1:n,i-1),LA(i-1:n,i),c,-s);
  % Annihilate fill in LA(i-1,i) by left rotation, remember last column.
  [c,s,LA(i,i)] = gen_giv(LA(i,i),LA(i-1,i)); LA(i-1,i) = 0;
  [LA(i,1:i-1),LA(i-1,1:i-1)] = app_giv(LA(i,1:i-1),LA(i-1,1:i-1),c,s);
  [LA(i,n+1),LA(i-1,n+1)] = app_giv(LA(i,n+1),LA(i-1,n+1),c,s);
  % Propagate into UA from the right.
  if (uAflag)
    [UA(:,i),UA(:,i-1)] = app_giv(UA(:,i),UA(:,i-1),c,s);
  end
end

% Eliminate last (unit) element of d, incl. scaling of L(1,1) and LA(1,1),
% and clear up.
LA(:,1) = LA(:,1) + LA(:,n+1)/L(1,1);
LA = LA(:,1:n);
nL = nL+1;

% Make the updated decomposition rank-revealing.

% Initialize.
smin          = 0;                              % No 0th singular value.
smin_p_plus_1 = 0;                              % No (n+1)th singular value.
norm_tol_ref  = norm(LA,'fro')*tol_ref/sqrt(n); % Value used to verify ...
                                                % ... the upper bound tol_ref.

% Use a priori knowledge about rank changes.
if (fixed_rank), p_min = p; else p_min = 0; end

% Apparent increase in rank of LA by 2.
p = min(nL,p+2);

% Estimate of the p'th singular value and the corresponding left
% singular vector via the generalized LINPACK condition estimator.
[smin,umin] = ccvl(LA(1:p,1:p)');

if (smin < tol_rank)
  % The rank increases by one, stays the same or decreases.

  while ((smin < tol_rank) & (p > p_min))
    % Apply deflation procedure to p'th row of LA in the ULLV decomposition.
    [LA,L,V,UA,UB] = ullv_rdef(LA,L,V,UA,UB,p,umin);

    % Refinement loop.
    num_ref = 0;                                % Init. refinement counter.
    while (norm(LA(p,1:p-1)) > norm_tol_ref) & (num_ref < max_ref)
      % Apply one QR-iteration to p'th row of LA in the ULLV decomposition.
      [LA,L,V,UA,UB] = ullv_ref(LA,L,V,UA,UB,p);
      num_ref = num_ref + 1;
    end

    % New rank estimate after the problem has been deflated.
    p = p - 1;
    smin_p_plus_1 = smin;

    % Estimate of the p'th singular value and the corresponding left
    % singular vector via the generalized LINPACK condition estimator.
    if (p > 0)
      [smin,umin] = ccvl(LA(1:p,1:p)');
    else
      smin = 0;                                 % No 0th singular value.
    end
  end
elseif (p < n)
  % The rank increases by two.

  % Refinement loop.
  num_ref = 0;                                  % Init. refinement counter.
  while (norm(LA(p+1,1:p)) > norm_tol_ref) & (num_ref < max_ref)
    % Apply one QR-iteration to p+1'th row of LA in the ULLV decomposition.
    [LA,L,V,UA,UB] = ullv_ref(LA,L,V,UA,UB,p+1);
    num_ref = num_ref + 1;
  end

  % Estimate of the (p+1)th singular value and the corresponding left
  % singular vector via the generalized LINPACK condition estimator.
  [smin_p_plus_1,umin] = ccvl(LA(1:p+1,1:p+1)');
end

% Normalize the columns of V in order to maintain orthogonality.
for (i = 1:n)
  V(:,i) = V(:,i)./norm(V(:,i));
end

% Estimates that describe the quality of the decomposition.
if (vecflag)
  vec    = zeros(5,1);
  vec(1) = sqrt(n-p)*norm(LA(p+1:n,1:p),1);
  vec(2) = smin;
  vec(3) = smin_p_plus_1;
  vec(4) = (vec(1)*smin_p_plus_1)/(smin^2 - smin_p_plus_1^2);
  vec(5) = (vec(1)*smin)/(smin^2 - smin_p_plus_1^2);
end

%-----------------------------------------------------------------------
% End of function ulliv_up_B
%-----------------------------------------------------------------------