function [p,L,V] = lvsvsd(A,tol_rank,max_iter,est_type,fixed_rank)
%  lvsvsd --> Low-rank revealing decompostion of a symmetric semidefinite matrix.
%
%  <Synopsis>
%    [p,L,V] = lvsvsd(A)
%    [p,L,V] = lvsvsd(A,tol_rank)
%    [p,L,V] = lvsvsd(A,tol_rank,max_iter)
%    [p,L,V] = lvsvsd(A,tol_rank,max_iter,est_type)
%    [p,L,V] = lvsvsd(A,tol_rank,max_iter,est_type,fixed_rank)
%
%  <Description>
%    Computes a rank-revealing VSV decompostion A = V*(L'*L)*V' of a symmetric
%    semidefinite n-by-n matrix.  Only the upper triangular part of needs to
%    be specified.  Optimized for matrices whose rank p is small compared to n.
%
%  <Input parameters>
%    1. A          --> symmetric semidefinite matrix;   
%    2. tol_rank   --> rank decision tolerance;
%    3. max_iter   --> max. number of inverse iterations per deflation step,
%                      used in the singular vector estimator;
%    4. est_type   --> if true, then estimate singular vectors by means of
%                      the Lanczos procedure, else use the power method;
%    5. fixed_rank --> deflate to the fixed rank given by fixed_rank instead
%                      of using the rank decision tolerance;
%
%    Defaults:   tol_rank = n*norm(A,1)*eps;
%                max_iter = 5;
%                est_type = 0 (power method);
%
%  <Output parameters> 
%    1. p --> numerical rank of A;
%    2. L --> lower triangular matrix in A = V*(L'*L)*V'; 
%    3. V --> orthogonal matrix in A = V*(L'*L)*V';
%
%  <Algorithm>
%    The symmetric semidefinite matrix A is preprocessed by a pivoted 
%    Cholesky factorization and then postprocessed by a low-rank revealing
%    ULV decomposition, using either power or Lanczos iterations to estimate
%    the dominant singular vectors.  An indefinite matrix results in an error
%    message during the Cholesky factorization.
%
%  <See Also>
%    hvsvid   --> High-rank revealing VSV alg. for symmetric indefinite matrices
%    hvsvid_R --> High-rank revealing VSV alg., sym. indef. matrices, R version
%    hvsvsd   --> High-rank revealing VSV alg. for symmetric semidefinite matrices
%    lvsvid   --> Low-rank revealing VSV alg. for symmetric indefinite matrices

%  <References> 
%  [1] P.C. Hansen & P.Y. Yalamov, "Computing Symmetric Rank-Revealing
%      Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%      Appl., 23 (2001), pp. 443--458.
%
%  <Revision>
%    P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: July 13, 2003
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 1)
  error('Not enough input arguments.')
end

[m,n] = size(A);
if (m*n == 0)
  error('Empty input matrix A not allowed.')
elseif (m ~= n)
  error('Matrix A must be square.')
end

% Check the optional input arguments, and set defaults.
if (nargin == 1)
  tol_rank = n*norm(A,1)*eps;
  max_iter = 5;
  est_type = 0;
  fixed_rank = n;
elseif (nargin == 2)
  if isempty(tol_rank), tol_rank = n*norm(A,1)*eps; end
  max_iter = 5;
  est_type = 0;
  fixed_rank = n;
elseif (nargin == 3)
  if isempty(tol_rank), tol_rank = n*norm(A,1)*eps; end
  if isempty(max_iter), max_iter = 5; end
  est_type = 0;
  fixed_rank = n;
elseif (nargin == 4)
  if isempty(tol_rank), tol_rank = n*norm(A,1)*eps; end
  if isempty(max_iter), max_iter = 5; end
  if isempty(est_type), est_type = 0; end
  fixed_rank = n;
else
  tol_rank = realmin;
  if isempty(max_iter), max_iter = 5; end
  if isempty(est_type), est_type = 0; end
end

if (fixed_rank ~= abs(round(fixed_rank))) | (fixed_rank > n)
  error('Requires fixed_rank to be an integer between 0 and n.')
elseif (tol_rank ~= abs(tol_rank))
  error('Requires positive values for tol_rank.')
end

% Relation between singular values of C and A
tol_rank = sqrt(tol_rank);

% Check the number of output arguments.
if (nargout == 3)
  V = eye(n);
else
  V = [];
end
  
% Initial factorization using a pivoted Cholesky factorization.
A = triu(A) + triu(A,1)';  % Make sure A is symmetric.
[C,Pi] = cholp(A);
L = C(n:-1:1,n:-1:1);

% Rank-revealing procedure

% Estimate the principal singular value and the corresponding left
% singular vector of L by using the Lanczos or power method.
umax = ones(n,1)/sqrt(n);
if (est_type)
  [umax,smax] = lanczos(L,max_iter,umax);
else
  [umax,smax] = powiter(L,max_iter,umax);
end

p = 0;                                         % Init. loop to rank zero.
while ((smax >= tol_rank) & (p < fixed_rank))
  % New rank estimate after the problem is deflated.
  p = p + 1;

  % Apply deflation procedure to p'th column of L in the ULV decomposition.
  [L,V] = ulv_cdef(L,V,[],p,umax);

  % Estimate the principal singular value and the corresponding left
  % singular vector of L(p+1:n,p+1:n) by using the Lanczos or power method.
  if (p < n)
    umax = ones(n-p,1)/sqrt(n-p);
    if (est_type)
      [umax,smax] = lanczos(L(p+1:n,p+1:n),max_iter,umax);
    else
      [umax,smax] = powiter(L(p+1:n,p+1:n),max_iter,umax);
    end
  else
    smax = 0;                           % No (n+1)th singular value
  end
end

if (nargout == 3) 
  V = Pi*V(n:-1:1,:);
end

%-----------------------------------------------------------------------
% End of function lvsvsd
%-----------------------------------------------------------------------