function [L,Omega,V] = lvsvid_cdef(L,Omega,V,r,vmax)
%  lvsvid_cdef --> Deflate one column of L in the low-rank ULV-like VSV algorithm.
%
%  <Synopsis>
%    [L,Omega,V] = lvsvid_cdef(L,Omega,V,r,vmax)
%
%  <Description>
%    Given the VSV decomposition V*L'*Omega*L*V', the function deflates the
%    first column of L(r:n,r:n). vmax is an estimate of the right singular
%    vector of L(r:n,r:n)'*Omega(r:n,r:n)*L(r:n,r:n) associated with the
%    largest singular value sigma_1. On return, the norm of the first column
%    of L(r:n,r:n)'*Omega(r:n,r:n)*L(r:n,r:n) is of the order sigma_1.
%    The matrix V can be left out by inserting an empty matrix [].
%
%  <Input parameters>
%    1. L     --> lower triangular matrix in A = V*(L'*Omega*L)*V';   
%    2. Omega --> signature matrix in A = V*(L'*Omega*L)*V';
%    3. V     --> orthogonal matrix in A = V*(L'*Omega*L)*V';
%    4. r     --> size of submatrix to be deflated;
%    5. vmin  --> estimate of the smallest right singular vector of
%                 the product L(1:r,1:r)'*Omega(1:r,1:r)*L(1:r,1:r);
%
%  <Output parameters>
%    1. L     --> lower triangular matrix in resulting A = V*(L'*Omega*L)*V';
%    2. Omega --> signature matrix in resulting A = V*(L'*Omega*L)*V';  
%    3. V     --> orthogonal matrix in resulting A = V*(L'*Omega*L)*V';

%  <References>
%  [1] R.D. Fierro and P.C. Hansen, "Low-Rank Revealing UTV Decompositions",
%      Numerical Algorithms, 15 (1997), pp. 37--55.
%
%  <Revision>
%    Per Christian Hansen, IMM, Technical University of Denmark
%
%    Last revised: February 29, 2004
%-----------------------------------------------------------------------

% Check the input arguments.
vflag = 1;
if (isempty(V)), vflag = 0; end

% Initialize.
[n,n] = size(L);

for i = n:-1:r+1
  % Transform vmax to e_1.
  [c,s,vmax(i-r)] = gen_giv(vmax(i-r),vmax(i-r+1));

  % Apply rotation to L from the right.
  [L(i-1:n,i-1),L(i-1:n,i)] = app_giv(L(i-1:n,i-1),L(i-1:n,i),c,s);

  % Apply rotation to V, if required.
  if (vflag)
    [V(:,i-1),V(:,i)] = app_giv(V(:,i-1),V(:,i),c,s);
  end
  
  % If hypernormal transformation, use the best conditioned column pair.
  if Omega(i-1,i-1) ~= Omega(i,i)
    d = abs( abs(L(i-1,i-1:i)) - abs(L(i,i-1:i)) );
    if d(2) > d(1)
      L(i-1:n,[i-1,i]) = L(i-1:n,[i,i-1]);
      V(:,[i-1,i]) = V(:,[i,i-1]);
    end
  end

  % Restore L to lower triangular form using quadratic rotation from the left.
  [c,s,L(i,i),sgn] = gen_qrot(L(i,i),L(i-1,i),Omega(i,i),Omega(i-1,i-1));
  L(i-1,i) = 0;                             % Eliminate L(i-1,i).
  [L(i,1:i-1),L(i-1,1:i-1),Omega(i,i),Omega(i-1,i-1)] = ...
    app_qrot(L(i,1:i-1),L(i-1,1:i-1),c,s,Omega(i,i),Omega(i-1,i-1),sgn);
end

%-----------------------------------------------------------------------
% End of function lvsvid_cdef
%-----------------------------------------------------------------------