function lsvdrrdemo
% lsvdrrdemo --> demonstration of lsvdrr for NMR data
%
% The Toeplitz matrix is complex and contains NMR data.  We compute
% the principal k singular values and right singular vectors for
% k = 5, 10, 15 and 20 by means of both lsvdrr and svds, and
% compare with the results from svd.
%
% The data is the file csi-10-10.mat from Data Set 002 at the BioSource
% database of MRS imaging, available at:
%   http://www.esat.kuleuven.ac.be/sista/members/biomed/data002.htm

% Per Christian Hansen, IMM, Technical University of Denmark
% Ricardo D. Fierro, California State University San Marcos
%
% Last revised: February 3, 2005.
%-----------------------------------------------------------------------

% Load the complex signal and create the Toeplitz matrix.
load csi-10-10  % Load signal.
N = length(signal);
Nh = N/2;
m = Nh+1;
n = Nh;
A = toeplitz(signal(Nh:N),signal(Nh:-1:1));
P = [5,10,15,20];

% Display information about the test.
clc
disp('Running lsvdrrdemo:')
disp(['The Toeplitz matrix is complex, of size ',num2str(m),'-by-',num2str(n),...
      ', and contains'])
disp('NMR data from the BioSource database of NMR imaging.  We compute')
disp(['the principal p = ',num2str(P),' singular values and right singular'])
disp('vectors by means of both lsvdrr and svds, and compare with the')
disp('results from svd.')

% Create an object with the Toeplitz data.
T.col = signal(Nh:N);
T.row = signal(Nh:-1:1);

% Compute the SVD of the matrix.
if exist('csidata.mat')
    load csidata  % This loads Vref and sref
else
    tic
    [Vref,sref] = svd(A');
    t_svd = toc;
    sref = diag(sref);
    save csidata Vref sref t_svd
end
disp(['Full SVD computed in ',num2str(t_svd),...
      ' seconds; please wait for results ...'])

% Loop over several values of k (no. of desired singular values).
for i=1:length(P)
    
    p = P(i);

    % Compute the principal singular triplets by means of svds.
    tic 
    [UU,SS,VV] = svds(A,p);
    t(i,1) = toc;
    
    % Compute the principal singular values and right singular
    % vectors by means of lsvdrr.  Also compute approximate left
    % singular vectors and improved right singular vectors. 
    tic
    tol = 1e-10;  % Use approximately same tolerance as in svds.
    [s,W,nits,ntst,errests] = lsvdrr(T,p,tol);
    lambda = tprodinit(T.col,T.row);  % Compute A*W;
    TW = zeros(m,p);
    for j=1:p
        TW(:,j) = tprod(lambda,m,n,W(:,j));
    end
    [U,S,V] = svd(TW);  % SVD of A*W.
    V = W*V;
    t(i,2) = toc;

    sacc(i,1) = norm(sref(1:p)-diag(SS),1); % svds
    sacc(i,2) = norm(sref(1:p)-diag(S) ,1);  % lsvdrr
    
    Vacc(i,1) = subspace(Vref(:,1:p),VV); % svds
    Vacc(i,2) = subspace(Vref(:,1:p),V);  % lsvdrr
        
end

% Plot the results.
figure(1), clf
xtl = ' ';
for i=1:length(P), xtl = strvcat(xtl,num2str(P(i))); end
subplot(2,2,1)
  stem(t(:,1),'-o'), axis([0 length(P)+1 0 max(t(:,1))*1.1])
  set(gca,'xticklabel',[])
  title('svds run time (sec)')
subplot(2,2,2)
  stem(t(:,2),'-*'), axis([0 length(P)+1 0 max(t(:,2))*1.1])
  set(gca,'xticklabel',[])
  title('lsvdrr run time (sec)')
subplot(2,2,3)
  semilogy(1:length(P),sacc(:,1),'o',1:length(P),sacc(:,2),'*')
  axis([0 length(P)+1 min(min(sacc))/2 max(max(sacc))*2])
  set(gca,'xticklabel',xtl)
  title('Max error in singular values')
  xlabel('p')
subplot(2,2,4)
  semilogy(1:length(P),Vacc(:,1),'o',1:length(P),Vacc(:,2),'*')
  axis([0 length(P)+1 min(min(Vacc))/5 max(max(Vacc))*5])
  set(gca,'xticklabel',xtl)
  title('Subspace angle')
  legend('svds','lsvdrr',4)
  xlabel('p')