function [p,L,V] = hvsvsd(A,tol_rank,fixed_rank)
%  hvsvsd --> High-rank revealing decompostion of a symmetric semidefinite matrix.
%
%  <Synopsis>
%    [p,L,V] = hvsvsd(A)
%    [p,L,V] = hvsvsd(A,tol_rank)
%    [p,L,V] = hvsvsd(A,tol_rank,fixed_rank)
%
%  <Description>
%    Computes a rank-revealing VSV decomposition A = V*(L'*L)*V' of a symmetric
%    semidefinite n-by-n matrix.  Only the upper triangular part needs to be
%    specified.  Optimized for matrices whose rank p is close to n.
%
%  <Input parameters>
%    1. A          --> symmetric semidefinite matrix;   
%    2. tol_rank   --> rank decision tolerance;
%    3. fixed_rank --> deflate to the fixed rank given by fixed_rank instead
%                      of using the rank decision tolerance;
%
%    Defaults:   tol_rank = n*norm(A,1)*eps;
%
%  <Output parameters> 
%    1. p --> numerical rank of A;
%    2. L --> lower triangular matrix in A = V*(L'*L)*V'; 
%    3. V --> orthogonal matrix in A = V*(L'*L)*V';
%
%  <Algorithm>
%    The symmetric semidefinite matrix A is preprocessed by a pivoted 
%    Cholesky factorization and then postprocessed by a high-rank-revealing
%    ULV decomposition.  An indefinite matrix results in an error message
%    during the Cholesky factorization.
%
%  <See Also>
%    hvsvid_L --> High-rank revealing VSV alg. for sym. indef. matrices, L version
%    hvsvid_R --> High-rank revealing VSV alg. for sym. indef. matrices, R version
%    lvsvid   --> Low-rank revealing VSV alg. for symmetric indefinite matrices
%    lvsvsd   --> Low-rank revealing VSV alg. for symmetric semidefinite matrices

%  <References> 
%  [1] P.C. Hansen & P.Y. Yalamov, "Computing Symmetric Rank-Revealing
%      Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%      Appl., 23 (2001), pp. 443--458.
%
%  <Revision>
%    J. Bratland, J. Frimodt & P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: July 13, 2003
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 1)
  error('Not enough input arguments.')
end

[m,n] = size(A);
if (m*n == 0)
  error('Empty input matrix A not allowed.')
elseif (m ~= n)
  error('Matrix A must be square.')
end

% Check the optional input arguments, and set defaults.
if (nargin == 1)
  tol_rank = n*norm(A,1)*eps;
  fixed_rank = 0;
elseif (nargin == 2)
  if isempty(tol_rank), tol_rank = n*norm(A,1)*eps; end
  fixed_rank = 0;
elseif (nargin == 3)
  tol_rank = realmax;
end

if (fixed_rank ~= abs(round(fixed_rank))) | (fixed_rank > n)
    error('Requires fixed_rank to be an integer between 0 and n.')
elseif (tol_rank ~= abs(tol_rank))
  error('Requires positive values for tol_rank.')
end

% Relation between singular values of C and A
tol_rank = sqrt(tol_rank);

% Check the number of output arguments.
if (nargout == 3)
  V = eye(n);
else
  V = [];
end
  
% Initial factorization using a pivoted Cholesky factorization.
A = triu(A) + triu(A,1)';  % Make sure A is symmetric.
[C,Pi] = cholp(A);
L = C(n:-1:1,n:-1:1);

% Rank-revealing procedure

% Estimate the smallest singular value and the corresponding left singular
% vector of L via the generalized LINPACK condition estimator.
[smin,umin] = ccvl(L(1:n,1:n)');

p = n;                                         % Init. loop to full rank n.
while ((smin < tol_rank) & (p > fixed_rank))
  % Apply deflation procedure to p'th row of L in the ULV decomposition.
  [L,V] = ulv_rdef(L,V,[],p,umin);

  % New rank estimate after the problem has been deflated.
  p = p - 1;
  
  % Estimate the smallest singular value and the corresponding left singular
  % vector of L(1:p,1:p) via the generalized LINPACK condition estimator.
  if (p > 0)
    [smin,umin] = ccvl(L(1:p,1:p)');
  else
    smin = 0;                                  % No 0th singular value.
  end
end

if (nargout == 3) 
  V = Pi*V(n:-1:1,:);
end

%-----------------------------------------------------------------------
% End of function hvsvsd
%-----------------------------------------------------------------------