function [R,Omega,V,fail] = hvsvid_cdef(R,Omega,V,r,vmin)
%  hvsvid_cdef --> Deflate one column of R in the high-rank URV-like VSV algorithm.
%
%  <Synopsis>
%    [R,Omega,V,fail] = hvsvid_cdef(R,Omega,V,r,vmin)
%
%  <Description>
%    Given the decomposition V*R'*Omega*R*V', the function deflates the
%    last column of R(1:r,1:r). vmin is an estimate of the right singular
%    vector of R(1:r,1:r)'*Omega(1:r,1:r)*R(1:r,1:r) associated with the
%    smallest singular value sigma_r. On return, the norm of the last column
%    of R(1:r,1:r)'*Omega(1:r,1:r)*R(1:r,1:r) is of the order sigma_r.
%    The matrix V can be left out by inserting an empty matrix [].
%
%  <Input parameters>
%    1. R     --> upper triangular matrix in A = V*(R'*Omega*R)*V';   
%    2. Omega --> signature matrix in A = V*(R'*Omega*R)*V';
%    3. V     --> orthogonal matrix in A = V*(R'*Omega*R)*V';
%    4. r     --> size of submatrix to be deflated;
%    5. vmin  --> estimate of the smallest right singular vector of
%                 the product R(1:r,1:r)'*Omega(1:r,1:r)*R(1:r,1:r);
%
%  <Output parameters> 
%    1. R     --> upper triangular matrix in resulting A = V*(R'*Omega*R)*V';
%    2. Omega --> signature matrix in resulting A = V*(R'*Omega*R)*V';  
%    3. V     --> orthogonal matrix in resulting A = V*(R'*Omega*R)*V';
%    4. fail  --> true if a hypernormal rotation is ill defined;

%  <References>
%  [1] P.C. Hansen & P.Y. Yalamov, "Computing Symmetric Rank-Revealing
%       Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%       Appl., 23 (2001), pp. 450-454.
%
%  <Revision>
%    J. Bratland, J. Frimodt & P.C. Hansen, IMM, Techincal University of Denmark.
%
%    Last revised: March 7, 2004
%-----------------------------------------------------------------------

% Check if V is required.
if isempty(V), vflag = 0; else vflag = 1; end

% Initialize.
n = size(R,1);
thr = 1e-5;  % Threshold of undefined hypernormal rotation.
fail = 0;

for i = 2:r
  % Transform vmin to e_n.
  [c,s,vmin(i)] = gen_giv(vmin(i),vmin(i-1));
  
  % Apply rotation to R on the right.
  [R(1:i,i-1),R(1:i,i)] = app_giv(R(1:i,i-1),R(1:i,i),c,-s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(:,i-1),V(:,i)] = app_giv(V(:,i-1),V(:,i),c,-s);
  end
  
  % If hypernormal transformation is ill defined, resort to fix.
  if Omega(i-1,i-1) ~= Omega(i,i)
    d = abs( abs(R(i-1,i-1)) - abs(R(i,i-1)) );
    if d < thr*norm(R(i-1:i,i))
      fail = 1;
      R(:,[i-1:i+1]) = R(:,[i,i+1,i-1]);
      V(:,[i-1:i+1]) = V(:,[i,i+1,i-1]);
      [c,s,R(i-1,i-1),sgn] = gen_qrot(R(i-1,i-1),R(i,i-1),Omega(i-1,i-1),Omega(i,i));
      R(i,i-1) = 0;  % Eliminate R(i,i-1).
      [R(i-1,i:n),R(i,i:n),Omega(i-1,i-1),Omega(i,i)] = ...
        app_qrot(R(i-1,i:n),R(i,i:n),c,s,Omega(i-1,i-1),Omega(i,i),sgn);
      [c,s,R(i,i),sgn] = gen_qrot(R(i,i),R(i+1,i),Omega(i,i),Omega(i+1,i+1));
      R(i+1,i) = 0;  % Eliminate R(i+1,i).
      [R(i,i+1:n),R(i+1,i+1:n),Omega(i,i),Omega(i+1,i+1)] = ...
        app_qrot(R(i,i+1:n),R(i+1,i+1:n),c,s,Omega(i,i),Omega(i+1,i+1),sgn);
      break
    end
  end

  if ~fail
    % Restore R to upper triangular form using hypernormal rotation
    % on the left. The rotation is applied to Omega as well, if needed.
    [c,s,R(i-1,i-1),sgn] = gen_qrot(R(i-1,i-1),R(i,i-1),Omega(i-1,i-1),Omega(i,i));
    R(i,i-1) = 0;  % Eliminate R(i,i-1).
    [R(i-1,i:n),R(i,i:n),Omega(i-1,i-1),Omega(i,i)] = ...
      app_qrot(R(i-1,i:n),R(i,i:n),c,s,Omega(i-1,i-1),Omega(i,i),sgn);
  end
  
end

%-----------------------------------------------------------------------
% End of function hvsvid_cdef
%-----------------------------------------------------------------------