function [p,L,P,Q] = hqlp(A,gap_tol)
%  hqlp --> High-rank gap-revealing QLP factorization.
%
%  <Synopsis>
%    [p,L,P,Q] = hqlp(A)
%    [p,L,P,Q] = hqlp(A,gap_tol)
%
%  <Description>
%    Computes a gap-revealing factorization A = Q*L*P', in which Q and P
%    are orthogonal matrices, and L is a lower triangular matrix whose
%    diagonal elements sometimes approximate the singular values of A.
%    Also returns the largest p such that abs(L(p,p)/L(p+1,p+1)) > gap_tol.
%    Designed for high-rank matrices; use lqlp for low-rank matrices.
%
%  <Input parameters>
%    1. A       --> general matrix;
%    2. gap_tol --> tolerance for gap detection;
%
%    Defaults: gap_tol = min(size(A))/eps;
%
%  <Output parameters> 
%    1. p --> estimate of numerical rank of A;
%    2. L --> lower triangular matrix in A = Q*L*P'; 
%    3. P --> right orthogonal matrix;
%    4. Q --> left orthogonal matrix;
%
%  <Algorithm>
%    A pivoted QR factorization A*Pi_p = Q*R is followed by a pivoted QR
%    factorization R'*Pi_q = P*L'; thus A = (Q*Pi_q)*L*(Pi_p*P)'.  The diagonal
%    elements of L sometimes track the singular values of A, but this is not
%    guaranteed; hence the factorization cannot be guaranteed to reveal rank.
%
%  <See Also>
%    lqlp --> pivoted QLP matrix approximation with interleaved factorizations.

%  <References> 
%  [1] G.W. Stewart, "The QLP approximation to the singular value
%      decomposition," SIAM J. Sci. Comp., 20 (1999), pp. 1336-1348.
%
%  <Revision>
%    P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: September 3, 2003
%-----------------------------------------------------------------------

% Check input parameters.
if (nargin == 1)
  gap_tol = min(size(A))/eps;
end
if (gap_tol < 1)
  error('gap_tol must be greater than one.')
end

% Check the required input arguments.
[m,n] = size(A);
if (m*n == 0)
  error('Empty input matrix A not allowed.')
end
  
% First pivoted QR factorization (economy size version).
[Q,L,pip] = qr(A,0);
if nargout < 4, clear Q, end

% Second pivoted QR factorization (economy size version);
[P,L,piq] = qr(L',0);
L = L';
if nargout > 2
  P(pip,:) = P;
  if nargout == 4
    Q = Q(:,piq);
  end
end

% Estimate numerical rank.
d = abs(diag(L));
rho = d(1:end-1)./d(2:end);
p = find(rho > gap_tol);
if isempty(p)
  p = min(size(A));
else
  p = p(end);
end

%-----------------------------------------------------------------------
% End of function hqlp
%-----------------------------------------------------------------------