function [u1,u2] = app_hyp(v1,v2,c,s,sgn)
%  app_hyp --> Apply a stabilized hyperbolic rotation (left/right).
%
%  <Synopsis>
%    [u1,u2] = app_hyp(v1,v2,c,s,sgn)
%
%  <Description>
%    Apply a stabilized hyperbolic rotation, defined by the parameters
%    c and s, from the left to the row vectors v1 and v2 such that
%
%                   [u1] = [ ch -sh ] [v1]
%                   [u2]   [-sh  ch ] [v2]
%
%    or from the right to the column vectors v1 and v2 such that
%
%                [u1 u2] = [v1 v2] [ ch -sh ]
%                                  [-sh  ch ]
%    where ch = 1/s and sh = c/s.
%
%  <See Also>
%    gen_hyp --> Determine a 2-by-2 stabilized hyperbolic rotation.

%  <References>
%  [1] S.T. Alexander, C.-T. Pan & R.J. Plemmons, "Analysis of recursive
%      least squares hyperbolic rotation algorithms for signal processing,"
%      Lin. Alg. Appl. 98 (1998), 3-40.
%
%  <Revision>
%    Per Christian Hansen, IMM, Technical University of Denmark
%
%    Last revised: March 28, 1999
%-----------------------------------------------------------------------

if (sgn >= 0)
  u1 =  v1/s - v2*c/s;
  u2 = -u1*c + v2*s;
else
  u1 =  v2/s - v1*c/s;
  u2 = -u1*c + v1*s;
end

%-----------------------------------------------------------------------
% End of function app_hyp
%-----------------------------------------------------------------------