function [smin,vmin] = TOTinviter(T,Omega,itmax) 
%  TOTinviter --> Inverse iterations for T'*Omega*T.
% 
%  <Synopsis>
%    [smin,vmin] = TOTinviter(T,Omega,itmax) 
% 
%  <Description>
%    Inverse iterations on the product T'*Omega*T to compute the smallest
%    singular value and the corresponding singular vector.
% 
%  <Input parameters>
%    1.  T     --> triangular matrix;
%    2.  Omega --> diagonal matrix;
%    3.  itmax --> maximum number of iterations;
%
%  <Output parameters>
%    1.  smin  --> estimate of smallest singular value;
%    2.  vmin  --> estimate of corresponding singular vector;

%  <Revision>
%    J. Bratland, J. Frimodt & P.C. Hansen, IMM, Technical University of Denmark.
%
%    Last revised: July 10, 2003
%-----------------------------------------------------------------------

% Initialization.
[q,n]  = size(T);
if q~=n, error('T matrix must be square'), end
reltol = 1e-5;  % Relative tolerance of estimate.

% If T is a column or a scalar.
if (n == 1)
  vmin = 1;
  smin = abs(T);
  return;
end
 
% Replace zero elements in Omega with small elements. 
s = diag(Omega); 
z = find(s==0); 
s(z) = eps*ones(size(z));

% Fix very small diagonal entries of T.
tol = sqrt(n)*norm(T,1)*eps;
p = find(abs(diag(T)) < tol);
for (i = 1:length(p))
  T(p(i),p(i)) = tol;
end
 
% Inverse iterations with T'*Omega*T. 
vmin = ones(n,1)/sqrt(n);
smin = 1;
sminold = 0; 
it = 0; 
while (abs(sqrt(sminold/smin)-1) > reltol & it < itmax) 
  it = it+1;
  z = (vmin'/T)'; 
  z = z./s; 
  z = T\z; 
  sminold = smin; 
  smin = norm(z); 
  vmin = z/smin; 
end

smin = 1/smin;

%-----------------------------------------------------------------------
% End of function TOTinviter
%-----------------------------------------------------------------------