function [p,L,V] = vsvsd_up(p,L,V,v,tol_rank,fixed_rank)
%  vsvsd_up --> Rank-one update of VSV decomposition of semidefinite matrix.
%
%  <Synopsis>
%    [p,L,V] = vsvsd_up(p,L,V,v)
%    [p,L,V] = vsvsd_up(p,L,V,v,tol_rank)
%    [p,L,V] = vsvsd_up(p,R,V,v,tol_rank,fixed_rank)
%
%  <Description> 
%    Given a rank-revealing VSV decomposition of a symmetric semidefinite
%    matrix A = V*(L'*L)*V', the function computes the updated rank-revealing
%    decomposition of A + v*v'.  Use function vsvid_mod with omega = -1 to
%    downdate the VSV decomposition of a symmetric semidefinite matrix.
%
%  <Input Parameters>
%    1.   p          --> numerical rank of A;
%    2.   L          --> lower triangular matrix in A = V*(L'*L)*V';
%    3.   V          --> orthogonal matrix in A = V*(L'*L)*V';
%    4.   v          --> rank-one update column vector;
%    5.   tol_rank   --> rank decision tolerance;
%    6.   fixed_rank --> deflate to the fixed rank given by fixed_rank instead
%                        of using the rank decision tolerance;
%
%    Defaults:   tol_rank = n*norm(L,1)*eps;
%              
%  <Output Parameters>
%    1. p --> numerical rank of updated A.
%    2. L --> updated lower triangular matrix.
%    3. V --> updated orthogonal matrix.
%
%  <See Also>
%    vsvid_L_mod --> Rank-one mod. of VSV decomp. of sy,. indef. matrix, L version.
%    vsvid_R_mod --> Rank-one mod. of VSV decomp. of sy,. indef. matrix, R version.

%  <References>
%  [1] P.C. Hansen & P.Y. Yalamov, "Computing Symmetric Rank-Revealing
%      Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%      Appl., 23 (2001), pp. 443--458.
%
%  <Revision>
%    J. Bratland, J. Frimodt & P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: July 10, 2003
%-----------------------------------------------------------------------

% Check input
if (nargin < 4)
   error('Not enough input parameters.');
end

[mL, nL] = size(L);
[mV, nV] = size(V);
[mv, nv] = size(v);

if (mL*nL == 0) | (mV*nV == 0)
  error('Empty input matrices L and V not allowed.')
elseif (mL ~= nL | mV ~= nV )
   error('Input matrix L and V must be square.');
elseif (mL ~= mV)
   error('Input matrix L and V must have same size.')
elseif (min(mv,nv) ~= 1 & max(mv,nv) ~= mL & nv ~= 1)
   error('Wrong dimensions of update column vector v.')
elseif (p ~= abs(round(p))) | (p > nL)
  error('Requires the rank p to be an integer between 0 and n.')
end

% Set defaults
if (nargin == 4) 
  tol_rank = nL*norm(L,1)*eps;
  fixed_rank = 0;
elseif (nargin == 5)
  if isempty(tol_rank), tol_rank = nL*norm(L,1)*eps; end
  fixed_rank = 0;
elseif (nargin == 6)
  tol_rank = realmax;
end

if (tol_rank ~= abs(tol_rank))
  error('Requires positive value for tol_rank.')
end

% Perform ULV update of L and V.
v = v(:);
[p,L,V] = ulv_up(p,L,V,[],v',1,sqrt(tol_rank),[],[],fixed_rank);

%-----------------------------------------------------------------------
% End of function vsvsd_up
%-----------------------------------------------------------------------