function [L,Omega,V] = hvsvid_rdef(L,Omega,V,r,vmin)
%  hvsvid_rdef --> Deflate one row of L in the high-rank ULV-like VSV algorithm.
%
%  <Synopsis>
%    [L,V,Omega] = hvsvid_rdef(L,V,Omega,r,vmin)
%
%  <Description>
%    Given the decomposition V*L'*Omega*L*V', the function deflates the
%    last row of L(1:r,1:r). vmin is an estimate of the right singular
%    vector of L(1:r,1:r)'*Omega(1:r,1:r)*L(1:r,1:r) associated with the
%    smallest singular value sigma_r. On return, the norm of the last column
%    of L(1:r,1:r)'*Omega(1:r,1:r)*L(1:r,1:r) is of the order sigma_r.
%    The matrix V can be left out by inserting an empty matrix [].
%
%  <Input parameters>
%    1. L     --> lower triangular matrix in A = V*(L'*Omega*L)*V';   
%    2. Omega --> signature matrix in A = V*(L'*Omega*L)*V';
%    3. V     --> orthogonal matrix in A = V*(L'*Omega*L)*V';
%    4. r     --> size of submatrix to be deflated;
%    5. vmin  --> estimate of the smallest right singular vector of
%                 the product L(1:r,1:r)'*Omega(1:r,1:r)*L(1:r,1:r);
%
%  <Output parameters>
%    1.   L     --> upper triangular matrix in resulting V*(L'*Omega*L)*V';  
%    2.   V     --> orthogonal matrix in resulting V*(L'*Omega*L)*V';  
%    3.   Omega --> signature matrix in resulting V*(L'*Omega*L)*V';

%  <References>
%  [1] P.C. Hansen & P.Y. Yalamov, "Computing Symmetric Rank-Revealing
%      Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%      Appl., 23 (2001), pp. 450-454.
%
%  <Revision>
%    P.C. Hansen, IMM, Techincal University of Denmark.
%
%    Last revised: July 12, 2003
%-----------------------------------------------------------------------

% Check if V is required.
if isempty(V), vflag = 0; else vflag = 1; end

% Initialize.
n = size(L,1);

for i = 2:r
  % Transform vmin to e_n.
  [c,s,vmin(i)] = gen_giv(vmin(i),vmin(i-1));

  % Apply rotation to L from the right.
  [L(i-1:n,i-1),L(i-1:n,i)] = app_giv(L(i-1:n,i-1),L(i-1:n,i),c,-s);

  % Apply rotation to V on the right.
  if (vflag)
    [V(:,i-1),V(:,i)] = app_giv(V(:,i-1),V(:,i),c,-s);
  end
      
  % Restore L to lower triangular form using hypernormal rotation
  % on the left. The rotation is applied to Omega as well, if needed.
  [c,s,L(i,i),sgn] = gen_qrot(L(i,i),L(i-1,i),Omega(i,i),Omega(i-1,i-1));
  L(i-1,i) = 0;  % Eliminate L(i,i-1).
  [L(i,1:i-1),L(i-1,1:i-1),Omega(i,i),Omega(i-1,i-1)] = ...
    app_qrot(L(i,1:i-1),L(i-1,1:i-1),c,s,Omega(i,i),Omega(i-1,i-1),sgn);
end

%-----------------------------------------------------------------------
% End of function hvsvid_rdef
%-----------------------------------------------------------------------