function [rtzvals,rtzvecs,errests,nconv,num_iter,vnext,smax] = ...
                          getrtzp(A,k,ncrp,Vk,max_iter,vinit,tol,smax)
%  getrtzp --> Compute (additional) Ritz pairs for a cross-product matrix
%
%  <Synopsis>
%    [rtzvals,rtzvecs,errests,nconv,num_iter,vnext,smax] = ...
%                     getrtzp(A,k,ncrp,Vk,max_iter,vinit,tol,smax)
%
%  <Description>
%    Applies Lanczos iterations to the matrix A'*A; the Lanczos vectors
%    are explicitly reorthogonalized internally, as well as with respect
%    to an existing set of converged vectors.  A singular value estimate
%    is considered as converged when an error estimate is below smax*tol.
%
%  <Input parameters>
%    1. A        --> matrix;
%    2. k        --> number of desired singular values;
%    3. ncrp     --> number of converged Ritz pairs so far;
%    4. Vk       --> previously converged Ritz vectors;
%    5. max_iter --> max. no. of Lanczos iterations in this call to getrtzp;
%    6. vinit    --> start vector;
%    7. tol      --> relative residual tolerance;
%    8. smax     --> current estimate of largest singular value;
%
%  <Output parameters> 
%    1. rtzvals  --> converged Ritz values og A'*A;
%    2. rtzvecs  --> corresponding converged Ritz vectors;
%    3. errests  --> corresponding error estimates of singular values;
%    4. nconv    --> number of converged Ritz pairs;
%    5. num_iter --> number of iterations used in the call to getrtzp;
%    6. vnext    --> vector for next call to getrtzp;
%    7. smax     --> updated value of smax;

%  <References>
%  [1] R.D. Fierro and E.P. Jiang, "Lanczos and the Riemannian SVD in
%      information retrieval applications," Numer. Lin. Alg. Appl., to appear.
%
%  <Revision>
%    R.D. Fierro, Dept. of Mathematics, CSUSM
%    P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: February 21, 2004.
%-----------------------------------------------------------------------

% Initialize.
if isstruct(A)
    m = length(A.col);
    n = length(A.row);
else
    [m,n] = size(A);
end
T  = zeros(max_iter+1,max_iter+1);  % Tridiagonal matrix.
VT = zeros(n,max_iter+1);           % Ritz vectors.
minbeta = 100*eps*smax^2;  % Threshold for small off-diag. elements in T.

% Prepare for iterations.
i = 0; 
v = zeros(n,1);
vv = vinit;
alpha = realmax;
beta = 0;
VT(:,1) = vv;

while i < max_iter
   
   i = i+1;
   
   if isstruct(A)
       w = tprod(A.lambda,m,n,vv);
       w = tprod(A.lambda,m,n,w,1);
   else
       w = A'*(A*vv);
   end
   w = w - beta*v;
   alpha = vv'*w;
   T(i,i) = alpha;
   w = w - alpha*vv;
                  
   % Reorthog. vs converged Ritz vectors and current Lanczos vectors.
   w = w - Vk*(Vk'*w);
   w = w - VT(:,1:i)*(VT(:,1:i)'*w);
   beta = norm(w);
   
   % Guard against "breakdown."
   if beta < minbeta
      w = randn(n,1);
      w = w/norm(w);
      w = w - Vk*(Vk'*w);
      w = w - VT(:,1:i)*(VT(:,1:i)'*w);
      w = w/norm(w);
   else
      w = w/norm(beta);
   end

   T(i+1,i) = beta;
   T(i,i+1) = beta;
   v = vv;
   vv = w;
   VT(:,i+1) = vv;

end % Lanczos-iterations.
num_iter = i;

% Eigenvalue decomposition; sort eigenvalues in descending order.
[V,D] = eig(T(1:num_iter,1:num_iter));
[D,I] = sort(-abs(diag(D)));
ritzvals = -D;  % Ritz values.
V = V(:,I);
ritzvecs = VT(:,1:num_iter)*V;  % Ritz vectors.

smax = max(smax,sqrt(max(ritzvals)));  % Update smax.

% Locate converged Ritz values via error estimates.
errest = abs(beta*V(num_iter,:)')./sqrt(abs(ritzvals) + eps);
I = find(errest < smax*tol);
nconv = length(I);

if nconv == 0  % No converged Ritz pairs.
   rtzvals = [];
   temp = ritzvecs(:,1);
   temp = temp - Vk*(Vk'*temp);
   rtzvecs = temp/norm(temp);
   errests = [];
   vnext  = [];
   return
end

% Store just enough converged Ritz pairs (or all, if in "safety call").
if ncrp < k, nconv = min(k-ncrp,nconv); end
I = I(1:nconv);
rtzvals = ritzvals(I); 
rtzvecs = ritzvecs(:,I);
errests = errest(I);

% Compute the vector vnext for next call to getrtzp.
I = find(errest > smax*tol);
if isempty(I)
   temp  = randn(n,1);
   vnext = temp/norm(temp);
else
   vnext = ritzvecs(:,min(I));
end

% -------------------------------------------------------------------
% End of getrtzp
% -------------------------------------------------------------------