function [c,s,r,sgn] = gen_hyp(a,b)
%  gen_hyp --> Determine a 2-by-2 stabilized hyperbolic rotation matrix.
%
%  <Synopsis>
%    [c,s,r,sgn] = gen_hyp(a,b)
%
%  <Description>
%    Compute a stabilized hyperbolic rotation to annihilate b using a, i.e.,
%    compute parameters c, s, and r such that
%
%        [ ch -sh ] [a] = [r]   with   [ ch -sh ] S [ ch -sh ] = sgn*S
%        [-sh  ch ] [b]   [0]          [-sh  ch ]   [-sh  ch ]
%
%   where ch = 1/s and sh = c/s, and where the signature matrix S is either
%
%        S = [-1  0 ]   or   S = [ 1  0 ] .
%            [ 0  1 ]            [ 0 -1 ]
%
%  <See Also>
%    app_hyp --> Apply a stabilized hyperbolic rotation.

%  <References>
%  [1] S.T. Alexander, C.-T. Pan & R.J. Plemmons, "Analysis of recursive
%      least squares hyperbolic rotation algorithms for signal processing,"
%      Lin. Alg. Appl. 98 (1998), 3-40.
%
%  <Revision>
%    Per Christian Hansen, IMM, Technical University of Denmark
%
%    Last revised: March 23, 1999
%-----------------------------------------------------------------------

if (b == 0)
  s = 1;
  c = 0;
  r = a;
  sgn = 1;
elseif (abs(a)==abs(b))
  error('Hyperbolic rotation not defined')
else
  if (abs(a) > abs(b))
    c = b/a;
    s = sign(a)*sqrt(1 - c^2);
    r = abs(a*s);
    sgn = 1;
  else
    c = a/b;
    s = sign(b)*sqrt(1 - c^2);
    r = abs(b*s);
    sgn = -1;
  end
end

%-----------------------------------------------------------------------
% End of function gen_hyp
%-----------------------------------------------------------------------