function [smin,vmin] = TOTpowiter(T,Omega,itmax) 
%  TOTpowiter --> Power iterations for T'*Omega*T.
% 
%  <Synopsis>
%    [smin,vmin] = TOTpowiter(T,Omega,itmax) 
% 
%  <Description>
%    Power iterations on the product T'*Omega*T to compute the largest
%    singular value and the corresponding singular vector.
% 
%  <Input parameters>
%    1.  T     --> triangular matrix;
%    2.  Omega --> diagonal matrix;
%    3.  itmax --> maximum number of iterations;
%
%  <Output parameters>
%    1.  smin  --> estimate of largest singular value;
%    2.  vmin  --> estimate of corresponding singular vector;
%
%  <See also>
%    TOTlanczos --> Lanczos procedure for T'*Omega*T

%  <Revision>
%    Per Christian Hansen, IMM, Technical University of Denmark
%
%    Last revised: July 10, 2003
%-----------------------------------------------------------------------

% Initialization.
[n,n]  = size(T); 
reltol = 1e-5;  % Relative tolerance of estimate.

% If T is a scalar.
if (n == 1)
  vmin = T/abs(T);
  smin = abs(T);
  return;
end
 
% Vector with diagonal elements of Omega.
s = diag(Omega); 
 
% Power iterations with T'*Omega*T. 
vmin = ones(n,1)/sqrt(n); 
smin = 1; 
sminold = 0; 
it = 0; 
while (abs(sqrt(sminold/smin)-1) > reltol & it < itmax) 
  it = it+1;
  z = T'*(s.*(T*vmin)); 
  sminold = smin; 
  smin = norm(z); 
  vmin = z/smin; 
end

%-----------------------------------------------------------------------
% End of function TOTpowiter
%-----------------------------------------------------------------------