function lambda = tprodinit(colT,rowT);
%  tprodinit -- > Initialization routine for tprod (Toeplitz matrix-vector product)
%
%  <Synopsis>
%    lambda = tprodinit(colT,rowT)
%
%  <Description>
%    lambda contains the eigenvalues of a related circulant matrix, needed for
%    matrix-vector multiplication with the Toeplitz matrix specified by colT
%    and rowT.  The length of lmbda is a powere of 2.
%
%  <Input Parameters>
%    1. colT --> contains the first column of the Toeplitz matrix;
%    2. rowT --> contains the first row of the Toeplitz matrix;
%
%  <Output Parameters>
%    1. lambda --> contains the eigenvalues of an extended circulant matrix;
%
%  <Algorithm>
%    lambda contains the eigenvalues of the circulant matrix C whose first
%    column is c = [colT;z;rowT(end:-1:2)], where z is a zero columns with
%    dimensions such that length(c) is a power of 2 (for efficiency).
%
%  <See Also>
%    tprod -- > Toeplitz matrix-vector product, using tprodinit

%  <Reference>
%  [1] P.C. Hansen, "Decconvolution and regularization with Toeplitz matrices,"
%      Numer. Algo. 29 (2002), pp. 323-378.
%
%  <Revision>
%    Originally developed by James G. Nagy, Emory University.
%    Slightly modified by R.D. Fierro, CSUSM, and P.C. Hansen, IMM, DTU.
%
%    Last revised: February 21, 2004
%-----------------------------------------------------------------------

% Ensure colT and rowT are column vectors.
colT = colT(:);
mcolT = length(colT);
rowT = rowT(:);
mrowT = length(rowT);

% Find the next power of 2.
p = nextpow2(mcolT + mrowT - 1);
n = 2^p;

% Create the first column of the n-by-n circulant matrix.
c = zeros(n,1);
c(1:mcolT) = colT;
c(n-mrowT+2:n) = flipud(rowT(2:mrowT));

% Compute the eigenvalues of the circulant matrix.
lambda = fft(c);

%-----------------------------------------------------------------------
% End of function tprodinit
%-----------------------------------------------------------------------