function [sk,Vk,nits,nrst,errests] = ...
                      lsvdrr(A,k,tol,max_iter,max_restarts,conserve,safety)
%  lsvdrr --> Lanczos SVD with reorthogonalization and explicit restarts
%
%  <Synopsis>
%    [sk,Vk,nits,nrst,errests] = lsvdrr(A,k,max_iter,max_restarts,conserve,safety)
%
%  <Description>
%    Computes approximations to the k dominant singular triplets, using the
%    Lanczos method with reorthogonalization and explicit restarts.
%    The stopping criterion is that the error in each computed singular value
%    must be smaller than tol times the largest singular value.
%
%  <Input parameters>
%    1. A            --> dense/sparse matrix or structure with Toeplitz matrix;
%    2. k            --> number of desired singular triplets;
%    3. tol          --> tolerance for relative residual of triplets;
%    4. max_iter     --> maximum number of Lanczos iterations per restart;
%    5. max_restarts --> maximum number of Lanczos restarts;
%    6. conserve     --> if 1 then max_iter Lanczos vectors are used in total,
%                        otherwise max_iter vectors are used in each restart;
%    7. safety       --> perform additional safety restarts, after k Ritz
%                        values have converged;
%
%    Defaults: tol          = 1e-04;
%              max_iter     = min(2*k,n)
%              max_restarts = 100;
%              conserve     = 0;
%              safety       = 2;
%
%  <Output parameters>
%    1. sk      --> vector of singular value estimates;
%    2. Vk      --> matrix of right singular vector approximations;
%    3. nits    --> number of times A and A' combined have been referenced;
%    4. nrst    --> number of restarts;
%    5. errests --> vector of error estimates for singular values;
%
%  <Matrix Representation>
%    The input parameter A can be either a matrix (dense or sparse) or
%    a structure that holds information about a Toeplitz matrix:
%      A.col = first column of A
%      A.row = first row of A.

%  <References>
%  [1] R.D. Fierro and E.P. Jiang, "Lanczos and the Riemannian SVD in
%      information retrieval applications," Numer. Lin. Alg. Appl., to appear.
%
%  <Revision>
%    R.D. Fierro, California State University San Marcos
%    P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised:  January 24, 2005.
%-----------------------------------------------------------------------

% Input check.
if isstruct(A)
    m = length(A.col);
    n = length(A.row);
    lambda = tprodinit(A.col,A.row);
    A.lambda = lambda;
else
    [m,n] = size(A);
end
if nargin < 3, tol = 1e-4; end
if nargin < 4, max_iter = min(2*k,n); end
if nargin < 5, max_restarts = 100; end
if nargin < 6, conserve = 0; end
if nargin < 7, safety = 2; end
if isempty(tol), tol = 1e-4; end
if isempty(max_iter), max_iter = min(2*k,n); end
if isempty(max_restarts), max_restarts = 100; end
if isempty(conserve), conserve = 1; end
if isempty(safety), safety = 2; end
if max_iter < k & conserve == 1
    error('max_iter < k not allowed when conserve = 1')
end

% Variable initializations.
smax = 0;  % Estimate of largest singular value.
ncrp = 0;  % Cumulative number of converged Ritz pairs.
nrst = -1; % Number of restarts.
nits = 0;  % Cumulative number of Lanczos steps.
maxit = max_iter;  % Default number of Lanczos iterations.

% Storage allocation.
Vk = zeros(n,k);      % Converged orthogonal Ritz vectors.
emax = zeros(k,1);    % Converged Ritz values.
errests = zeros(k,1); % Error in the Ritz pairs.

% Deterministic starting vector.
vinit = ones(m,1);
if isstruct(A)
    vinit = tprod(lambda,m,n,vinit,1);
else
    vinit = A'*vinit;
end
vinit = vinit/norm(vinit);

% Main loop starts here.
while ncrp < k
   if conserve == 1, maxit = max_iter - ncrp; end
   [rtzvals,rtzvecs,ests,nconv,num_iter,vnext,smax] = ...
            getrtzp(A,k,ncrp,Vk(:,1:ncrp),maxit,vinit,tol,smax);
   nrst = nrst + 1;
   nits = nits + num_iter;  % Number of Lanczos iterations so far.
   if nconv == 0   % No converged ritzpairs, explicit restart.
      vinit = rtzvecs;
   else
      emax(ncrp+1:ncrp+nconv,1) = rtzvals;  % Store new Ritz values.
      Vk(:,ncrp+1:ncrp+nconv) = rtzvecs;    % Store new Ritz vectors.
      errests(ncrp+1:ncrp+nconv,1) = ests;  % Store error estimates.
      ncrp = ncrp + nconv;  % Update the number of converged Ritz pairs.
      vinit = vnext - Vk(:,1:ncrp)*(Vk(:,1:ncrp)'*vnext);
      vinit = vinit/norm(vinit);
   end
end % while loop

% Sort the eigenvalues and eigenvectors.
[emax,I] = sort(-emax);
emax = -emax;
Vk = Vk(:,I(1:k));

% Additional restarts improve the likelihood that no intermediate
% singular values are missing.
for i=1:safety
   [rtzvals,rtzvecs,ests,nconv,num_iter,vnext,smax] = ...
            getrtzp(A,k,ncrp,Vk(:,1:ncrp),maxit,vinit,tol,smax);
   nrst = nrst + 1;
   nits = nits + num_iter;
   if max(rtzvals) > min(emax) % Swap Ritz values.
       [emax,I] = sort(-[emax;rtzvals]);
       emax = -emax(1:k);
       I = I(1:k);
       Vk = [Vk(:,I(find(I<=k))),rtzvecs(:,I(find(I>k))-k)];
       errests = [errests(I(find(I<=k)));ests(I(find(I>k))-k)];
   end
end

sk = sqrt(emax);  % Singular values are square roots of eigenvalues.

%-------------------------------------------------------------------------
% End of lsvdrr
%-------------------------------------------------------------------------