function [R, P, I] = cholp(A, piv)
%CHOLP  Cholesky factorization with pivoting of a positive semidefinite matrix.
%       [R, P] = CHOLP(A) returns an upper trapezoidal matrix R and a
%       permutation matrix P such that R'*R = P'*A*P.  Only the upper
%       triangular part of A is used.
%
%       [R, P, I] = CHOLP(A) returns also a positive integer I equal to the
%       number of rows in R.  R is I-by-N with P'*A*P - R'*R zero in
%       columns 1:I and rows 1:I.
%       [R, I] = CHOLP(A, 0) forces P = EYE(SIZE(A)) and therefore behaves
%       similarly to [R, I] = CHOL(A).
%
%       Note: this function does not attempt to detect non-positive
%       semidefinite A.

%       This routine is based on the LINPACK routine CCHDC.  It works
%       for both real and complex matrices.
%
%       Reference:
%       N. J. Higham, Accuracy and Stability of Numerical Algorithms,
%       Second edition, Society for Industrial and Applied Mathematics,
%       Philadelphia, PA, 2002; sec. 10.3.

if nargin == 1, piv = 1; end

n = length(A);
pp = 1:n;
I = n;
tol = (n*eps/2)*max(diag(A));

for k = 1:n

    if piv
       d = diag(A);
       [big, m] = max( d(k:n) );
       m = m+k-1;
    else
       big = A(k,k);  m = k;
    end
%    [big, tol]
    if big <= tol, I = k-1; break, end

%   Symmetric row/column permutations.
    if m ~= k
       A(:, [k m]) = A(:, [m k]);
       A([k m], :) = A([m k], :);
       pp( [k m] ) = pp( [m k] );
    end

    if big == 0
      if norm(A(k+1:n,k)) ~= 0
         I = k; break
      else
         continue
      end
    end

    A(k,k) = sqrt( A(k,k) );
    if k == n, break, end
    A(k, k+1:n) = A(k, k+1:n) / A(k,k);

%   For simplicity update the whole of the remaining submatrix (rather
%   than just the upper triangle).

    j = k+1:n;
    A(j,j) = A(j,j) - A(k,j)'*A(k,j);

end

R = triu(A);
if I > 0
    R = R(1:I,:);
end

if piv == 0
   P = I;
else
   P = eye(n); P = P(:,pp);
end