function vsviddemo
% vsviddemo --> demonstration of symmetric indefinite VSV computations
%
% Generate a batch of rank-deficient KKT matrices; compute the VSV
% decomposition of each matrix by means of hvsvid_L, hvsvid_R and
% lvsvid, and show a histogram of backward errors and subspace angles
% between SVD- and VSV-based null spaces.

% Per Christian Hansen, IMM, Technical University of Denmark
%
% Last revised: February 3, 2005.
%-----------------------------------------------------------------------

% Problem parameters.
n = 12;        % Size a 1,1-block.
q = 2;         % Number of columns in 1,2-block.
N = 500;       % Number of test problems.
thr = 1e-5;    % Threshold for rank decision.
delta = 1e-7;  % Size of "small" singular values.

% Display information about the test.
clc
disp('Running vsviddemo:')
disp(['Generate a batch of ',num2str(N),...
      ' rank-deficient KKT matrices of order ',num2str(n+q)])
disp(['and with a 1,1-block of order ',num2str(n),...
      '.  Then compute the VSV decomposition'])
disp('of each matrix by means of hvsvid_L, hvsvid_R and lvsvid, and show')
disp('a histogram of backward errors and subspace angles between SVD- and')
disp('VSV-based null spaces.  Please wait for results ...')

% Loop over all test problems.
rnk = zeros(N,3);  % Storage for failure reports.
hwait = waitbar(0,'vsviddemo');
for k=1:N
    
    waitbar(k/N,hwait);

    % Generate high-rank KKT matrix.
    r = 9;   % Rank of 1,1-block.
    [Q,R] = qr(rand(n));
    s = [rand(r,1)+1e-10;delta*rand(n-r,1)];
    A = Q*diag(s)*Q';
    B = rand(q,n)*A;
    A = [A,B';B,zeros(q)];
    [UU,SS,VV] = svd(A);
    rng = sum(diag(SS) > thr);
    VV = VV(:,rng+1:end);
    
    % Compute high-rank VSV decompositions and backward errors.
    [pR,R,Omega,V] = hvsvid_R(A,thr);
    if pR~=rng, rnk(k,1) = pR; end
    E(k,1) = norm(A - V*R'*Omega*R*V');
    Theta(k,1) = max(eps,subspace(V(:,rng+1:end),VV));
    [pL,L,Omega,V] = hvsvid_L(A,thr);
    if pL~=rng, rnk(k,2) = pL; end
    E(k,2) = norm(A - V*L'*Omega*L*V');
    Theta(k,2) = max(eps,subspace(V(:,rng+1:end),VV));

    % Generate low-rank KKT matrix.
    r = 2;   % Rank of 1,1-block.
    s = [rand(r,1)+1e-10;delta*rand(n-r,1)];
    A = Q*diag(s)*Q';
    B = rand(q,n)*A;
    A = [A,B';B,zeros(q)];
    [UU,SS,VV] = svd(A);
    rng = sum(diag(SS) > thr);
    VV = VV(:,rng+1:end);
    
    % Compute low-rank VSV decomposition and backward errors.
    [pL,L,Omega,V] = lvsvid(A,thr);
    if pL~=rng, rnk(k,3) = pL; end
    E(k,3) = norm(A - V*L'*Omega*L*V');
    Theta(k,3) = max(eps,subspace(V(:,rng+1:end),VV));

end
close(hwait)

% Report rank error.
if norm(rnk,1)~=0, rnk, return, end

% Plot the results.
figure(1), clf
subplot(2,1,1)
hist(log10(E))
legend('hvsvid\_R','hvsvid\_L','lvsvid')
title('Histogram of backward errors  || A - V T^T \Omega T V^T ||_2')
subplot(2,1,2)
hist(log10(Theta))
legend('hvsvid\_R','hvsvid\_L','lvsvid')
title('Histogram of subspace angles  sin\Theta( Null(A) , V_2 )')
xlabel(['log_{10}(errors) of ',num2str(N),' test problems'])