function y = tprod(lambda,m,n,x,transp)
%  tprod -- > Toeplitz matrix-vector multiplication via FFT.
%
%  <Synopsis>
%    y = tprod(lambda,m,n,x,transp)
%
%  <Description>
%    This routine computes T*x for tranps = 0 or T'*x for value = 1, 
%    where T is an m-by-n Toeplitz matrix, using the eigenvalues lambda
%    of a related circulant matrix computed ny means of tprodinit.
%
%  <Input Parameters>
%    1.    lambda --> eigenvalue vector need for the FFT;
%    2.-3. m,n    --> dimensions of Toeplitz matrix;
%    4.    x      --> vector to be multiplied by T;
%    5.    transp --> if 0 or nonexisting, multiply with T, ortherwise
%                      multiply with T';
%
%  <Output Parameters>
%    1. y --> matrix-vector product
%
%  <Algorithm>
%     Let lambda be the eigenvalues of a circulant matrix derived from T
%     (see tprodinit); then the product T*x consists of the first m elements
%     of the vector ifft(lambda.*fft([x,z])), where z is a vector of zeros,
%     while the product T'*x consists of the first n elements of the vector
%     ifft(conj(lambda).*fft([x,z])).
%
%  <See Also>
%    tprodinit --> Initialization routine for tprod

%  <Reference>
%  [1] P.C. Hansen, "Decconvolution and regularization with Toeplitz matrices,"
%      Numer. Algo. 29 (2002), pp. 323-378.
%
%  <Revision>
%    R.D. Fierro, Dept. of Mathematics, CSUSM
%    P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: February 21, 2004
%-----------------------------------------------------------------------

% Compute product.
p = length(lambda);
if nargin == 4 | transp == 0
    % Compute y = T*x
    y = ifft(lambda .* fft([x; zeros(p-n,1)]));
    y = y(1:m);
else
    % Compute y = T'*x
    y = ifft(conj(lambda) .* fft([x; zeros(p-m,1)]));
    y = y(1:n,1);
end

% Return real result, if required.
if isreal(x) & norm( lambda(2:p)-conj(lambda(2:p)) ) == 0
    y = real(y);
end

%-----------------------------------------------------------------------
% End of function tprod
%-----------------------------------------------------------------------