function [p,L,Omega,V] = hvsvid_L(A,tol_rank,max_iter,fixed_rank)
%  hvsvid_L --> High-rank revealing decomp. of a sym. indef. matrix, L version.
%
%  <Synopsis>
%    [p,L,Omega,V] = hvsvid_L(A)
%    [p,L,Omega,V] = hvsvid_L(A,tol_rank)
%    [p,L,Omega,V] = hvsvid_L(A,tol_rank,max_iter)
%    [p,L,Omega,V] = hvsvid_L(A,tol_rank,max_iter,fixed_rank)
%
%  <Description>
%    Computes a rank-revealing VSV decompostion A = V*(L'*Omega*L)*V' of a
%    symmetric indefinite n-by-n matrix.  Only the upper triangular part needs
%    to be specified.  Optimized for matrices whose rank p close to n.
%    Function hvsvid_R computes the R version.
%
%  <Input parameters>
%    1. A          --> symmetric indefinite matrix;
%    2. tol_rank   --> rank decision tolerance;
%    3. max_iter   --> max. number of inverse iterations per deflation step,
%                      used in the singular vector estimator;
%    4. fixed_rank --> deflate to the fixed rank given by fixed_rank instead of
%                      using the rank decision tolerance;
%
%   Defaults:   tol_rank = n*norm(A,1)*eps;
%               max_iter = 5;
%
%  <Output parameters> 
%    1. p     --> numerical rank of A;
%    2. L     --> lower triangular matrix in A = V*(L'*Omega*L)*V';
%    3. Omega --> signature matrix in A = V*(L'*Omega*L)*V';
%    4. V     --> orthogonal matrix in A = V*(L'*Omega*L)*V';
%
%  <Algorithm>
%    The symmetric indefinite matrix A is preprocessed by a pivoted LDL'
%    factorization.  An interim stage (where D is made diagonal) is followed
%    by a rank-revealing ULV-like decomposition, using inverse iterations
%    for singular vector estimation.
%
%  <See Also>
%    hvsvid_R --> High-rank revealing VSV alg. for sym. indef. matrices, R version
%    hvsvsd   --> High-rank revealing VSV alg. for symmetric semidefinite matrices
%    lvsvid   --> Low-rank revealing VSV alg. for symmetric indefinite matrices
%    lvsvsd   --> Low-rank revealing VSV alg. for symmetric semidefinite matrices

%  <References> 
%  [1] P.C. Hansen & P.Y. Yalamov, "Computing Symmetric Rank-Revealing
%      Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%      Appl., 23 (2001), pp. 443-458.
%
%  <Revision>
%    P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: July 14, 2003
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 1)
  error('Not enough input arguments.')
end

[m,n] = size(A);
if (m*n == 0)
  error('Empty input matrix A not allowed.')
elseif (m ~= n)
  error('Matrix A must be square.')
end

% Check the optional input arguments, and set defaults.
if (nargin==1)
  tol_rank = n*norm(A,1)*eps;
  max_iter = 5;
  fixed_rank = 0;
elseif (nargin==2)
  if isempty(tol_rank), tol_rank = n*norm(A,1)*eps; end
  max_iter = 5;
  fixed_rank = 0;
elseif (nargin==3)
  if isempty(tol_rank), tol_rank = n*norm(A,1)*eps; end
  if isempty(max_iter), max_iter = 5; end
  fixed_rank = 0;
else
  tol_rank = realmax;
  if isempty(max_iter), max_iter = 5; end
end

if (fixed_rank ~= abs(round(fixed_rank))) | (fixed_rank > n)
    error('Requires fixed_rank to be an integer between 0 and n.')
elseif (max_iter ~= abs(round(max_iter)))
    error('Requires max_iter to be an integer > 0.')
elseif (tol_rank ~= abs(tol_rank))
  error('Requires positive values for tol_rank.')
end

% Check the number of output arguments.
if (nargout < 3)
  V = [];
else
  V = eye(n);
end

% Initial LDLT factorization A = Pi'*L*D*L'*Pi.
A = triu(A) + triu(A,1)';     % Make sure A is symmetric.
[L,D,Pi] = ldlt_symm(A,'r');  % Use rook pivoting.

% Interim process, L*D*L' = W*R'*Omega*R*W'.
[W,R,Omega] = vsvid_ip(L,D);

% Permutations for L-version of VSV decomposition.
L = R(n:-1:1,n:-1:1);
Omega = Omega(n:-1:1,n:-1:1);

% Rank-revealing procedure.

% Estimate the smallest singular value and the corresponding right
% singular vector of L'*Omega*L via inverse iterations.
[smin,vmin] = TOTinviter(L,Omega,max_iter);

p = n;  % Init. loop to rank zero.

while ((smin <= tol_rank) & (p > fixed_rank))

  % Apply deflation procedure to p'th row of L in the VSV decomposition.
  [L,Omega,V] = hvsvid_rdef(L,Omega,V,p,vmin);
  
  p = p - 1;
  % Estimate the smallest singular value and the corresponding right
  % singular vector of leading submatrix by using inverse iterations.
  if (p > 0)
    if norm(L(p+1:n,1:p),1) > 10*n*norm(L(p+1:n,p+1:n),1)
      % Perform block iteration to reduce the norm.
      [L,Omega,V] = vsv_qrit(p,1,L,Omega,V);
    end
    [smin,vmin] = TOTinviter(L(1:p,1:p),Omega(1:p,1:p),max_iter);
  else
    smin = 0;  % No 0'th singular value
  end
end

% Return optional output.
if (nargout >= 3) 
  V = Pi'*W(:,n:-1:1)*V;
end

%-----------------------------------------------------------------------
% End of function hvsvid_L
%-----------------------------------------------------------------------