function [smax,vmax] = TOTlanczos(T,Omega,itmax,reorth)
%  TOTlanczos --> Symmetric Lanczos procedure for T'*Omega*T.
%
%  <Synopsis>
%    [smax,vmax] = TOTlanczos(T,Omega,itmax)
%    [smax,vmax] = TOTlanczos(T,Omega,itmax,reorth)
%
%  <Description>
%    Computes an estimate of the largest singular value and the associated
%    singular vector of the matrix T'*Omega*T using a maximum of itmax
%    Lanczos iterations.  If reorth = 1, then MGS reorthogonalization is used.
%
%  <Input parameters>
%    1. T      --> matrix;
%    2. Omega  --> diagonal matrix;
%    2. itmax  --> maximum number of iterations;
%    4. reorth --> MGS reorthogonalization if true;
%
%    Defaults: reorth = 1 (reorthogonalization).
%
% <Output parameters>
%   1.  smin  --> estimate of largest singular value;
%   2.  vmin  --> estimate of corresponding singular vector;
%
%  <See Also>
%    TOTpowiter --> Power iterations for T'*Omega*T

%  <References>
%  [1] G.H. Golub and C.F. Van Loan, "Matrix Computations", Johns Hopkins
%      University Press, 3. Ed., 1996; p. 480.
%
%  <Revision>
%    Per Christian Hansen, IMM, Technical University of Denmark
%
%    Last revised: February 29, 2004.
%-----------------------------------------------------------------------

[m,n]    = size(T);
thresh   = 1e-5;                           % Relative stopping criterion.
max_iter = min([itmax; n-1; m-1]);

% Set defaults.
if (nargin == 3)
  reorth = 1;
end

% If T'*Omega*T is 1-by-1.
if (n == 1)
  smax = abs(T'*Omega*T);
  vmax = 1;
  return;
end

% Initialize.
s = diag(Omega);
v = zeros(n,1);
TT = zeros(max_iter,max_iter);
V = zeros(n,max_iter);

w = ones(n,1)/sqrt(n);
beta = 1;
V(:,1) = w;

v = T'*(s.*(T*w));
alpha = w'*v;
TT(1,1) = alpha;
v = v - alpha*w;

smax_old = 0;

% Perform Lanczos bidiagonalization.
for (i = 2:max_iter+1)
  
  % Re-orthogonalize.
  if (reorth)
    for (step = 1:i-1)
      v = v - V(:,step)*(V(:,step)'*v);
    end
  end
  V(:,i) = v;
  
  beta = norm(v);
  TT(i-1,i) = beta;
  TT(i,i-1) = beta;
    
  t = w;
  w = v/beta;
  v = -beta*t;
    
  v = v + T'*(s.*(T*w));
  alpha = w'*v;
  TT(i,i) = alpha;
  v = v - alpha*w;

  % Check for convergence.
  ss    = svd(TT(1:i,1:i));
  smax  = ss(1,1);
  check = abs(smax-smax_old)/smax;
  if (check < thresh)
    break;
  else
    smax_old = smax;
  end
  
end

% Compute estimates of the principal one-dimensional subspaces.
[uu,ss,vv] = svd(TT(1:i,1:i));
smax = ss(1,1);
vmax = V(:,1:i)*vv(:,1);

%-----------------------------------------------------------------------
% End of function TOTlanczos
%-----------------------------------------------------------------------