function [p,L,Omega,V] = vsvid_L_mod(p,L,Omega,V,omega,v,tol_rank,inv_iter,fixed_rank)
%  vsvid_L_mod --> Rank-one modification of VSV decomp. of sym. indef. matrix, L version.
%
%  <Synopsis>
%    [p,L,Omega,V] = vsvid_L_mod(p,L,Omega,V,omega,v)
%    [p,L,Omega,V] = vsvid_L_mod(p,L,Omega,V,omega,v,tol_rank)
%    [p,L,Omega,V] = vsvid_L_mod(p,L,Omega,V,omega,v,tol_rank,inv_iter)
%    [p,L,Omega,V] = vsvid_L_mod(p,L,Omega,V,omega,v,tol_rank,inv_iter,fixed_rank)
%
%  <Description>
%    Given a rank-revealing VSV decomposition of a symmetric indefinite matrix
%    A = V*(L'*Omega*L)*V', the function computes the updated rank-revealing
%    decomposition of the matrix A + omega*v*v', where omega = +1 or -1.
%
%  <Input parameters>
%    1. p          --> numerical rank of A;
%    2. L          --> lower triangular matrix in A = V*(L'*Omega*L)*V';
%    3. Omega      --> signature matrix in A = V*(L'*Omega*L)*V';   
%    4. V          --> orthogonal matrix in A = V*(L'*Omega*L)*V';
%    5. omega      --> update (+1) or downdate (-1);
%    6. v          --> rank-one update column vector;
%    7. tol_rank   --> rank decision tolerance;
%    8. inv_iter   --> number of inverse iterations per deflation step;
%    9. fixed_rank --> deflate to the fixed rank given by fixed_rank instead
%                      of using the rank decision tolerance;
%
%    Defaults:   tol_rank = n*norm(L,1)*eps;
%                inv_iter = 5;
%
%  <Output parameters> 
%    1. p     --> numerical rank of modified matrix;
%    2. L     --> updated lower triangular matrix;
%    3. Omega --> updated signature matrix;
%    4. V     --> updated orthogonal matrix;
%
%  <See Also>
%    vsvid_R_mod --> Rank-one mod. of VSV decomp. of sym. indef. matrix, R version.
%    vsvsd_up    --> Rank-one update of VSV decomposition of semidefinite matrix.

%  <References>
%  [1] P.C. Hansen & P.Y. Yalamov, "Computing Symmetric Rank-Revealing
%      Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%      Appl., 23 (2001), pp. 443--458.
%
%  <Revision>
%    P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: July 15, 2003
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 6)
  error('Not enough input arguments.')
end

[mL,n]  = size(L);
[mV,nV] = size(V);
if (mL*n == 0) | (mV*nV == 0)
  error('Empty input matrices R and V not allowed.')
elseif (mL ~= n) | (mV ~= nV)
  error('Square n-by-n matrices R and V required.')
elseif (nV ~= n)
  error('Not a valid VSV decomposition.')
end

if (length(v) ~= n)
  error('Not a valid update vector.')
end

if (p ~= abs(round(p))) | (p > n)
  error('Requires the rank p to be an integer between 0 and n.')
end

% Check the optional input arguments, and set defaults.
if (nargin == 6)
  tol_rank = n*norm(L,1)*eps;
  inv_iter = 5;
  fixed_rank = 0;
elseif (nargin == 7)
  if isempty(tol_rank), tol_rank = n*norm(L,1)*eps; end
  inv_iter = 5;
  fixed_rank = 0;
elseif (nargin == 8)
  if isempty(tol_rank), tol_rank = n*norm(L,1)*eps; end
  if isempyt(inv_iter), inv_iter = 5; end
  fixed_rank = 0;
elseif (nargin == 9)
  tol_rank = realmax;
  if isempyt(inv_iter), inv_iter = 5; end
end

if (tol_rank ~= abs(tol_rank))
  error('Requires positive values for tol_rank.')
end

% The row vector d = v'*V is appended to the bottom of L.
d = v(:)'*V;

% Eliminate all but the first element of d(p+1:n) using rotations.
for (i = n:-1:p+2)    
  % Eliminate d(i).  
  [c,s,d(i-1)] = gen_giv(d(i-1),d(i));
  d(i) = 0;                                 % Eliminate d(i).

  % Apply rotation to L from the right.
  [L(i-1:n,i-1),L(i-1:n,i)] = app_giv(L(i-1:n,i-1),L(i-1:n,i),c,s);

  % Apply rotation to V from the right.
  [V(1:n,i-1),V(1:n,i)] = app_giv(V(1:n,i-1),V(1:n,i),c,s);

  % Restore L to upper triangular form using quadratic rotations from the left.
  [c,s,L(i,i),sgn] = gen_qrot(L(i,i),L(i-1,i),Omega(i,i),Omega(i-1,i-1));
  L(i-1,i) = 0;                             % Eliminate R(i,i-1).
  [L(i,1:i-1),L(i-1,1:i-1),Omega(i,i),Omega(i-1,i-1)] = ...
      app_qrot(L(i,1:i-1),L(i-1,1:i-1),c,s,Omega(i,i),Omega(i-1,i-1),sgn); %PCH-s?
end

% Remove the remaining elements of d by means of left quadratic rotations.
for i = p+1:-1:1
  [c,s,L(i,i),sgn] = gen_qrot(L(i,i),d(i),Omega(i,i),omega);
  [L(i,1:i-1),d(1:i-1),Omega(i,i),omega] = ...
      app_qrot(L(i,1:i-1),d(1:i-1),c,s,Omega(i,i),omega,sgn);
end

% Make the updated decomposition rank-revealing.
% Note that rank can increase by one, stay the same, or decrease by one.

%First initialize.
smin = 0;                             % No 0th singular value.

% Use a priori knowledge about rank changes.
if (fixed_rank)
  p_min = p;
else
  p_min = 0;
end

% Allow possible increase in rank.
if (p < n)
  p = p+1;
end

% Estimate of p'th singular value and corresponding right singular vector.
[smin,vmin] = TOTinviter(L(1:p,1:p),Omega(1:p,1:p),inv_iter);

while ((smin < tol_rank) & (p > p_min))
  % Apply deflation procedure to p'th column of R in the URV-based VSV algorithm.
  [L,Omega,V] = hvsvid_rdef(L,Omega,V,p,vmin);

  % New rank estimate after the problem has been deflated.
  p = p - 1;

  % Est. of p'th singular value and corresponding right singular vector.
  if (p > 0)
    [smin,vmin] = TOTinviter(L(1:p,1:p),Omega(1:p,1:p),inv_iter);
  else
    smin = 0;                                % No 0th singular value.
  end
end

% Normalize the columns of V in order to maintain orthogonality.
for i = 1:n
  V(:,i) = V(:,i)/norm(V(:,i));
end

%-----------------------------------------------------------------------
% End of function vsvid_L_mod
%----------------------------------------------------------------------