function ullivdemo
% ullivdemo --> demonstration of ulliv for noise and interference reduction
%
% Clean signal: sum of sinusoids with random amplitude and frequency.
% Noise signal: white noise.
% Interfering signal: sum of sinusoids with random amplitude and frequency.
%
% Plot the Fourier spectra for the clean, noisy and reconstructed signals.

% Reference: P.C. Hansen & S.H. Jensen, Prewhitening for Rank-Deficient Noise
% in Subspace Methods for Noise Reduction; IEEE Trans. Signal Proc.,
% to appear.
%
% Per Christian Hansen, IMM, Technical University of Denmark
%
% Last revised: February 3, 2005.
%-----------------------------------------------------------------------

% Clean signal: sum of sinusoids.
N = 256;          % Signal length
signaltones = 9;  % Number of signal tones.
xclean = zeros(N,1);
t = 2*pi*[1:N]'/N;
for i=1:signaltones
    xclean = xclean + sin((N/2/(signaltones+1.5))*i*t);
end

% Intefering signal: sum of sinusoids.  Make two similar signals; one is
% the actual interference, the other a representative signal used for the
% interference reduction.
interftones = 16; % Number of interference tones.
e  = zeros(N,1);
er = zeros(N,1);
for i=1:interftones
    ae = 0.5;
    fe = (N/2/(interftones+3))*(i-0.5);
    e  = e  + ae*sin(fe*t);
    ar = ae*(1+randn/20);
    fr = fe*(1+randn/40);
    er = er + ar*sin(fr*t);
end

% Display information about the test.
clc
disp('Running ullivdemo (noise reduction):')
disp(['The signal is of length ',num2str(N),...
      ' and consists of a sum of ',num2str(signaltones),...
      ' sinusoids'])
disp('with additive white noise and an interfering signal consisting of')
disp(['a sum of ',num2str(interftones),' sinusoids with random amplitude and frequency.'])
disp('The noise reduction must diminish the peaks in the spectrum')
disp('coming from the noise, and thus increase the SNR.')

% Complete signal with white noise and interference.
noiselevel = 0.02;  % Level of white noise.
x = xclean + noiselevel*randn(N,1) + e;

% Make Hankel matrices from signals.
n = 40;  % Model order (number of columns).
m = N-n+1;
X = hankel(x(1:m),x(m:N));
E = hankel(er(1:m),er(m:N));

% Rank reduction of E.
[pE,LE,VE] = hulv(E,1e-10);
if pE ~= 2*interftones, error('Wrong rank detection of E'), end
E = LE(1:pE,1:pE)*VE(:,1:pE)';

% Compute ULLIV decomposition.
tol = 2;  % Tolerance for rank decision.
[p,LX,L,V,UX] = ulliv(X,E,tol);

% Reconstruct signal matrix.
Xrec = UX(:,1:p)*LX(1:p,1:p)*L(1:p,1:p)*V(:,1:p)' + ...
       UX(:,pE+1:n)*[LX(pE+1:n,1:pE)*L,LX(pE+1:n,pE+1:n)]*V';
xrec = aaad(Xrec);
   
% Plot the Fourier spectra of the clean, noisy and reconstructed signals.
figure(1), clf
subplot(3,1,1)
  plot(1:N,abs(fft(xclean)))
  title('Clean signal')
  axis([0 N/2 0 100])
  set(gca,'xticklabel',[])
subplot(3,1,2)
  plot(1:N,abs(fft(x)))
  title(['Noisy signal,  SNR = ',...
      num2str(20*log10(norm(xclean)/norm(xclean-x)))])
  axis([0 N/2 0 100])
  set(gca,'xticklabel',[])
subplot(3,1,3)
  plot(1:N,abs(fft(xrec)))
  title(['Reconstructed signal,  SNR = ',...
      num2str(20*log10(norm(xclean)/norm(xclean-xrec)))])
  axis([0 N/2 0 100])

%--------------------------------------------------------------------------

function h = aaad(A)
%AAAD Average along anti-diagonals
%
%  h = aaad(A)
%
% The vector h consists of the first column/last row of the Hankel matrix
% is obtained by averaging along the anti-dagonals of the general matrix A.

% Per Christian Hansen, IMM, June 29, 2002.

[m,n] = size(A);
A = fliplr(A);
h = zeros(m+n-1,1);

for i=1:m+n-1
    d = diag(A,n-i);
    h(i) = mean(d);
end