function [p,LA,L,V,UA,UB,vec] = ulliv_up_A(p,LA,L,V,UA,UB,a,tol_rank,...
                                           tol_ref,max_ref,fixed_rank)
%  ulliv_up_a --> Update the A-part of the rank-revealing ULLIV decomposition.
%
%  <Synopsis>
%    [p,LA,L,V,UA,UB,vec] = ulliv_up_A(p,LA,L,V,UA,UB,a)
%    [p,LA,L,V,UA,UB,vec] = ulliv_up_A(p,LA,L,V,UA,UB,a,tol_rank)
%    [p,LA,L,V,UA,UB,vec] = ulliv_up_A(p,LA,L,V,UA,UB,a,tol_rank,...
%                                                    tol_ref,max_ref)
%    [p,LA,L,V,UA,UB,vec] = ulliv_up_A(p,LA,L,V,UA,UB,a,tol_rank,...
%                                                    tol_ref,max_ref,fixed_rank)
%
%  <Description>
%    Given a rank-revealing ULLIV decomposition of the mA-by-n matrix with
%    mA >= n, and the full-rank mB-by-n matrix B = UB*L*V' with mB < n,
%    the function computes the updated decomposition
%
%             [ A ] = UA*LA*[ L 0 ]*V'  and  B = UB*[ L 0 ]*V'
%             [ a ]         [ 0 I ]
%
%    where a is the new row added to A. Note that B must have full row rank,
%    that the row dimension of UA will increase by one, and that the matrices
%    UA and UB can be left out by inserting an empty matrix [] while V is required.
%
%  <Input Parameters>
%    1.   p            --> numerical rank of A*pinv(B)_A revealed in LA;
%    2-6. LA,L,V,UA,UB --> the ULLIV factors of A and B;
%    7.   a            --> the new row added to A;
%    8.   tol_rank     --> rank decision tolerance;
%    9.   tol_ref      --> upper bound on the 2-norm of the off-diagonal block
%                          LA(p+1:mB,1:p) relative to the Frobenius-norm of LA;
%    10.  max_ref      --> max. number of refinement steps per singular value
%                          to achieve the upper bound tol_ref;
%    11.  fixed_rank   --> if true, deflate to the fixed rank given by p
%                          instead of using the rank decision tolerance;
%
%    Defaults: tol_rank = sqrt(n)*norm(LA,1)*eps;
%              tol_ref  = 1e-04;
%              max_ref  = 0;
%
%  <Output Parameters>
%    1.   p            --> numerical rank of updated quotient;
%    2-6. LA,L,V,UA,UB --> the updated ULLV factors;
%    7.   vec          --> a 5-by-1 vector with:
%         vec(1) = upper bound of norm(LA(p+1:mB,1:p)),
%         vec(2) = estimate of pth singular value,
%         vec(3) = estimate of (p+1)th singular value,
%         vec(4) = a posteriori upper bound of num. nullspace angle,
%         vec(5) = a posteriori upper bound of num. range angle.
%
%  <See Also>
%    ulliv_up_B --> Update the B-part of the rank-revealing ULLIV decomp.

%  <References>
%  [1] F.T.Luk and S. Qiao, "A New Matrix Decomposition for Signal
%      Processing", Automatica, 30 (1994), pp. 39--43.
%  [2] F.T.Luk and S. Qiao, "An adaptive algoithm for interference
%      cancelling in array processing; in F.T. Luk (Ed.), "Advanced
%      Signal Processing Algorithms, Architectures, and Implementations
%      VI," SPIE Proceedings, Vol. 2846 (1996), pp. 151-161.
%
%  <Revision>
%    Ricardo D. Fierro, California State University San Marcos
%    Per Christian Hansen, IMM, Technical University of Denmark
%
%    Last revised: February 11, 2004
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 7)
  error('Not enough input arguments.')
end

[mLA,n] = size(LA);
[mL,nL] = size(L);
[mV,nV] = size(V);
if (mLA*n == 0) | (mL*nL == 0) | (mV*nV == 0)
  error('Empty input matrices LA, L and V not allowed.')
elseif (mLA ~= n) | (mL ~= nL) | (mV ~= nV)
  error('Square matrices LA, L and V required.')
elseif (nV ~= n)
  error('LA and V must have the same dimensions.')
elseif (nL >= n)
  error('L is too large.')
end

[mA,nA] = size(UA);
if (mA*nA == 0)
  uAflag = 0;
elseif (nA ~= n)
  error('Wrong no. of columns in UA.')
elseif (mA < n)
  error('The A-part of the system is underdetermined.')
else
  uAflag = 1;
  UA(mA+1,n+1) = 1;
end

[mB,nB] = size(UB);
if (mB*nB == 0)
  uBflag = 0;
elseif (nB ~= nL)
  error('Wrong no. of columns in UB.')
else
  uBflag = 1;
end

if (length(a) ~= n)
  error('Not a valid update vector.')
end

if (p ~= abs(round(p))) | (p > nL)
  error('Rank p must be an integer between 0 and size(L,1).')
end

% Check the optional input arguments, and set defaults.
if (nargin == 7)
  tol_rank   = sqrt(n)*norm(LA,1)*eps;
  tol_ref    = 1e-04;
  max_ref    = 0;
  fixed_rank = 0;
elseif (nargin == 8)
  if isempty(tol_rank), tol_rank = sqrt(n)*norm(LA,1)*eps; end
  tol_ref    = 1e-04;
  max_ref    = 0;
  fixed_rank = 0;
elseif (nargin == 9)
  if isempty(tol_rank), tol_rank = sqrt(n)*norm(LA,1)*eps; end
  if isempty(tol_ref), tol_ref  = 1e-04; end
  max_ref    = 0;
  fixed_rank = 0;
elseif (nargin == 10)
  if isempty(tol_rank), tol_rank = sqrt(n)*norm(LA,1)*eps; end
  if isempty(tol_ref), tol_ref  = 1e-04; end
  if isempty(max_ref), max_ref  = 0; end
  fixed_rank = 0;
elseif (nargin == 11)
  if isempty(tol_ref), tol_ref  = 1e-04; end
  if isempty(max_ref), max_ref  = 0; end
  if (fixed_rank)
    tol_rank = realmax;
  else
    if isempty(tol_rank), tol_rank = sqrt(n)*norm(LA,1)*eps; end
    fixed_rank = 0;
  end
end

if (tol_rank ~= abs(tol_rank)) | (tol_ref ~= abs(tol_ref))
  error('Requires positive values for tol_rank and tol_ref.')
end
if (max_ref ~= abs(round(max_ref)))
  error('Requires positive integer value for max_ref.')
end

% Check the number of output arguments.
if (nargout ~= 7)
  vecflag = 0;
else
  vecflag = 1;
end

% Update the decomposition.

% The row vector d is appended to the bottom of LA.
d = a(:)'*V;
d(1:nL) = d(1:nL)/L;
  
% Eliminate all elements of d(nL+1:n) using left rotations.
for (i = n:-1:nL+1)
  % Eliminate d(i) using the i-th row of LA (left rot.).
  [c,s,LA(i,i)] = gen_giv(LA(i,i),d(i));
  [LA(i,1:i-1),d(1:i-1)] = app_giv(LA(i,1:i-1),d(1:i-1),c,s); d(i) = 0;
  % Apply rotation to UA on the right.
  if (uAflag)
    [UA(:,i),UA(:,n+1)] = app_giv(UA(:,i),UA(:,n+1),c,s);
  end
end
  
% Eliminate all elements of d(2:nL) using interleaved right/left rotations.
for (i = nL:-1:2)
  % Eliminate d(i) using d(i-1) (right rotation).
  [c,s,d(i-1)] = gen_giv(d(i-1),d(i));
  [LA(i-1:n,i-1),LA(i-1:n,i)] = app_giv(LA(i-1:n,i-1),LA(i-1:n,i),c,s); d(i) = 0;
  % Apply rotation to L from the left.
  [L(i-1,1:i),L(i,1:i)] = app_giv(L(i-1,1:i),L(i,1:i),c,s);
  % Apply rotation to UB on the right.
  if (uBflag)
    [UB(:,i-1),UB(:,i)] = app_giv(UB(:,i-1),UB(:,i),c,s);
  end
    
  % Restore triangular form of LA using left rotation.
  [c,s,LA(i,i)] = gen_giv(LA(i,i),LA(i-1,i));
  [LA(i,1:i-1),LA(i-1,1:i-1)] = app_giv(LA(i,1:i-1),LA(i-1,1:i-1),c,s); LA(i-1,i)=0;
  % Apply rotation to UA on the right.
  if (uAflag)
    [UA(:,i-1),UA(:,i)] = app_giv(UA(:,i-1),UA(:,i),c,-s);
  end
    
  % Restore triangular form of L using right rotation.
  [c,s,L(i-1,i-1)] = gen_giv(L(i-1,i-1),L(i-1,i));
  [L(i:nL,i-1),L(i:nL,i)] = app_giv(L(i:nL,i-1),L(i:nL,i),c,s); L(i-1,i)=0;
  % Apply rotation to V on the right.
  [V(:,i-1),V(:,i)] = app_giv(V(:,i-1),V(:,i),c,s);
end
  
% Finish up with left rotation.
[c,s,LA(1,1)] = gen_giv(LA(1,1),d(1));
if (uAflag)
  [UA(:,1),UA(:,n+1)] = app_giv(UA(:,1),UA(:,n+1),c,s);
  UA = UA(:,1:n);
end

% Make the updated decomposition rank-revealing.

% Initialize.
smin          = 0;                              % No 0th singular value.
smin_p_plus_1 = 0;                              % No (n+1)th singular value.
norm_tol_ref  = norm(LA,'fro')*tol_ref/sqrt(n); % Value used to verify ...
                                                % ... the upper bound tol_ref.

% Use a priori knowledge about rank changes.
if (fixed_rank), p_min = p; else p_min = 0; end

% Apparent increase in rank.
if (p < nL), p = p+1; end

% Estimate of the p'th singular value and the corresponding left
% singular vector via the generalized LINPACK condition estimator.
[smin,umin] = ccvl(LA(1:p,1:p)');

if (smin < tol_rank)
  % The rank stays the same or decreases.

  while ((smin < tol_rank) & (p > p_min))
    % Apply deflation procedure to p'th row of LA in the ULLV decomposition.
    [LA,L,V,UA,UB] = ullv_rdef(LA,L,V,UA,UB,p,umin);

    % Refinement loop.
    num_ref = 0;                                % Init. refinement counter.
    while (norm(LA(p,1:p-1)) > norm_tol_ref) & (num_ref < max_ref)
      % Apply one QR-iteration to p'th row of LA in the ULLV decomposition.
      [LA,L,V,UA,UB] = ullv_ref(LA,L,V,UA,UB,p);
      num_ref = num_ref + 1;
    end

    % New rank estimate after the problem has been deflated.
    p = p - 1;
    smin_p_plus_1 = smin;

    % Estimate of the p'th singular value and the corresponding left
    % singular vector via the generalized LINPACK condition estimator.
    if (p > 0)
      [smin,umin] = ccvl(LA(1:p,1:p)');
    else
      smin = 0;                                 % No 0th singular value.
    end
  end
elseif (p < nL)
  % The rank increases by one.

  % Refinement loop.
  num_ref = 0;                                  % Init. refinement counter.
  while (norm(LA(p+1,1:p)) > norm_tol_ref) & (num_ref < max_ref)
    % Apply one QR-iteration to p+1'th row of LA in the ULLV decomposition.
    [LA,L,V,UA,UB] = ullv_ref(LA,L,V,UA,UB,p+1);
    num_ref = num_ref + 1;
  end

  % Estimate of the (p+1)th singular value and the corresponding left
  % singular vector via the generalized LINPACK condition estimator.
  [smin_p_plus_1,umin] = ccvl(LA(1:p+1,1:p+1)');
end

% Normalize the columns of V in order to maintain orthogonality.
for (i = 1:n)
  V(:,i) = V(:,i)./norm(V(:,i));
end

% Estimates that describe the quality of the decomposition.
if (vecflag)
  vec    = zeros(5,1);
  vec(1) = sqrt(n-p)*norm(LA(p+1:n,1:p),1);
  vec(2) = smin;
  vec(3) = smin_p_plus_1;
  vec(4) = (vec(1)*smin_p_plus_1)/(smin^2 - smin_p_plus_1^2);
  vec(5) = (vec(1)*smin)/(smin^2 - smin_p_plus_1^2);
end

%-----------------------------------------------------------------------
% End of function ulliv_up_A
%-----------------------------------------------------------------------