function x_tvsv = tvsv(varargin)
%  tvsv --> Solves a rank-deficient system using the VSV decomposition.
%
%  <Synopsis>
%    x_tvsv = tvsv(L,V,p,b)
%    x_tvsv = tvsv(R,Omega,V,p,b)
%    x_tvsv = tvsv(L,Omega,V,p,b)
%
%  <Description>
%    Solves the symmetric and near-rank deficient system of equations A x = b,
%    using the rank-revealing VSV decomposition of A.  Three decompositions
%    of A can be used:
%        A = V*L'*L*V'           (semidefinite A, lower triangular L)
%        A = V*R'*Omega*R*V'     (indefinite A, upper triangular R)
%        A = V*L'*Omega*L*V'     (indefinite A, lower triangular L)
%
%  <Input Parameters>
%    Semidefite case:
%      1. L --> lower triangular matrix;
%      2. V --> orthogonal matrix;
%      3. p --> numerical rank;
%      4. b --> right-hand side;
%    Indefinite case:
%      1. L or R --> lower or upper triangular matrix;
%      2. Omega  --> signature matrix;
%      3. V      --> orthogonal matrix;
%      4. p      --> numerical rank;
%      5. b      --> right-hand side;
%
%  <Output Parameters>
%    1. x_tvsv --> truncated VSV solution;

%  <References> 
%    [1] P. C. Hansen & P. Y. Yalamov, "Computing Symmetric Rank-Revealing
%        Decompositions via Triangular Factorization", SIAM J. Matrix Anal.
%        Appl., 23 (2001), pp. 443--458.
%
%  <Revision>
%    P.C. Hansen, IMM, Technical University of Denmark
%
%    Last revised: July 15, 2003
%-----------------------------------------------------------------------

% Check the required input arguments.
if (nargin < 4)
  error('Not enough input arguments.')
end

if (nargin==4)
  % Semidefinite case.
  L = varargin{1};
  V = varargin{2};
  p = varargin{3};
  b = varargin{4};
  x_tvsv = V(:,1:p)*(L(1:p,1:p)\(L(1:p,1:p)'\(V(:,1:p)'*b)));
else
  % Indefinite case.
  L = varargin{1};
  omega = diag(varargin{2});
  V = varargin{3};
  p = varargin{4};
  b = varargin{5};
  x_tvsv = V(:,1:p)*(L(1:p,1:p)\(omega.*(L(1:p,1:p)'\(V(:,1:p)'*b))));
end